<?php

namespace App\Actions\Fortify;

use App\Models\Team;
use App\Models\User;
use App\Models\TeamInvitation; // [NEW]
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Laravel\Fortify\Contracts\CreatesNewUsers;
use Laravel\Jetstream\Jetstream;

class CreateNewUser implements CreatesNewUsers
{
    use PasswordValidationRules;
    use HandlesOnboardingTasks;
    /**
     * Create a personal team for the user and set it as their current team.
     *
     * @param  \App\Models\User  $user
     * @return void
     */
    protected function createTeam(User $user): void
    {
        // We use the switchTeam method to create the team and
        // immediately set it as the user's current team.
        // This solves the "currentTeam->name" on null error.
        $user->switchTeam($team = $user->ownedTeams()->create([
            'name' => $user->first_name . ' ' . $user->last_name . "'s Team",
            'personal_team' => true,
        ]));
    }

    /**
     * Validate and create a newly registered user.
     *
     * @param  array<string, string>  $input
     * @return \App\Models\User
     */
    public function create(array $input): User
    {
        Validator::make($input, [
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'username' => ['required', 'string', 'alpha_dash', 'max:255', 'unique:users'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => $this->passwordRules(),
            'terms' => Jetstream::hasTermsAndPrivacyPolicyFeature() ? ['accepted', 'required'] : '',
            'invitation_token' => ['nullable', 'string', 'exists:team_invitations,token'], // [NEW]
        ])->validate();

        return DB::transaction(function () use ($input) {
            return tap(User::create([
                'first_name' => $input['first_name'],
                'last_name' => $input['last_name'],
                'username' => $input['username'],
                'email' => $input['email'],
                'password' => Hash::make($input['password']),
            ]), function (User $user) use ($input) {
                // [NEW LOGIC]
                // اگر کاربر از طریق یک لینک دعوت‌نامه ثبت‌نام کرده باشد...
                if (isset($input['invitation_token'])) {
                    $invitation = TeamInvitation::where('token', $input['invitation_token'])->first();
                    if ($invitation) {
                        // ...او را به تیم مربوطه اضافه می‌کنیم.
                        $invitation->team->users()->attach($user, ['role' => $invitation->role]);

                        $this->createOnboardingTasks($user, $invitation->team);
                        // و دعوت‌نامه را حذف می‌کنیم.
                        $invitation->delete();
                    }
                }
                // در غیر این صورت، یک تیم شخصی برای او ایجاد می‌کنیم.
                else {
                    $this->createTeam($user);
                }
            });
        });
    }
}
