<?php

namespace App\Console\Commands;

use App\Models\Project;
use App\Services\AIService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class AnalyzeProjectRisks extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:analyze-project-risks';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Analyzes active projects for potential risks using historical data and AI.';

    /**
     * Execute the console command.
     */
    public function handle(AIService $aiService)
    {
        $this->info('Starting project risk analysis...');

        // 1. Gather historical data from completed projects to create a baseline
        $completedProjects = Project::whereIn('status', ['Completed', 'Archived'])
            ->withCount(['tasks', 'tasks as overdue_tasks_count' => function ($query) {
                $query->where('status', '!=', 'Done')->where('due_date', '<', now());
            }])
            ->get();

        if ($completedProjects->isEmpty()) {
            $this->warn('No completed projects found to create a historical baseline. Analysis may be less accurate.');
            $historicalContext = 'No historical data available.';
        } else {
            $avgOverduePercentage = $completedProjects->avg(function ($p) {
                return $p->tasks_count > 0 ? ($p->overdue_tasks_count / $p->tasks_count) * 100 : 0;
            });
            $historicalContext = "Historically, completed projects had an average of " . round($avgOverduePercentage, 2) . "% overdue tasks.";
        }

        // 2. Get active projects to analyze
        $activeProjects = Project::whereNotIn('status', ['Completed', 'Archived'])
            ->withCount(['tasks', 'tasks as overdue_tasks_count' => function ($query) {
                $query->where('status', '!=', 'Done')->where('due_date', '<', now());
            }])
            ->get();

        if ($activeProjects->isEmpty()) {
            $this->info('No active projects to analyze.');
            return 0;
        }

        foreach ($activeProjects as $project) {
            $this->line("Analyzing project: {$project->title}");

            $progress = $project->tasks_count > 0 ? ($project->tasks()->where('status', 'Done')->count() / $project->tasks_count) * 100 : 0;
            $overduePercentage = $project->tasks_count > 0 ? ($project->overdue_tasks_count / $project->tasks_count) * 100 : 0;

            $prompt = "
                As a project management expert, analyze the risk level of the following project based on historical data and its current status.
                The output MUST be a valid JSON object with two keys: 'risk_level' and 'risk_analysis'.
                - 'risk_level' must be one of 'low', 'medium', or 'high'.
                - 'risk_analysis' must be a brief, one-paragraph summary in Persian explaining the reasoning.

                Historical Context: {$historicalContext}

                Current Project Status:
                - Title: {$project->title}
                - Total Tasks: {$project->tasks_count}
                - Progress (Done %): " . round($progress, 2) . "%
                - Overdue Tasks %: " . round($overduePercentage, 2) . "%

                Analyze the risk of this project falling behind schedule. A high percentage of overdue tasks relative to progress is a major risk factor.
            ";

            // 3. Call the AI Service
            $result = $aiService->getStructuredResponse($prompt);

            if ($result && !empty($result['risk_level']) && !empty($result['risk_analysis'])) {
                $project->update([
                    'risk_level' => $result['risk_level'],
                    'risk_analysis' => $result['risk_analysis'],
                ]);
                $this->info(" -> Success! Risk level set to: {$result['risk_level']}");
            } else {
                $this->error(" -> Failed: AI could not provide a valid analysis for this project.");
                Log::warning("AI risk analysis failed for project ID: {$project->id}", ['result' => $result]);
            }
        }

        $this->info('Project risk analysis completed successfully.');
        return 0;
    }
}
