<?php

namespace App\Console\Commands;

use App\Models\SupplierResponseItem;
use App\Models\FinancialAlert;
use App\Services\AIService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class AnalyzePurchaseHistory extends Command
{
    protected $signature = 'app:analyze-purchase-history';
    protected $description = 'Analyzes historical purchase prices to detect anomalies using AI.';

    public function handle(AIService $aiService)
    {
        $this->info('Starting purchase history analysis...');

        // Get all successfully purchased items
        $purchasedItems = SupplierResponseItem::whereHas('originalItem', fn($q) => $q->where('status', 'offer_accepted'))
            ->with(['originalItem', 'response.supplier', 'response.purchaseRequest.project.researchTeam'])
            ->get();

        // Group items by their name and supplier
        $groupedItems = $purchasedItems->groupBy(function ($item) {
            return $item->originalItem->item_name . '|' . $item->response->supplier_id;
        });

        foreach ($groupedItems as $group) {
            // We only analyze items that have been purchased more than once
            if ($group->count() < 2) {
                continue;
            }

            $latestItem = $group->last();
            $team = $latestItem->response->purchaseRequest->project->researchTeam;
            $itemName = $latestItem->originalItem->item_name;
            $supplierName = $latestItem->response->supplier->company_name;

            $priceHistory = $group->map(fn($item) => "Date: " . $item->created_at->format('Y-m-d') . ", Price: " . $item->price_per_unit)->implode("\n");

            $prompt = "
                As a financial analyst, analyze the following price history for a specific item from a specific supplier.
                Detect any significant price increases or anomalies. The latest purchase is the last one in the list.
                The output MUST be a valid JSON object with two keys: 'is_anomaly' (boolean) and 'analysis' (a brief, one-sentence summary in Persian explaining the finding).

                Item: {$itemName}
                Supplier: {$supplierName}
                Price History (in Toman):
                {$priceHistory}

                Analyze if the latest price shows a significant jump (e.g., more than 20%) compared to the previous price or the average price.
            ";

            $result = $aiService->getStructuredResponse($prompt);

            if ($result && isset($result['is_anomaly']) && $result['is_anomaly'] === true) {
                FinancialAlert::create([
                    'research_team_id' => $team->id,
                    'title' => "ناهنجاری قیمت در خرید '{$itemName}'",
                    'message' => $result['analysis'],
                    'level' => 'warning',
                ]);
                $this->info("Anomaly detected for '{$itemName}' from '{$supplierName}'.");
            }
        }

        $this->info('Purchase history analysis completed.');
        return 0;
    }
}
