<?php

namespace App\Console\Commands;

use App\Models\Shipment;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;

class AutoConfirmDeliveries extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'benchsync:auto-confirm-deliveries';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically confirms shipments that have passed their delivery grace period.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting to check for overdue shipments...');

        // Define the grace period. For example, 7 days.
        $gracePeriodDays = 7;
        $cutoffDate = Carbon::now()->subDays($gracePeriodDays);

        // Find all shipments that were dispatched before the cutoff date
        // and are still in 'shipped' status.
        $overdueShipments = Shipment::where('status', 'shipped')
            ->where('dispatched_at', '<=', $cutoffDate)
            ->get();

        if ($overdueShipments->isEmpty()) {
            $this->info('No overdue shipments found. All good!');
            return 0;
        }

        $this->info("Found {$overdueShipments->count()} overdue shipment(s) to process.");

        foreach ($overdueShipments as $shipment) {
            $shipment->update([
                'status' => 'auto_delivered',
                'delivered_at' => now(),
            ]);
            $this->line("Shipment #{$shipment->id} for Purchase Request #{$shipment->purchase_request_id} was auto-confirmed.");
        }

        $this->info('All overdue shipments have been processed successfully.');
        return 0;
    }
}
