<?php

namespace App\Console\Commands;

use App\Models\InventoryItem;
use App\Services\AIService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ForecastInventoryStockout extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:forecast-inventory';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Forecasts the stockout date for inventory items using AI analysis of consumption patterns.';

    /**
     * Execute the console command.
     */
    public function handle(AIService $aiService)
    {
        $this->info('Starting inventory stockout forecasting...');

        // Get all items that have a consumption history
        $itemsToForecast = InventoryItem::whereHas('consumptionRequests')->get();

        if ($itemsToForecast->isEmpty()) {
            $this->info('No items with consumption history found to forecast.');
            return 0;
        }

        foreach ($itemsToForecast as $item) {
            $this->line("Forecasting for item: {$item->name} (SKU: {$item->sku})");

            // 1. Gather consumption data for the last 90 days
            $consumptionHistory = $item->consumptionRequests()
                ->where('type', 'consumption')
                ->where('status', 'approved')
                ->where('created_at', '>=', Carbon::now()->subDays(90))
                ->orderBy('created_at')
                ->get(['created_at', 'quantity']);

            if ($consumptionHistory->count() < 3) {
                $this->warn(" -> Skipping: Not enough consumption data (less than 3 records in 90 days).");
                continue;
            }

            // 2. Format data for the AI prompt
            $historyText = $consumptionHistory->map(fn($log) => "Date: " . $log->created_at->format('Y-m-d') . ", Quantity Used: " . $log->quantity)->implode("\n");
            $currentQuantity = $item->quantity;

            $prompt = "
                Analyze the following consumption data for an inventory item.
                Based on this historical trend, predict the date when the stock will run out.
                The current available quantity is {$currentQuantity} {$item->unit}.

                Consumption History (last 90 days):
                {$historyText}

                Rules:
                1. Calculate the average daily consumption based on the provided history.
                2. If consumption is zero or negative, you cannot predict a date.
                3. Based on the average consumption and current quantity, calculate the number of days until stockout.
                4. The current date is " . Carbon::now()->format('Y-m-d') . ".
                5. Your final output MUST be a valid JSON object with a single key: 'predicted_date'.
                6. The value of 'predicted_date' should be a string in 'YYYY-MM-DD' format.
                7. If you cannot make a prediction (e.g., no clear trend, zero consumption), the value should be null.
            ";

            // 3. Call the AI Service (using a generic method for flexibility)
            $result = $aiService->getStructuredResponse($prompt); // We'll create this helper method in AIService

            if ($result && !empty($result['predicted_date'])) {
                $predictedDate = Carbon::parse($result['predicted_date']);
                $item->update(['predicted_stockout_date' => $predictedDate]);
                $this->info(" -> Success! Predicted stockout date: " . $predictedDate->format('Y-m-d'));
            } else {
                $this->error(" -> Failed: AI could not predict a date for this item.");
                Log::warning("AI forecasting failed for item ID: {$item->id}", ['result' => $result]);
            }
        }

        $this->info('Inventory stockout forecasting completed successfully.');
        return 0;
    }
}
