<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use Spatie\Permission\Models\Role as SpatieRole;
use Spatie\Permission\PermissionRegistrar;

class MigrateOldRolesToSpatie extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'benchsync:migrate-roles';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migrates users and roles from the old custom roles table to the Spatie permission tables.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting role migration process...');

        // Reset Spatie's cache before starting
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        // **مهم**: نام جدول واسط قدیمی شما که رابطه کاربر و نقش را ذخیره می‌کند
        // بر اساس مدل User شما، این نام 'role_user' است. اگر متفاوت است، آن را تغییر دهید.
        $oldPivotTable = 'role_user';

        // 1. خواندن تمام نقش‌های قدیمی
        $oldRoles = DB::table('roles')->get();
        if ($oldRoles->isEmpty()) {
            $this->warn('Old roles table is empty. Nothing to migrate.');
            return 0;
        }
        $this->info('Found ' . $oldRoles->count() . ' roles in the old table. Creating them in Spatie...');

        foreach ($oldRoles as $oldRole) {
            // ایجاد نقش در سیستم Spatie (اگر از قبل وجود نداشته باشد)
            SpatieRole::firstOrCreate(['name' => $oldRole->name, 'guard_name' => 'web']);
            $this->line('Role "' . $oldRole->name . '" created or already exists in Spatie.');
        }

        // 2. خواندن تمام تخصیص‌های قدیمی کاربر به نقش
        $oldAssignments = DB::table($oldPivotTable)->get();
        if ($oldAssignments->isEmpty()) {
            $this->warn('Old role assignments pivot table is empty.');
            return 0;
        }
        $this->info('Found ' . $oldAssignments->count() . ' old role assignments. Re-assigning to users...');

        $progressBar = $this->output->createProgressBar($oldAssignments->count());
        $progressBar->start();

        foreach ($oldAssignments as $assignment) {
            $user = User::find($assignment->user_id);
            $roleName = $oldRoles->firstWhere('id', $assignment->role_id)->name;

            if ($user && $roleName) {
                // تخصیص نقش جدید از طریق Spatie
                $user->assignRole($roleName);
            }
            $progressBar->advance();
        }

        $progressBar->finish();
        $this->info("\nRole migration completed successfully!");

        return 0;
    }
}
