<?php

namespace App\Console\Commands;

use App\Models\Equipment;
use App\Notifications\CalibrationDueNotification;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Notification;
use Carbon\Carbon;

class SendCalibrationReminders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'equipment:send-reminders';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Send reminder notifications for equipment that is due for calibration soon.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Searching for equipment with approaching calibration due dates...');

        // یادآوری برای تجهیزاتی که ۷ روز، ۳ روز یا امروز موعدشان است
        $reminderDays = [7, 3, 0];

        foreach ($reminderDays as $days) {
            $targetDate = Carbon::today()->addDays($days)->toDateString();

            $equipmentDue = Equipment::with('responsibleUser')
                ->whereNotNull('responsible_user_id') // فقط تجهیزاتی که کاربر مسئول دارند
                ->whereDate('next_calibration_date', $targetDate)
                ->get();

            if ($equipmentDue->isNotEmpty()) {
                $this->info("Found {$equipmentDue->count()} equipment items due in {$days} day(s). Sending notifications...");

                foreach ($equipmentDue as $equipment) {
                    // ارسال اعلان به کاربر مسئول
                    Notification::send($equipment->responsibleUser, new CalibrationDueNotification($equipment, $days));
                }
            }
        }

        $this->info('Calibration reminder process completed.');
        return 0;
    }
}
