<?php

namespace App\Console\Commands;

use App\Models\Shipment;
use App\Notifications\ConfirmDeliveryReminder;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;

class SendDeliveryReminderNotifications extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'benchsync:send-reminders';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sends reminder notifications to buyers to confirm shipment delivery.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Checking for shipments that need a delivery confirmation reminder...');

        // Define on which days before the auto-confirmation we should send a reminder.
        // e.g., 4 days before (day 10) and 1 day before (day 13).
        $reminderIntervals = [4, 1];
        $gracePeriodDays = 14;
        $sentCount = 0;

        foreach ($reminderIntervals as $daysBefore) {
            // Calculate the exact date the shipment should have been dispatched
            // for a reminder to be sent today.
            $targetDispatchDate = Carbon::now()->subDays($gracePeriodDays - $daysBefore)->startOfDay();

            $shipmentsToRemind = Shipment::where('status', 'shipped')
                ->whereDate('dispatched_at', $targetDispatchDate)
                ->with('purchaseRequest.requester')
                ->get();

            if ($shipmentsToRemind->isNotEmpty()) {
                $this->info("Found {$shipmentsToRemind->count()} shipment(s) dispatched on {$targetDispatchDate->toDateString()} to remind.");

                foreach ($shipmentsToRemind as $shipment) {
                    if ($shipment->purchaseRequest && $shipment->purchaseRequest->requester) {
                        $shipment->purchaseRequest->requester->notify(new ConfirmDeliveryReminder($shipment));
                        $this->line("Reminder sent for Shipment #{$shipment->id}.");
                        $sentCount++;
                    }
                }
            }
        }

        if ($sentCount > 0) {
            $this->info("Successfully sent {$sentCount} reminder(s).");
        } else {
            $this->info('No reminders were needed today.');
        }

        return 0;
    }
}
