<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Carbon\Carbon;

class CollaboratorReportController extends Controller
{
    /**
     * Display a performance report for a specific collaborator on a specific project.
     *
     * @param  \App\Models\Project  $project
     * @param  \App\Models\User  $user
     * @return \Illuminate\View\View
     */
    public function show(Project $project, User $user)
    {
        // [UPDATED] ابتدا بررسی می‌کنیم که آیا کاربر فعلی اجازه مشاهده این گزارش را دارد یا خیر
        Gate::authorize('viewCollaboratorReports', $project);

        // واکشی وظایف محول شده به کاربر، فقط در همین طرح خاص
        $tasks = $user->assignedTasks()
            ->where('project_id', $project->id)
            ->get();

        // آماده‌سازی داده‌ها برای نمودار میله‌ای تجمیعی
        $tasksByMonth = $tasks->whereNotNull('due_date')->groupBy(function ($task) {
            return $task->due_date->format('Y-m');
        });

        $sortedTasksByMonth = $tasksByMonth->sortKeys();

        $labels = $sortedTasksByMonth->keys()->map(function ($dateKey) {
            return verta(Carbon::createFromFormat('Y-m', $dateKey))->format('F Y');
        })->all();

        $dataDone = [];
        $dataInProgress = [];
        $dataTodo = [];

        foreach ($sortedTasksByMonth as $monthGroup) {
            $dataDone[] = $monthGroup->where('status', 'انجام شده')->count();
            $dataInProgress[] = $monthGroup->where('status', 'در حال انجام')->count();
            $dataTodo[] = $monthGroup->where('status', 'برای انجام')->count();
        }

        $datasets = [
            ['label' => 'انجام شده', 'data' => $dataDone, 'backgroundColor' => '#22c55e'],
            ['label' => 'در حال انجام', 'data' => $dataInProgress, 'backgroundColor' => '#f59e0b'],
            ['label' => 'برای انجام', 'data' => $dataTodo, 'backgroundColor' => '#3b82f6'],
        ];

        $chartData = ['labels' => $labels, 'datasets' => $datasets];

        // ارسال تمام داده‌ها به view اختصاصی گزارش
        return view('reports.collaborator', [
            'project' => $project,
            'collaborator' => $user,
            'tasks' => $tasks,
            'chartData' => $chartData,
        ]);
    }
}
