<?php
//
//namespace App\Http\Controllers;
//
//use App\Services\PaymentService;
//use Illuminate\Http\Request;
//
//class PaymentController extends Controller
//{
//    public function callback(Request $request, PaymentService $paymentService)
//    {
//        $payment = $paymentService->verifyPayment($request->all());
//
//        if ($payment && $payment->status === 'successful') {
//            // Update the related purchase request status
//            $supplierResponse = $payment->payable;
//            $purchaseRequest = $supplierResponse->purchaseRequest;
//            $purchaseRequest->update(['status' => 'payment_successful']);
//
//            // Redirect with a success message
//            return redirect()->route('purchase-requests.index')
//                ->with('message', 'پرداخت با موفقیت انجام شد. وضعیت درخواست به‌روز شد.');
//        }
//
//        // Redirect with a failure message
//        return redirect()->route('purchase-requests.index')
//            ->with('error', 'پرداخت ناموفق بود یا توسط شما لغو شد.');
//    }
//}


namespace App\Http\Controllers;

use App\Models\PurchaseRequest;
use App\Models\SupplierResponse;
use App\Services\PaymentService;
use Illuminate\Http\Request;

class PaymentController extends Controller
{
    /**
     * Handle the callback from the payment gateway.
     *
     * @param Request $request
     * @param PaymentService $paymentService
     * @return \Illuminate\Http\RedirectResponse
     */
    public function callback(Request $request, PaymentService $paymentService)
    {
        $payment = $paymentService->verifyPayment($request->all());

        if ($payment && $payment->status === 'successful') {
            // [BUGFIX] Instead of blindly updating the status,
            // we now check if all invoices for the request are paid.
            $purchaseRequest = $payment->payable->purchaseRequest;
            $this->checkAndFinalizePurchaseRequest($purchaseRequest);

            // Redirect with a success message
            return redirect()->route('financials.index') // Redirecting to financial manager is more relevant
            ->with('message', 'پرداخت با موفقیت انجام شد. وضعیت فاکتور به‌روز شد.');
        }

        // Redirect with a failure message
        return redirect()->route('financials.index')
            ->with('error', 'پرداخت ناموفق بود یا توسط شما لغو شد.');
    }

    /**
     * Checks if all winning invoices for a purchase request have been paid.
     * If so, updates the purchase request's status to 'payment_successful'.
     *
     * @param PurchaseRequest $purchaseRequest
     */
    private function checkAndFinalizePurchaseRequest(PurchaseRequest $purchaseRequest): void
    {
        // Eager load relationships to avoid N+1 query problems.
        $purchaseRequest->load('items.finalResponseItem.response.payments');

        // Get all unique winning supplier responses (invoices) for this purchase request.
        $winningInvoices = $purchaseRequest->items
            ->where('status', 'offer_accepted')
            ->pluck('finalResponseItem.response')
            ->filter() // Remove any null values
            ->unique('id');

        // Count how many of those unique invoices have at least one successful payment.
        $paidInvoicesCount = $winningInvoices->filter(function (SupplierResponse $response) {
            return $response->payments->where('status', 'successful')->isNotEmpty();
        })->count();

        // If the number of total winning invoices equals the number of paid invoices,
        // it means the entire purchase request is fully paid.
        if ($winningInvoices->count() > 0 && $winningInvoices->count() === $paidInvoicesCount) {
            $purchaseRequest->update(['status' => 'payment_successful']);
        }
    }
}
