<?php

namespace App\Http\Controllers;

use App\Models\InventoryItem; // این خط اضافه می‌شود
use App\Models\SupplierResponse;
use App\Models\PurchaseRequest; // این خط اضافه می‌شود
use Illuminate\Http\Request;
use Mccarlosen\LaravelMpdf\Facades\LaravelMpdf as PDF; // فراخوانی فاساد پکیج جدید
use Picqer\Barcode\BarcodeGeneratorPNG; // پکیج جدید بارکد
use App\Models\StorageLocation; // این خط اضافه می‌شود
use App\Models\Equipment;
use App\Models\Auditor;
use App\Models\Payment; // [NEW]
use App\Models\Supplier; // [NEW]
use Illuminate\Support\Facades\Auth; // [NEW]
use App\Models\user;
use App\Policies\FinancialPolicy;
use Hekmatinasser\Verta\Verta;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;


class PdfController extends Controller
{
    use AuthorizesRequests;
    /**
     * Generate and stream a proforma invoice for a given supplier response.
     *
     * @param  \App\Models\SupplierResponse  $response
     * @return \Illuminate\Http\Response
     */
    public function downloadProformaInvoice(SupplierResponse $response)
    {
        // ما می‌توانیم در اینجا با استفاده از Policy، دسترسی را نیز کنترل کنیم
        // $this->authorize('view', $response);

        // داده‌های مورد نیاز را به صورت بهینه بارگذاری می‌کنیم
        $response->load('supplier', 'purchaseRequest.requester.currentTeam', 'items.originalItem');

        // ما از پکیج جدید برای تولید PDF استفاده می‌کنیم
        $pdf = PDF::loadView('pdf.proforma_invoice', ['response' => $response], [], [
            'format' => 'A4',
            'default_font' => 'IranSans' // استفاده از فونت فارسی که در پیکربندی تعریف کردیم
        ]);

        // فایل PDF را برای نمایش در مرورگر، به کاربر ارسال می‌کنیم
        return $pdf->stream('proforma-invoice-'.$response->id.'.pdf');
    }

    /**
     * Generate and stream a final invoice for a supplier's winning items on a specific request.
     */
    public function downloadOrderInvoice(PurchaseRequest $request)
    {
        // دریافت تأمین‌کننده لاگین‌کرده
        $supplier = auth()->guard('supplier')->user();

        // پیدا کردن تمام آیتم‌های برنده شده توسط این تأمین‌کننده برای این درخواست خاص
        $winningItems = $request->items()
            ->whereHas('finalResponseItem.response', function ($query) use ($supplier) {
                $query->where('supplier_id', $supplier->id);
            })
            ->with(['finalResponseItem.response.supplier', 'finalResponseItem'])
            ->get();

        // اگر هیچ آیتم برنده‌ای وجود نداشت، عملیات متوقف می‌شود
        if ($winningItems->isEmpty()) {
            abort(404, 'هیچ آیتم تایید شده‌ای برای این فاکتور یافت نشد.');
        }

        $response = SupplierResponse::where('purchase_request_id', $request->id)
            ->where('supplier_id', $supplier->id)
            ->first();

        $data = [
            'request' => $request->load('requester.currentTeam'),
            'supplier' => $supplier,
            'winningItems' => $winningItems,
            'response' => $response, // ارسال متغیر جدید
        ];

        $pdf = PDF::loadView('pdf.invoice', $data, [], [
            'format' => 'A4',
            'default_font' => 'IranSans'
        ]);

        return $pdf->stream('invoice-'.$request->request_code.'.pdf');
    }

    /**
     * Generate a shipping label for the Sender (Supplier).
     */
    public function downloadSenderLabel()
    {
        $supplier = auth()->guard('supplier')->user();

        $data = [
            'title' => 'اطلاعات فرستنده',
            'name' => $supplier->company_name,
            'address' => $supplier->address,
            'phone' => $supplier->phone,
        ];

        $pdf = PDF::loadView('pdf.shipping_label', $data, [], [
            'format' => [100, 150] // Standard label size in mm
        ]);

        return $pdf->stream('sender-label.pdf');
    }

    /**
     * Generate a shipping label for the Receiver (Requester's Institution).
     */
    public function downloadReceiverLabel(PurchaseRequest $request)
    {
        // We need to load the relationships to get the institution's details
        $request->load('requester.currentTeam');

        $institution = $request->requester->currentTeam;

        $data = [
            'title' => 'اطلاعات گیرنده',
            'name' => $institution->name,
            'address' => $institution->address, // Assuming Institution model has an 'address' field
            'phone' => $institution->phone,   // Assuming Institution model has a 'phone' field
        ];

        $pdf = PDF::loadView('pdf.shipping_label', $data, [], [
            'format' => [100, 150]
        ]);

        return $pdf->stream('receiver-label.pdf');
    }

    /**
     * Generate and stream a barcode label for a given inventory item.
     */
    public function downloadBarcodeLabel(InventoryItem $item)
    {
        // ما می‌توانیم در اینجا با استفاده از Policy، دسترسی را نیز کنترل کنیم
        // $this->authorize('view', $item);

        // ۱. یک نمونه از ژنراتور بارکد ایجاد می‌کنیم
        $generator = new BarcodeGeneratorPNG();

        // ۲. یک تصویر بارکد از کد SKU آیتم تولید کرده و آن را به صورت base64 انکود می‌کنیم
        $barcode = base64_encode($generator->getBarcode($item->sku, $generator::TYPE_CODE_128, 2, 50));

        $data = [
            'item' => $item,
            'barcode' => $barcode,
        ];

        // ۳. فایل Blade را با داده‌های مربوطه به PDF تبدیل می‌کنیم
        $pdf = PDF::loadView('pdf.barcode_label', $data, [], [
            // سایز استاندارد برای یک برچسب کوچک (مثلاً ۵۰ در ۲۵ میلی‌متر)
            'format' => [50, 25],
            'margin_left' => 2,
            'margin_right' => 2,
            'margin_top' => 2,
            'margin_bottom' => 2,
            'default_font' => 'IranSans'
        ]);

        // ۴. فایل PDF را برای نمایش در مرورگر، به کاربر ارسال می‌کنیم
        return $pdf->stream('barcode-'.$item->sku.'.pdf');
    }

    /**
     * Generate and stream barcode labels for multiple items.
     */
//    public function downloadBulkBarcodeLabels()
//    {
//        $itemIds = session()->pull('selected_barcode_items', []);
//
//        if (empty($itemIds)) {
//            abort(400, 'هیچ آیتمی برای چاپ انتخاب نشده است.');
//        }
//
//        $items = InventoryItem::whereIn('id', $itemIds)->get();
//        $barcodes = [];
//        $generator = new BarcodeGeneratorPNG();
//
//        foreach ($items as $item) {
//            $barcodes[$item->id] = base64_encode($generator->getBarcode($item->sku, $generator::TYPE_CODE_128, 2, 50));
//        }
//
//        $pdf = PDF::loadView('pdf.barcode_label', ['items' => $items, 'barcodes' => $barcodes], [], [
//            'format' => 'A4', // A4 page for multiple labels
//            'default_font' => 'IranSans'
//        ]);
//
//        return $pdf->stream('bulk-barcodes.pdf');
//    }

    public function downloadBulkBarcodeLabels(Request $request)
    {
        // ما شناسه‌ها را مستقیماً از درخواست GET دریافت می‌کنیم
        $itemIds = $request->input('items', []);

        if (empty($itemIds)) {
            abort(400, 'هیچ آیتمی برای چاپ انتخاب نشده است.');
        }

        $items = InventoryItem::whereIn('id', $itemIds)->get();
        $barcodes = [];
        $generator = new BarcodeGeneratorPNG();

        foreach ($items as $item) {
            $barcodes[$item->id] = base64_encode($generator->getBarcode($item->sku, $generator::TYPE_CODE_128, 2, 50));
        }

        $pdf = PDF::loadView('pdf.bulk_barcode_Label', ['items' => $items, 'barcodes' => $barcodes], [], [
            'format' => 'A4', // صفحه A4 برای چاپ چندین برچسب
            'default_font' => 'IranSans'
        ]);

        return $pdf->stream('bulk-barcodes.pdf');
    }

    /**
     * Generate and stream a barcode label for a single storage location.
     */
    public function downloadLocationBarcodeLabel(StorageLocation $location)
    {
        $generator = new BarcodeGeneratorPNG();
        $barcode = base64_encode($generator->getBarcode($location->location_code, $generator::TYPE_CODE_128, 2, 50));
        $pdf = PDF::loadView('pdf.location_barcode_label', ['locations' => collect([$location]), 'barcodes' => [$location->id => $barcode]], [], [
            'format' => [50, 50],
            'margin_left' => 2, 'margin_right' => 2, 'margin_top' => 2, 'margin_bottom' => 2,
            'default_font' => 'IranSans'
        ]);
        return $pdf->stream('location-barcode-'.$location->location_code.'.pdf');
    }

    /**
     * Generate and stream barcode labels for multiple locations.
     */
    public function downloadBulkLocationBarcodeLabels(Request $request)
    {
        $locationIds = $request->input('locations', []);
        if (empty($locationIds)) {
            abort(400, 'هیچ مکانی برای چاپ انتخاب نشده است.');
        }
        $locations = StorageLocation::whereIn('id', $locationIds)->get();
        $barcodes = [];
        $generator = new BarcodeGeneratorPNG();
        foreach ($locations as $location) {
            $barcodes[$location->id] = base64_encode($generator->getBarcode($location->location_code, $generator::TYPE_CODE_128, 2, 50));
        }
        $pdf = PDF::loadView('pdf.location_barcode_label', ['locations' => $locations, 'barcodes' => $barcodes], [], [
            'format' => 'A4',
            'default_font' => 'IranSans'
        ]);
        return $pdf->stream('bulk-location-barcodes.pdf');
    }
    /**
     * [NEW] Display a dedicated report page for an equipment's temperature log.
     *
     * @param  \App\Models\Equipment  $equipment
     * @return \Illuminate\View\View
     */
    public function showTemperatureLogReport(Equipment $equipment, Request $request)
    {
//        $this->authorize('view', $equipment->researchTeam);

        $equipment->load('location.parent', 'researchTeam');

        $logsQuery = $equipment->temperatureLogs()->with('auditor')->latest();

        // [FIXED] Convert Verta date to Carbon before using startOfDay/endOfDay
        if ($request->filled('start_date')) {
            $logsQuery->where('created_at', '>=', Verta::parse($request->input('start_date'))->toCarbon()->startOfDay());
        }
        if ($request->filled('end_date')) {
            $logsQuery->where('created_at', '<=', Verta::parse($request->input('end_date'))->toCarbon()->endOfDay());
        }
        if ($request->filled('auditor')) {
            $logsQuery->where('auditor_id', $request->input('auditor'));
        }

        // [FIXED] 1. Get all logs for the chart before paginating.
        $allLogsForChart = $logsQuery->clone()->get();

        // [FIXED] 2. Paginate the results for the table view.
        $paginatedLogs = $logsQuery->paginate(20);

        $chartData = [
            'labels' => $allLogsForChart->pluck('created_at')->map(fn($date) => verta($date)->format('Y/m/d H:i'))->reverse()->values(),
            'data' => $allLogsForChart->pluck('temperature')->reverse()->values(),
        ];

        $auditors = Auditor::where('research_team_id', $equipment->research_team_id)->where('is_active', true)->get();

        return view('reports.temperature-log-report', [
            'equipment' => $equipment,
            'logs' => $paginatedLogs,
            'chartData' => $chartData,
            'auditors' => $auditors,
        ]);
    }

    /**
     * [NEW] Generate and stream a PDF report of the financial transaction history.
     */
    public function downloadFinancialHistory(Request $request)
    {
        // Authorization check (can be expanded later)
        $user = Auth::user();
        // [FIXED] Manually check the policy to ensure correct authorization.
        $policy = new FinancialPolicy();
        if (!$policy->viewAny($user)) {
            abort(403, 'شما مجوز لازم برای دسترسی را ندارید.');
        }
        $managedTeamIds = $user->researchTeams()->wherePivot('role', 'admin')->pluck('research_teams.id');

        $historyQuery = Payment::query()->with([
            'user',
            'payable.supplier',
            'payable.purchaseRequest.project'
        ]);

        if (!$user->hasRole('purchasing_manager')) {
            $historyQuery->whereHasMorph('payable', [SupplierResponse::class], function ($query) use ($managedTeamIds) {
                $query->whereHas('purchaseRequest.project', function ($subQuery) use ($managedTeamIds) {
                    $subQuery->whereIn('research_team_id', $managedTeamIds);
                });
            });
        }

        // Apply filters from the request query string
        $historyQuery->when($request->input('supplier'), fn($q, $supplierId) => $q->whereHasMorph('payable', [SupplierResponse::class], fn($sq) => $sq->where('supplier_id', $supplierId)));
        $historyQuery->when($request->input('status'), fn($q, $status) => $q->where('status', 'like', $status));
        $historyQuery->when($request->input('start_date'), fn($q, $date) => $q->where('created_at', '>=', Verta::parse($date)->toCarbon()->startOfDay()));
        $historyQuery->when($request->input('end_date'), fn($q, $date) => $q->where('created_at', '<=', Verta::parse($date)->toCarbon()->endOfDay()));

        $data = [
            'logs' => $historyQuery->get(),
            'filters' => [
                'supplier' => $request->input('supplier') ? Supplier::find($request->input('supplier'))->company_name : 'همه',
                'status' => $request->input('status') ?: 'همه',
                'start_date' => $request->input('start_date') ?: '---',
                'end_date' => $request->input('end_date') ?: '---',
            ]
        ];

        $pdf = PDF::loadView('pdf.financial_history', $data, [], ['default_font' => 'IranSans']);
        return $pdf->stream('financial-history.pdf');
    }
}
