<?php

namespace App\Livewire\Admin;

use App\Models\Payment;
use App\Models\SupplierResponse;
use Illuminate\Database\Eloquent\Builder;
use Livewire\Component;
use Livewire\WithPagination;

class FinancialSettlement extends Component
{
    use WithPagination;

    public string $activeTab = 'pending_settlement';
    public $paymentToSettleId = null; // [FIXED] Changed from model object to ID
    public bool $confirmingSettlement = false;

    public function switchTab($tab)
    {
        $this->resetPage();
        $this->activeTab = $tab;
    }

    public function confirmSettlement(Payment $payment)
    {
        $this->paymentToSettleId = $payment->id; // [FIXED] Store only the ID
        $this->confirmingSettlement = true;
    }

    public function markAsSettled()
    {
        if (!$this->paymentToSettleId) {
            return;
        }

        // [FIXED] Re-fetch the model from the database using the stored ID
        $payment = Payment::find($this->paymentToSettleId);

        if ($payment) {
            // Add authorization check for admin/financial manager role here
            // abort_unless(auth()->user()->hasRole('Super Admin'), 403);

            $payment->update([
                'settlement_status' => 'settled',
                'settled_at' => now(),
            ]);

            session()->flash('message', 'تسویه حساب با موفقیت ثبت شد.');
        } else {
            session()->flash('error', 'خطا: تراکنش مورد نظر یافت نشد.');
        }


        $this->confirmingSettlement = false;
        $this->paymentToSettleId = null; // Reset the ID
    }

    public function render()
    {
        $query = Payment::query()
            ->where('status', 'successful')
            ->whereHasMorph('payable', [SupplierResponse::class], function (Builder $query) {
                // Ensure the related shipment has been delivered
                $query->whereHas('purchaseRequest.shipments', function (Builder $shipmentQuery) {
                    $shipmentQuery->whereIn('status', ['delivered', 'auto_delivered']);
                });
            })
            ->with(['payable.supplier', 'payable.purchaseRequest']);

        if ($this->activeTab === 'pending_settlement') {
            $query->where('settlement_status', 'pending');
        } else {
            $query->where('settlement_status', 'settled');
        }

        $payments = $query->latest('paid_at')->paginate(10);

        return view('livewire.admin.financial-settlement', [
            'payments' => $payments,
        ])->layout('layouts.admin');
    }
}
