<?php

namespace App\Livewire\Admin;

use App\Models\Institution;
use App\Models\InstitutionRequest;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Livewire\Component;

class InstitutionRequestManager extends Component
{
    public ?InstitutionRequest $viewingRequest = null;
    public $confirmingRequestView = false;
    // Properties for the new "Create Contact User" modal
    public $confirmingContactUserCreation = false;
    public $contact_first_name, $contact_last_name, $contact_username, $contact_email, $contact_password, $contact_password_confirmation;

    // Property for search filter
    public $search = '';


    /**
     * Show the modal with the full details of the request.
     */
    public function viewRequest(InstitutionRequest $request)
    {
        $this->viewingRequest = $request->load('user', 'contactPersonUser');
        $this->confirmingRequestView = true;
    }

    /**
     * Show the modal for creating the contact person's user account.
     */
    public function confirmContactUserCreation()
    {
        $this->resetErrorBag();
        // Pre-fill the name from the request if possible
        $fullName = explode(' ', $this->viewingRequest->contact_person_name, 2);
        $this->contact_first_name = $fullName[0] ?? '';
        $this->contact_last_name = $fullName[1] ?? '';
        $this->reset(['contact_email', 'contact_username', 'contact_password', 'contact_password_confirmation']);

        $this->confirmingContactUserCreation = true;
    }

    /**
     * Create the new user account for the contact person.
     */
    public function createContactUser()
    {
        $this->validate([
            'contact_first_name' => 'required|string|max:255',
            'contact_last_name' => 'required|string|max:255',
            'contact_username' => 'required|string|max:255|unique:users,username',
            'contact_email' => 'required|string|email|max:255|unique:users,email',
            'contact_password' => 'required|string|min:8|confirmed',
        ]);

        $user = User::create([
            'first_name' => $this->contact_first_name,
            'last_name' => $this->contact_last_name,
            'username' => $this->contact_username,
            'email' => $this->contact_email,
            'password' => Hash::make($this->contact_password),
        ]);

        // --- منطق جدید: اتصال کاربر رابط به درخواست ---

        $this->viewingRequest->update([
            'contact_person_user_id' => $user->id
        ]);

        $this->viewingRequest->load('contactPersonUser'); // Refresh the relationship
        $this->confirmingContactUserCreation = false;
        session()->flash('message', 'حساب کاربری برای فرد رابط با موفقیت ایجاد شد.');
    }
    /**
     * Approve the request, create the institution, and assign the user as admin.
     */
    /**
     * Approve the request, create the institution, and assign the user as admin.
     */
    public function approveRequest()
    {
        if (!$this->viewingRequest) {
            return;
        }

        DB::transaction(function () {
            // Step 1: Create a unique code for the new institution
            do {
                $generatedCode = strtoupper(Str::random(3));
            } while (Institution::where('code', $generatedCode)->exists());

            // Step 2: Create the new institution record
            $institution = Institution::create([
                'name' => $this->viewingRequest->name,
                'code' => $generatedCode,
                'type' => $this->viewingRequest->type,
                'address' => $this->viewingRequest->address,
                'phone' => $this->viewingRequest->phone,
                'website' => $this->viewingRequest->website,
                'contact_person_id' => $this->viewingRequest->contact_person_user_id ?? $this->viewingRequest->user_id,
                'status' => 'approved',
            ]);

            // Step 3: Find the pending membership request for the main user and approve it
            $membership = $this->viewingRequest->user->institutionMemberships()
                ->where('status', 'pending')->first();

            if ($membership) {
                $membership->update([
                    'institution_id' => $institution->id,
                    'role' => 'admin',
                    'status' => 'approved',
                ]);
            } else {
                // Or create a new one if it doesn't exist for some reason
                $institution->memberships()->create([
                    'user_id' => $this->viewingRequest->user_id,
                    'role' => 'admin',
                    'status' => 'approved',
                ]);
            }

            // Step 4: If a contact person user was created, create their membership too
            if ($this->viewingRequest->contact_person_user_id) {
                $institution->memberships()->create([
                    'user_id' => $this->viewingRequest->contact_person_user_id,
                    'role' => 'member', // یا هر نقش پیش‌فرض دیگری
                    'status' => 'approved',
                ]);
            }

            // Step 5: Update the request status
            $this->viewingRequest->update(['status' => 'approved']);
        });

        $this->confirmingRequestView = false;
        session()->flash('message', 'درخواست با موفقیت تایید و سازمان جدید ایجاد شد.');
    }

    /**
     * Reject the institution creation request.
     */
    public function rejectRequest()
    {
        if (!$this->viewingRequest) {
            return;
        }
        $this->viewingRequest->update(['status' => 'rejected']);
        $this->confirmingRequestView = false;
        session()->flash('message', 'درخواست ایجاد سازمان با موفقیت رد شد.');
    }

    public function render()
    {
        $requestsQuery = InstitutionRequest::where('status', 'approved')
            ->with('user');

        if (!empty($this->search)) {
            $requestsQuery->where(function($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('tracking_code', 'like', '%' . $this->search . '%');
            });
        }

        $requests = $requestsQuery->latest()->get();

        return view('livewire.admin.institution-request-manager', [
            'requests' => $requests,
        ])->layout('layouts.admin');
    }
}



