<?php

namespace App\Livewire;

use App\Models\PurchaseRequest;
use App\Models\Project;
use App\Models\Supplier;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class ApprovalQueueManager extends Component
{
    use AuthorizesRequests;

    public ?PurchaseRequest $viewingRequest = null;
    public $confirmingRequestView = false;

    // Properties for the editable form inside the modal
    public $items = [];
    public $notes, $priority;

    // Properties for Assigned Supplier
    public $assigned_supplier_id;
    public string $supplierSearch = '';
    public $supplierSearchResults = [];
// Property to hold the original state for comparison
    public array $originalState = [];
    protected function rules()
    {
        return [
            'notes' => 'nullable|string',
            'priority' => 'required|in:normal,high,low',
            'items' => 'required|array|min:1',
            'items.*.item_name' => 'required|string|max:255',
            'items.*.description' => 'nullable|string',
            'items.*.quantity' => 'required|numeric|min:1',
            'items.*.unit' => 'required|string|max:50',
            'items.*.supplier_id' => 'nullable|exists:suppliers,id',
        ];
    }


    /**
     * Show the modal with the full details of the request in an editable form.
     */
    public function viewRequest(PurchaseRequest $request)
    {
        $this->resetErrorBag();
        $this->viewingRequest = $request->load('requester', 'project', 'items.supplier', 'assignedSupplier');

        // Populate the editable properties
        $this->notes = $this->viewingRequest->notes;
        $this->priority = $this->viewingRequest->priority;
        $this->items = $this->viewingRequest->items->toArray();

        // Populate supplier info
        $this->assigned_supplier_id = $this->viewingRequest->assigned_supplier_id;
        $this->supplierSearch = $this->viewingRequest->assignedSupplier?->company_name ?? '';
        $this->supplierSearchResults = [];

        // Store the original state for comparison later
        $this->originalState = [
            'notes' => $this->notes,
            'priority' => $this->priority,
            'items' => $this->items,
        ];

        $this->confirmingRequestView = true;
    }

    /**
     * Check if any data in the form has been changed.
     */
    private function hasChanges(): bool
    {
        return $this->originalState['notes'] !== $this->notes ||
            $this->originalState['priority'] !== $this->priority ||
            $this->originalState['items'] !== $this->items;
    }

    public function updatedSupplierSearch()
    {
        if (strlen($this->supplierSearch) < 2) {
            $this->supplierSearchResults = [];
            return;
        }
        $this->supplierSearchResults = Supplier::where('company_name', 'like', '%' . $this->supplierSearch . '%')
            ->take(5)
            ->get();
    }

    public function selectSupplier(Supplier $supplier)
    {
        $this->assigned_supplier_id = $supplier->id;
        $this->supplierSearch = $supplier->company_name;
        $this->supplierSearchResults = [];
    }

    public function clearSelectedSupplier()
    {
        $this->assigned_supplier_id = null;
        $this->supplierSearch = '';
        $this->supplierSearchResults = [];
    }

    public function addItemRow()
    {
        $this->items[] = ['item_name' => '', 'description' => '', 'quantity' => 1, 'unit' => '', 'supplier_id' => null];
    }

    public function removeItemRow($index)
    {
        unset($this->items[$index]);
        $this->items = array_values($this->items);
    }

    /**
     * Update and then approve the purchase request.
     */
    public function approveRequest()
    {
        if (!$this->viewingRequest) return;

//        // First, update the request with any changes
//        $this->viewingRequest->update($this->validate([
//            'notes' => 'nullable|string',
//            'priority' => 'required|in:normal,high,low',
//            'assigned_supplier_id' => 'nullable|exists:suppliers,id',
//        ]));
//
//        // Then, update the items
//        $this->validate(['items' => 'required|array|min:1']);
//        $this->viewingRequest->items()->delete(); // Simple way: delete old items
//        foreach($this->items as $item) {
//            $this->viewingRequest->items()->create($item); // Create new ones
//        }

        // --- منطق جدید و هوشمندانه ---
        // فقط در صورتی که تغییری ایجاد شده باشد، اعتبارسنجی و به‌روزرسانی را انجام می‌دهیم
        if ($this->hasChanges()) {
            $this->validate();

            $this->viewingRequest->update([
                'notes' => $this->notes,
                'priority' => $this->priority,
            ]);

            $this->viewingRequest->items()->delete();
            foreach($this->items as $item) {
                $this->viewingRequest->items()->create($item);
            }
        }

        // Finally, approve the request
        $this->viewingRequest->update([
            'status' => 'pending_merchant', // وضعیت به "در انتظار بازرگان" تغییر می‌کند
            'approved_at' => now(),
        ]);

        $this->confirmingRequestView = false;
        session()->flash('message', 'درخواست با موفقیت تایید و برای بازرگان ارسال شد.');
    }

    /**
     * Reject the purchase request.
     */
    /**
     * Reject the purchase request.
     */
    public function rejectRequest()
    {
        if (!$this->viewingRequest) {
            return;
        }

        $this->viewingRequest->update([
            'status' => 'rejected_by_owner', // وضعیت به "رد شده توسط مالک" تغییر می‌کند
        ]);

        $this->confirmingRequestView = false;
        session()->flash('message', 'درخواست با موفقیت رد شد.');
    }


    public function render()
    {
        // ما فقط درخواست‌هایی را می‌خوانیم که کاربر فعلی، تأیید کننده آن‌هاست
        // و وضعیت آن‌ها "در انتظار تایید" است.
        $requests = PurchaseRequest::where('approver_id', Auth::id())
            ->where('status', 'pending_approval')
            ->with(['requester', 'project'])
            ->withCount('items') // شمارش تعداد آیتم‌های هر درخواست
            ->latest()
            ->get();

        $suppliers = Supplier::all();

        return view('livewire.approval-queue-manager', [
            'requests' => $requests,
            'suppliers' => $suppliers
        ])->layout('layouts.app');
    }
}
