<?php

namespace App\Livewire;

use App\Models\InventoryItem;
use App\Models\ResearchTeam;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use App\Models\StorageLocation;
use Hekmatinasser\Verta\Verta;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\BiologicalSampleRequest;
use Illuminate\Support\Collection;
use Livewire\WithPagination;

class BioInventoryManager extends Component
{
    use AuthorizesRequests;
    use WithPagination;

    // --- [NEW] Properties for Tank Creation Modal ---
    public bool $confirmingTankCreation = false;
    public array $newTankState = [];
    // Properties for Create/Edit Modals
    public bool $confirmingItemAction = false;
    public ?InventoryItem $editingItem = null;
    public array $state = [];
    public string $generatedSku = '---';

    // Properties for Search and Filter
    public string $searchQuery = '';
    public $searchTeamId = '';
    protected $predefinedCellTypes = ['رده سلولی', 'پرایمری', 'نمونه بافت'];


    // --- Properties for Tank & Location Management (Merged from StorageLocationManager) ---
    public bool $confirmingLocationCreation = false;
    public ?StorageLocation $parentLocation = null;
    public array $newLocationState = [];
    public $allowedLocationTypes = [];
    public $canHaveAutoChildren = false;
    public ?StorageLocation $editingLocation = null;
    public bool $confirmingLocationEdit = false;
    public ?StorageLocation $deletingLocation = null;
    public bool $confirmingLocationDeletion = false;
    public $locations;

    // --- Properties for Search and Filter ---

//    public $researchTeams = [];
    public Collection $researchTeams;
    // [NEW] Properties for Checkout Modal
    public bool $confirmingCheckout = false;
    public array $checkoutState = [];
    public ?InventoryItem $itemForCheckout = null;
    public Collection $authorizedHandlers;

    // [NEW] Properties for managing checkout requests, merged from SampleRequestManager
    public string $statusFilter = 'all';
    public bool $confirmingViewRequest = false;
    public ?BiologicalSampleRequest $viewingRequest = null;
    public bool $confirmingReturnRequest = false;
    public ?BiologicalSampleRequest $returningRequest = null;
    // [FIX] Listeners for events dispatched from the tank-item component
    protected $listeners = [
//        'confirmLocationCreation' => 'confirmTankCreation',
        'confirmLocationCreation' => 'confirmLocationCreation',
        'confirmLocationEdit' => 'confirmLocationEdit',
        'confirmLocationDeletion' => 'confirmLocationDeletion',
    ];

    protected function rules()
    {
        return [
            'state.name' => 'required|string|max:255',
            'state.research_team_id' => 'required|exists:research_teams,id',
            'state.quantity' => 'required|numeric|min:0',
            'state.unit' => 'required|string|max:50',
            // New biological fields
            'state.passage_number' => 'nullable|integer|min:0',
            'state.freeze_date' => 'nullable|date',
            'state.cell_type' => 'nullable|string|max:255',
            'state.custom_cell_type' => 'required_if:state.cell_type,سایر|nullable|string|max:255',
            'state.media_used' => 'nullable|string|max:255',
            // Other fields
            'state.price' => 'nullable|numeric|min:0',
            'state.brand' => 'nullable|string|max:255',
            'state.lot_number' => 'nullable|string|max:255',
        ];
    }

    // The hierarchical rules for locations
    protected const HIERARCHY_RULES = [
        'tank' => ['canister'],
        'canister' => ['rack', 'cane'],
        'rack' => ['box'],
        'cane' => [], // Canes hold items directly
        'box' => [], // Boxes hold items directly
    ];

    public function mount()
    {
//        $user = Auth::user();
//        $this->researchTeams = $user->researchTeams()->get();
//        $this->loadLocations();
        $this->researchTeams = $this->getAuthorizedTeams();
        $this->loadLocations();
    }


    /**
     * [NEW] A helper function to get the IDs of the teams the current user is authorized to manage.
     */
    private function getAuthorizedTeamIds(): Collection
    {
        $user = Auth::user();
        $teamIds = collect();

        // Super admins and institution admins can see all teams in their institution
        if ($user->hasRole('super_admin') || $user->hasRole('institution_admin')) {
            return ResearchTeam::where('institution_id', $user->institutionMemberships->first()->institution_id)->pluck('id');
        }

        // Team managers and warehouse managers can see their specific teams
        foreach ($user->researchTeams as $team) {
//            if ($user->hasRole("team_manager") || $user->hasRole("Warehouse_Manager-Team-{$team->id}")) {
            if ($team->pivot->role === 'admin' || $user->hasRole("Warehouse_Manager-Team-{$team->id}")) {
                $teamIds->push($team->id);
            }
        }

        return $teamIds->unique();
    }

    /**
     * [NEW] A helper function to get the full team models the user is authorized to manage.
     */
    private function getAuthorizedTeams(): Collection
    {
        $teamIds = $this->getAuthorizedTeamIds();
        return ResearchTeam::whereIn('id', $teamIds)->get();
    }

    // ========== LOCATION MANAGEMENT METHODS (from StorageLocationManager) ==========

    public function loadLocations()
    {
        $userTeamIds = $this->researchTeams->pluck('id');
        $allLocations = StorageLocation::whereIn('research_team_id', $userTeamIds)
            ->whereNull('parent_id') // Only load top-level locations
            ->where('location_type', 'tank') // Specifically load tanks
            ->with('children.children.children') // Eager load the entire hierarchy
            ->get();
        $this->locations = $allLocations->whereNull('parent_id');
    }

//    public function confirmLocationCreation($parentId = null)
//    {
//        $this->resetErrorBag();
//        $this->newLocationState = [
//            'name' => '',
//            'location_code' => 'TANK-' . strtoupper(str()->random(6)),
//            'location_type' => 'tank',
//            'internal_structure_type' => 'rack_based',
//            'canister_count' => 6,
//            'items_per_canister' => 5,
//        ];
//
//        if ($parentId) {
//            $this->parentLocation = StorageLocation::find($parentId);
//            $this->allowedLocationTypes = self::HIERARCHY_RULES[$this->parentLocation->location_type] ?? [];
//            $this->newLocationState['location_type'] = $this->allowedLocationTypes[0] ?? ''; // Set default type
//        } else {
//            $this->parentLocation = null;
//            $this->allowedLocationTypes = ['tank'];
//        }
//
//        $this->confirmingLocationCreation = true;
//    }
    public function confirmLocationCreation($parentId = null)
    {
        $this->resetErrorBag();
        $this->reset('newLocationState', 'parentLocation', 'allowedLocationTypes');

        if ($parentId) {
            $this->parentLocation = StorageLocation::find($parentId);
            $this->allowedLocationTypes = self::HIERARCHY_RULES[$this->parentLocation->location_type] ?? [];

            // Prepare the state for creating a child
            $this->newLocationState = [
                'name' => '',
                'location_type' => $this->allowedLocationTypes[0] ?? '', // Set default type
                'rows' => null, // For boxes/racks
                'columns' => null,
            ];
        } else {
            // Prepare the state for creating a new top-level tank
            $this->parentLocation = null;
            $this->allowedLocationTypes = ['tank'];
            $this->newLocationState = [
                'research_team_id' => '',
                'name' => '',
                'location_code' => 'TANK-' . strtoupper(str()->random(6)),
                'location_type' => 'tank',
                'internal_structure_type' => 'rack_based',
                'canister_count' => 6,
                'items_per_canister' => 5,
            ];
        }

        $this->confirmingLocationCreation = true;
    }
    /**
     * [REWRITTEN] The complete and non-summarized method for saving a new Nitrogen Tank.
     * This method is adapted from the logic of StorageLocationManager.
     */
    public function saveNewTank()
    {
        $validatedData = $this->validate([
            'newLocationState.research_team_id' => 'required|exists:research_teams,id',
            'newLocationState.name' => 'required|string|max:255',
            'newLocationState.location_code' => 'required|string|unique:storage_locations,location_code',
            'newLocationState.internal_structure_type' => 'required|in:rack_based,cane_based',
            'newLocationState.canister_count' => 'required|integer|min:1|max:20',
            'newLocationState.items_per_canister' => 'required|integer|min:1|max:50',
        ])['newLocationState'];

        DB::transaction(function () use ($validatedData) {
            // 1. Create the main tank record
            $tank = StorageLocation::create([
                'uuid' => Str::uuid(),
                'research_team_id' => $validatedData['research_team_id'],
                'name' => $validatedData['name'],
                'location_code' => $validatedData['location_code'],
                'location_type' => 'tank',
                'internal_structure_type' => $validatedData['internal_structure_type'],
                'path' => '/', // Top-level locations have a root path
            ]);

            // 2. Create the canisters inside the tank
            for ($i = 1; $i <= $validatedData['canister_count']; $i++) {
                $canister = $tank->children()->create([
                    'uuid' => Str::uuid(),
                    'research_team_id' => $validatedData['research_team_id'],
                    'name' => $i,
                    'location_type' => 'canister',
                    'location_code' => 'C-' . strtoupper(Str::random(7)),
                    'path' => $tank->path . $tank->id . '/', // Set parent path
                ]);

                // 3. Create racks or canes inside each canister
                $childType = $validatedData['internal_structure_type'] === 'rack_based' ? 'rack' : 'cane';
                for ($j = 1; $j <= $validatedData['items_per_canister']; $j++) {
                    $canister->children()->create([
                        'uuid' => Str::uuid(),
                        'research_team_id' => $validatedData['research_team_id'],
                        'name' => $j,
                        'location_type' => $childType,
                        'location_code' => 'RC-' . strtoupper(Str::random(7)),
                        'path' => $canister->path . $canister->id . '/', // Set parent path
                    ]);
                }
            }
        });


        $this->confirmingLocationCreation = false;
        session()->flash('message', 'تانک ازت جدید با ساختار داخلی آن با موفقیت ایجاد شد.');
        // Optionally, refresh the list of locations for displaying them
         $this->loadLocations();
    }

    /**
     * [NEW] Opens the modal for editing a location's name.
     */
    public function confirmLocationEdit($locationId)
    {
        $this->resetErrorBag();
        $this->editingLocation = StorageLocation::find($locationId);
        if ($this->editingLocation) {
            $this->newTankState['name'] = $this->editingLocation->name;
            $this->confirmingLocationEdit = true;
        }
    }

    /**
     * [NEW] Updates the location's name.
     */
//    public function updateLocation()
//    {
//        $this->validate(['newLocationState.name' => 'required|string|max:255']);
//        $this->editingLocation->update(['name' => $this->newLocationState['name']]);
//        $this->confirmingLocationEdit = false;
//        $this->loadLocations();
//        session()->flash('message', 'نام مکان با موفقیت ویرایش شد.');
//    }
    public function updateLocation()
    {
        $this->validate(['newLocationState.name' => 'required|string|max:255']);
        if ($this->editingLocation) {
            $this->editingLocation->update(['name' => $this->newLocationState['name']]);
        }
        $this->confirmingLocationEdit = false;
        $this->loadLocations();
        session()->flash('message', 'نام مکان با موفقیت ویرایش شد.');
    }

    /**
     * [NEW] Opens the modal for deleting a location.
     */
    public function confirmLocationDeletion($locationId)
    {
        $this->deletingLocation = StorageLocation::find($locationId);
        if ($this->deletingLocation) {
            $this->confirmingLocationDeletion = true;
        }
    }

    /**
     * [NEW] Deletes a location and all its children.
     */
//    public function deleteLocation()
//    {
//        // Add checks here to prevent deleting locations with items if needed
//        $this->deletingLocation->delete();
//        $this->confirmingLocationDeletion = false;
//        $this->loadLocations();
//        session()->flash('message', 'مکان و تمام زیرمجموعه‌های آن با موفقیت حذف شدند.');
//    }
    public function deleteLocation()
    {
        if ($this->deletingLocation) {
            // Now, simply calling delete() on the parent is enough.
            // The 'deleting' event in the StorageLocation model will handle the rest.
            $this->deletingLocation->delete();
        }

        $this->confirmingLocationDeletion = false;
        $this->loadLocations(); // Refresh the view
        session()->flash('message', 'مکان و تمام زیرمجموعه‌های آن با موفقیت حذف شدند.');
    }

    public function saveLocation()
    {
        // If it's a new Tank
        if (!$this->parentLocation) {
            $this->saveNewTank();
        } else {
            // Logic for saving a new sub-location (canister, rack, etc.)
            $this->saveNewSubLocation();
        }
    }

    /**
     * [NEW] Logic to save a new sub-location (canister, rack, box, cane).
     */
    private function saveNewSubLocation()
    {
        $this->validate([
            'newLocationState.name' => 'required|string|max:255',
            'newLocationState.location_type' => 'required|in:' . implode(',', $this->allowedLocationTypes),
            'newLocationState.rows' => 'required_if:newLocationState.location_type,box|nullable|integer|min:1',
            'newLocationState.columns' => 'required_if:newLocationState.location_type,box|nullable|integer|min:1',
        ]);

        $this->parentLocation->children()->create([
            'uuid' => Str::uuid(),
            'research_team_id' => $this->parentLocation->research_team_id,
            'name' => $this->newLocationState['name'],
            'location_type' => $this->newLocationState['location_type'],
            'location_code' => strtoupper($this->newLocationState['location_type'][0]) . '-' . strtoupper(Str::random(7)),
            'rows' => $this->newLocationState['rows'],
            'columns' => $this->newLocationState['columns'],
            'path' => $this->parentLocation->path . $this->parentLocation->id . '/',
        ]);

        $this->confirmingLocationCreation = false;
        $this->loadLocations();
        session()->flash('message', 'زیرمجموعه جدید با موفقیت ایجاد شد.');
    }

// ========== BIOLOGICAL ITEM MANAGEMENT METHODS ==========
    public function confirmItemAction(?InventoryItem $item = null)
    {
        $this->resetErrorBag();
        if ($item && $item->exists) {
            $this->editingItem = $item;
            $this->state = $item->toArray();
            // [NEW] Logic to handle the 'cell_type' dropdown when editing
            if (!in_array($item->cell_type, $this->predefinedCellTypes)) {
                $this->state['custom_cell_type'] = $item->cell_type;
                $this->state['cell_type'] = 'سایر';
            }
            // [NEW] Format date for Jalali date picker
            if ($item->freeze_date) {
                $this->state['freeze_date'] = Verta::instance($item->freeze_date)->format('Y/m/d');
            }
        } else {
            $this->editingItem = null;
            $this->state = []; // Reset state for new item
            $this->generatedSku = '---';
        }
        $this->confirmingItemAction = true;
    }
    /**
     * [NEW] This hook is called when the user selects a research team in the form.
     * It triggers the SKU generation process.
     *
     * @param int $teamId
     */
    public function updatedStateResearchTeamId($teamId)
    {
        if ($teamId) {
            $team = ResearchTeam::find($teamId);
            if ($team && $team->research_teams_code) {
                $this->generateSku($team->research_teams_code);
            } else {
                $this->generatedSku = 'کد تیم یافت نشد';
            }
        } else {
            $this->generatedSku = '---';
        }
    }

    /**
     * [NEW] Generates a unique SKU based on the specified format.
     *
     * @param string $teamCode
     */
    private function generateSku(string $teamCode)
    {
        do {
            // Generate a 5-digit random number
            $randomNumber = str_pad(rand(0, 99999), 5, '0', STR_PAD_LEFT);
            $sku = 'BIO-' . $teamCode . '-' . $randomNumber;
        } while (InventoryItem::where('sku', $sku)->exists());

        $this->generatedSku = $sku;
    }

    public function saveItem()
    {
        // [NEW] Convert Jalali date to Gregorian before validation
        try {
            if (!empty($this->state['freeze_date'])) {
                $this->state['freeze_date'] = Verta::parse($this->state['freeze_date'])->toCarbon();
            }
        } catch (\Exception $e) {
            $this->addError('state.freeze_date', 'فرمت تاریخ نامعتبر است.');
            return;
        }

        $validatedData = $this->validate();

        if ($validatedData['state']['cell_type'] === 'سایر') {
            $validatedData['state']['cell_type'] = $validatedData['state']['custom_cell_type'];
        }
        unset($validatedData['state']['custom_cell_type']); // Remove temporary field
        // Set mandatory fields for biological items
        $validatedData['state']['is_biological'] = true;
        $validatedData['state']['item_type'] = 'biological_sample'; // Or make this selectable
        $validatedData['state']['owner_id'] = Auth::id();

        if ($this->editingItem) {
            $this->editingItem->update($validatedData['state']);
            session()->flash('message', 'نمونه با موفقیت به‌روزرسانی شد.');
        } else {
            if ($this->generatedSku === '---' || $this->generatedSku === 'کد تیم یافت نشد') {
                $this->addError('state.research_team_id', 'لطفاً یک تیم معتبر برای تولید کد شناسایی انتخاب کنید.');
                return;
            }
            $validatedData['state']['sku'] = $this->generatedSku;
            InventoryItem::create($validatedData['state']);
            session()->flash('message', 'نمونه جدید با موفقیت ثبت شد.');
        }

        $this->confirmingItemAction = false;
    }

    /**
     * [NEW] Opens the modal for creating a new nitrogen tank.
     */
//    public function confirmTankCreation($parentId = null)
//    {
//
//        $this->resetErrorBag();
//        $this->newTankState = [
//            'research_team_id' => '',
//            'name' => '',
//            'location_code' => 'TANK-' . strtoupper(str()->random(7)),
//            'location_type' => 'tank',
//            'internal_structure_type' => 'rack_based',
//            'canister_count' => 6,
//            'items_per_canister' => 5, // Represents racks or canes
//        ];
//        $this->confirmingTankCreation = true;
//    }

    /**
     * [NEW] Validates and saves the new tank and its internal structure.
     */
//    public function saveTank()
//    {
//        $validatedData = $this->validate([
//            'newTankState.research_team_id' => 'required|exists:research_teams,id',
//            'newTankState.name' => 'required|string|max:255',
//            'newTankState.location_code' => 'required|string|unique:storage_locations,location_code',
//            'newTankState.internal_structure_type' => 'required|in:rack_based,cane_based',
//            'newTankState.canister_count' => 'required|integer|min:1|max:20',
//            'newTankState.items_per_canister' => 'required|integer|min:1|max:20',
//        ])['newTankState'];
//
//        DB::transaction(function () use ($validatedData) {
//            // 1. Create the main tank record
//            $tank = StorageLocation::create([
//                'uuid' => Str::uuid(),
//                'research_team_id' => $validatedData['research_team_id'],
//                'name' => $validatedData['name'],
//                'location_code' => $validatedData['location_code'],
//                'location_type' => 'tank',
//                'internal_structure_type' => $validatedData['internal_structure_type'],
//            ]);
//
//            // 2. Create the canisters inside the tank
//            for ($i = 1; $i <= $validatedData['canister_count']; $i++) {
//                $canister = $tank->children()->create([
//                    'uuid' => Str::uuid(),
//                    'research_team_id' => $validatedData['research_team_id'],
//                    'name' => $i,
//                    'location_type' => 'canister',
//                    'location_code' => 'canister-' . strtoupper(str()->random(7)),
//
//                ]);
//
//                // 3. Create racks or canes inside each canister
//                $childType = $validatedData['internal_structure_type'] === 'rack_based' ? 'rack' : 'cane';
//                for ($j = 1; $j <= $validatedData['items_per_canister']; $j++) {
//                    $canister->children()->create([
//                        'uuid' => Str::uuid(),
//                        'research_team_id' => $validatedData['research_team_id'],
//                        'name' => $j,
//                        'location_type' => $childType,
//                        'location_code' => 'RC-' . strtoupper(str()->random(7)),
//                    ]);
//                }
//            }
//        });
//
//        $this->confirmingTankCreation = false;
//        session()->flash('message', 'تانک ازت جدید با موفقیت ایجاد شد.');
//    }

    /**
     * [NEW] Opens the checkout request modal.
     * This method is now part of the main manager component.
     */
    public function confirmCheckout($itemId)
    {
        $this->itemForCheckout = InventoryItem::find($itemId);
        if (!$this->itemForCheckout) {
            session()->flash('error', 'نمونه مورد نظر یافت نشد.');
            return;
        }

        $team = $this->itemForCheckout->researchTeam;
        if ($team) {
            $this->authorizedHandlers = $team->users()->get();
        } else {
            $this->authorizedHandlers = collect();
        }

        $this->reset('checkoutState');
        $this->resetErrorBag();
        $this->confirmingCheckout = true;
    }

    /**
     * [NEW] Validates and saves the new checkout request.
     */
    public function saveCheckoutRequest()
    {
        $this->validate([
            'checkoutState.recipient_name' => 'required|string|max:255',
            'checkoutState.handler_id' => 'required|exists:users,id',
            'checkoutState.expected_return_date' => 'required|string',
            'checkoutState.notes' => 'nullable|string|max:2000',
        ]);
// [NEW] Check if there is enough quantity to check out
        if ($this->itemForCheckout->quantity < 1) {
            $this->addError('checkoutState.handler_id', 'موجودی این نمونه برای خروج کافی نیست.');
            return;
        }

        try {
            $returnDate = Verta::parse($this->checkoutState['expected_return_date'])->toCarbon();
        } catch (\Exception $e) {
            $this->addError('checkoutState.expected_return_date', 'فرمت تاریخ نامعتبر است.');
            return;
        }

//        BiologicalSampleRequest::create([
//            'inventory_item_id' => $this->itemForCheckout->id,
//            'research_team_id' => $this->itemForCheckout->research_team_id,
//            'requester_id' => Auth::id(),
//            'handler_id' => $this->checkoutState['handler_id'],
//            'recipient_name' => $this->checkoutState['recipient_name'],
//            'checkout_date' => now(),
//            'expected_return_date' => $returnDate,
//            'status' => 'checked_out',
//            'notes' => $this->checkoutState['notes'] ?? null,
//        ]);
//
//        $this->confirmingCheckout = false;
//        session()->flash('message', 'درخواست خروج نمونه با موفقیت ثبت شد و به تاریخچه اضافه گردید.');
// [NEW] Use a database transaction to ensure data integrity
        DB::transaction(function () use ($returnDate) {
            // 1. Create the request record
            BiologicalSampleRequest::create([
                'inventory_item_id' => $this->itemForCheckout->id,
                'research_team_id' => $this->itemForCheckout->research_team_id,
                'requester_id' => Auth::id(),
                'handler_id' => $this->checkoutState['handler_id'],
                'recipient_name' => $this->checkoutState['recipient_name'],
                'checkout_date' => now(),
                'expected_return_date' => $returnDate,
                'status' => 'checked_out',
                'notes' => $this->checkoutState['notes'] ?? null,
            ]);

            // 2. Decrement the inventory item's quantity
            $this->itemForCheckout->decrement('quantity');
        });

        $this->confirmingCheckout = false;
        session()->flash('message', 'خروج نمونه با موفقیت ثبت و از موجودی کسر گردید.');
    }

    // [NEW] Methods for managing checkout requests, merged from SampleRequestManager
    public function viewRequest(int $requestId)
    {
        $this->viewingRequest = BiologicalSampleRequest::with(['inventoryItem', 'requester', 'handler', 'researchTeam'])->find($requestId);
        $this->confirmingViewRequest = true;
    }

    public function confirmReturn(int $requestId)
    {
        $this->returningRequest = BiologicalSampleRequest::find($requestId);
        $this->confirmingReturnRequest = true;
    }

    public function markAsReturned()
    {
        if ($this->returningRequest) {
            DB::transaction(function () {
                $this->returningRequest->update([
                    'status' => 'returned',
                    'actual_return_date' => now(),
                ]);
                $this->returningRequest->inventoryItem->increment('quantity');
            });
            session()->flash('message', 'بازگشت نمونه با موفقیت ثبت و به موجودی اضافه گردید.');
        }
        $this->confirmingReturnRequest = false;
        $this->resetPage('requestsPage'); // Reset pagination for the requests tab
    }

    private function getFilteredItemsQuery()
    {
        $authorizedTeamIds = $this->getAuthorizedTeamIds();

        $query = InventoryItem::where('is_biological', true)
            ->whereIn('research_team_id', $authorizedTeamIds)
            ->with('researchTeam', 'location');

        if ($this->searchQuery) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%'.$this->searchQuery.'%')
                    ->orWhere('sku', 'like', '%'.$this->searchQuery.'%')
                    ->orWhere('cell_type', 'like', '%'.$this->searchQuery.'%');
            });
        }

        if ($this->searchTeamId) {
            // Ensure the user is authorized to view the selected team
            if($authorizedTeamIds->contains($this->searchTeamId)){
                $query->where('research_team_id', $this->searchTeamId);
            }
        }

        return $query;
    }

    private function getFilteredRequestsQuery()
    {
        $authorizedTeamIds = $this->getAuthorizedTeamIds();

        $requestsQuery = BiologicalSampleRequest::whereIn('research_team_id', $authorizedTeamIds)
            ->with(['inventoryItem', 'requester', 'handler']);

        if ($this->statusFilter !== 'all') {
            $requestsQuery->where('status', $this->statusFilter);
        }

        return $requestsQuery;
    }

    public function render()
    {
//        $user = Auth::user();
////        $userTeamIds = $user->researchTeams()->pluck('research_teams.id');
////        $researchTeams = ResearchTeam::whereIn('id', $userTeamIds)->get();
//        $userTeamIds = $this->researchTeams->pluck('id');
//        $query = InventoryItem::where('is_biological', true)
//            ->whereIn('research_team_id', $userTeamIds)
//            ->with('researchTeam', 'location');
//
//        $query = InventoryItem::where('is_biological', true)
//            ->whereIn('research_team_id', $userTeamIds)
//            ->with('researchTeam', 'location');
//
//        if ($this->searchQuery) {
//            $query->where(function ($q) {
//                $q->where('name', 'like', '%'.$this->searchQuery.'%')
//                    ->orWhere('sku', 'like', '%'.$this->searchQuery.'%')
//                    ->orWhere('cell_type', 'like', '%'.$this->searchQuery.'%');
//            });
//        }
//
//        if ($this->searchTeamId) {
//            $query->where('research_team_id', $this->searchTeamId);
//        }
//        // Query for checkout requests (Requests History Tab)
//        $requestsQuery = BiologicalSampleRequest::whereIn('research_team_id', $userTeamIds)
//            ->with(['inventoryItem', 'requester', 'handler']);
//
//        if ($this->statusFilter !== 'all') {
//            $requestsQuery->where('status', $this->statusFilter);
//        }
//
//        return view('livewire.bio-inventory-manager', [
//            'items' => $query->latest()->paginate(10, ['*'], 'itemsPage'),
//            'requests' => $requestsQuery->latest('checkout_date')->paginate(15, ['*'], 'requestsPage'),
////            'researchTeams' => $researchTeams,
//            'researchTeams' => $this->researchTeams,
//        ])->layout('layouts.app');
//    }
        return view('livewire.bio-inventory-manager', [
            'items' => $this->getFilteredItemsQuery()->latest()->paginate(10, ['*'], 'itemsPage'),
            'requests' => $this->getFilteredRequestsQuery()->latest('checkout_date')->paginate(15, ['*'], 'requestsPage'),
            'researchTeams' => $this->researchTeams,
        ])->layout('layouts.app');
    }
}
