<?php

namespace App\Livewire;

use App\Models\StorageLocation;
use Livewire\Component;
use App\Models\InventoryItem;
use Carbon\Carbon;


class BoxViewer extends Component
{
    public StorageLocation $location;
    public $itemsInGrid = [];
    public ?InventoryItem $selectedItem = null;
    public string $statusFilter = 'all'; // 'all', 'critical', 'warning', 'near_expiry', 'ok', 'empty'

    // Properties for Item Placement Modal
    public $confirmingItemPlacement = false;
    public $positionToPlace = null;
    public string $itemSearch = '';
    public $itemSearchResults = [];
    public $itemToPlaceId = null;
    // Properties for Item Removal
    public ?InventoryItem $itemToRemove = null;
    public $confirmingItemRemoval = false;

    // Properties for Item Movement
    public ?InventoryItem $itemToMove = null;
    public $confirmingItemMove = false;
    public $newPosition = '';
    public $availablePositions = [];

    // Properties for Highlighting
    public string $highlightSearch = '';
    public ?string $highlightedPosition = null;
    public bool $isBiologicalContext = false;



    public function mount(StorageLocation $location)
    {
        if (!$location->rows || !$location->columns) {
            abort(404, 'این مکان یک محفظه شبکه‌ای نیست.');
        }
        $this->location = $location;
        // [NEW] Check if the box is inside a nitrogen tank
        if ($this->location->isInside('tank')) {
            $this->isBiologicalContext = true;
        }
        $this->loadItems();
    }

    public function loadItems()
    {
//        $this->location->load('inventoryItems');
//
//        // --- منطق جدید: محاسبه وضعیت موجودی برای هر آیتم ---
//        $this->itemsInGrid = $this->location->inventoryItems->mapWithKeys(function ($item) {
//            $status = 'ok'; // وضعیت پیش‌فرض
//            if ($item->minimum_stock_level) {
//                if ($item->quantity <= $item->minimum_stock_level) {
//                    $status = 'critical'; // وضعیت بحرانی (قرمز)
//                } elseif ($item->quantity <= $item->minimum_stock_level * 1.2) { // آستانه هشدار: ۱۲۰٪
//                    $status = 'warning'; // وضعیت هشدار (صورتی/زرد)
//                }
//            }
//            // ما وضعیت محاسبه شده را به آبجکت آیتم اضافه می‌کنیم
//            $item->stock_status = $status;
//            return [$item->position_in_location => $item];
//        });
        // Eager load relationships for better performance
        $this->location->load('inventoryItems.researchTeam');
        $this->itemsInGrid = $this->location->inventoryItems->keyBy('position_in_location');
    }

    /**
     * Live search for unplaced items in the same team's inventory.
     */
    public function updatedItemSearch()
    {
        if (strlen($this->itemSearch) < 2) {
            $this->itemSearchResults = [];
            return;
        }

//        $this->itemSearchResults = InventoryItem::where('research_team_id', $this->location->research_team_id)
//            ->whereNull('position_in_location') // Only items not already in a box
//            ->where(function ($query) {
//                $query->where('name', 'like', '%' . $this->itemSearch . '%')
//                    ->orWhere('sku', 'like', '%' . $this->itemSearch . '%');
//            })
//            ->take(5)
//            ->get();
        $query = InventoryItem::where('research_team_id', $this->location->research_team_id)
            ->whereNull('position_in_location') // Only items not already in a box
            ->where(function ($q) {
                $q->where('name', 'like', '%' . $this->itemSearch . '%')
                    ->orWhere('sku', 'like', '%' . $this->itemSearch . '%');
            });

        // ** KEY CHANGE: If inside a tank, only search for biological items **
        if ($this->isBiologicalContext) {
            $query->where('is_biological', true);
        } else {
            $query->where('is_biological', false);
        }

        $this->itemSearchResults = $query->take(5)->get();
    }

    /**
     * Select an item from the search results.
     */
    public function selectItemToPlace(InventoryItem $item)
    {
        $this->itemToPlaceId = $item->id;
        $this->itemSearch = $item->name . ' (' . $item->sku . ')';
        $this->itemSearchResults = [];
    }

    /**
     * Show the modal for placing an item in a specific position.
     */
    public function confirmItemPlacement($position)
    {
        $this->resetErrorBag();
        $this->reset(['itemSearch', 'itemSearchResults', 'itemToPlaceId']);
        $this->positionToPlace = $position;
        $this->confirmingItemPlacement = true;
    }

    /**
     * Assign the selected item to the chosen position in the box.
     */
    public function placeItem()
    {
        $this->validate([
            'itemToPlaceId' => 'required|exists:inventory_items,id',
            'positionToPlace' => 'required|string',
        ]);

        $item = InventoryItem::find($this->itemToPlaceId);

        // Update the item's location and position
        $item->update([
            'location_id' => $this->location->id,
            'position_in_location' => $this->positionToPlace,
            'storage_location_id' => $this->location->id,
        ]);

        $this->confirmingItemPlacement = false;
        $this->loadItems(); // Refresh the grid view
        session()->flash('message', 'آیتم با موفقیت در موقعیت ' . $this->positionToPlace . ' قرار گرفت.');
    }

    /**
     * Show the confirmation modal for removing an item from the box.
     */
    public function confirmItemRemoval(InventoryItem $item)
    {
        $this->itemToRemove = $item;
        $this->confirmingItemRemoval = true;
    }

    /**
     * Remove the item from the box (un-place it).
     */
    public function removeItem()
    {
        if ($this->itemToRemove) {
            $this->itemToRemove->update([
                'storage_location_id' => null,
                'location_id' => null,
                'position_in_location' => null,
            ]);
            $this->itemToRemove = null;
            $this->confirmingItemRemoval = false;
            $this->loadItems();
            session()->flash('message', 'آیتم با موفقیت از محفظه خارج شد.');
        }
    }

    /**
     * Show the modal for moving an item to a new position.
     */
    public function confirmItemMove(InventoryItem $item)
    {
        $this->resetErrorBag();
        $this->itemToMove = $item;
        $this->newPosition = '';

        // Generate a list of available empty positions
        $this->availablePositions = [];
        $occupiedPositions = $this->itemsInGrid->keys();

        for ($row = 0; $row < $this->location->rows; $row++) {
            for ($col = 1; $col <= $this->location->columns; $col++) {
                $position = chr(65 + $row) . $col;
                if (!$occupiedPositions->contains($position)) {
                    $this->availablePositions[] = $position;
                }
            }
        }

        $this->confirmingItemMove = true;
    }

    /**
     * Move the item to the new selected position.
     */
    public function moveItem()
    {
        if ($this->itemToMove) {
            $this->validate([
                'newPosition' => 'required|in:' . implode(',', $this->availablePositions),
            ]);

            $this->itemToMove->update([
                'position_in_location' => $this->newPosition,
            ]);

            $this->itemToMove = null;
            $this->confirmingItemMove = false;
            $this->loadItems();
            session()->flash('message', 'آیتم با موفقیت به موقعیت جدید منتقل شد.');
        }
    }

    /**
     * Live search for highlighting an item in the current box.
     */
    public function updatedHighlightSearch()
    {
        if (strlen($this->highlightSearch) < 2) {
            $this->highlightedPosition = null;
            return;
        }

        $foundItem = $this->location->inventoryItems()
            ->where(function ($query) {
                $query->where('name', 'like', '%' . $this->highlightSearch . '%')
                    ->orWhere('sku', 'like', '%' . $this->highlightSearch . '%');
            })
            ->first();

        $this->highlightedPosition = $foundItem?->position_in_location;
    }
    /**
     * [NEW] Selects an item to be displayed in the details pane.
     * Clears selection if the same item is clicked again.
     *
     * @param int|null $itemId
     */
    public function selectItem(?int $itemId)
    {
        if ($itemId === null) {
            $this->selectedItem = null;
            return;
        }

        // If the same item is clicked, deselect it
        if ($this->selectedItem && $this->selectedItem->id === $itemId) {
            $this->selectedItem = null;
        } else {
            $this->selectedItem = InventoryItem::with('researchTeam')->find($itemId);
        }
    }

    /**
     * [NEW] A helper function to determine the CSS class for each cell's background.
     * This creates the "Heatmap" effect.
     *
     * @param InventoryItem|null $item
     * @return string
     */
    public function getCellColorClass(?InventoryItem $item): string
    {
        if (!$item) {
            return 'bg-gray-100 dark:bg-gray-700/50'; // Empty
        }
//
//        // 1. Critical: Expired or quantity is at/below minimum
//        if ($item->expiration_date && Carbon::parse($item->expiration_date)->isPast()) {
//            return 'bg-red-200 dark:bg-red-800/50'; // Expired
//        }
//        if ($item->minimum_stock_level && $item->quantity <= $item->minimum_stock_level) {
//            return 'bg-red-200 dark:bg-red-800/50'; // Critical stock
//        }
//
//        // 2. Near Expiry: Expires in the next 3 months
//        if ($item->expiration_date && Carbon::parse($item->expiration_date)->isBefore(now()->addMonths(3))) {
//            return 'bg-orange-200 dark:bg-orange-800/50'; // Near expiry
//        }
//
//        // 3. Warning: Quantity is approaching minimum (e.g., 120%)
//        if ($item->minimum_stock_level && $item->quantity <= ($item->minimum_stock_level * 1.2)) {
//            return 'bg-yellow-200 dark:bg-yellow-800/50'; // Warning stock
//        }
//
//        // 4. OK: Item is in good standing
//        return 'bg-green-200 dark:bg-green-800/50'; // OK

        return match ($this->getItemStatus($item)) {
            'critical' => 'bg-red-200 dark:bg-red-800/50',
            'near_expiry' => 'bg-orange-200 dark:bg-orange-800/50',
            'warning' => 'bg-yellow-200 dark:bg-yellow-800/50',
            'ok' => 'bg-green-200 dark:bg-green-800/50',
            default => 'bg-gray-200 dark:bg-gray-700',
        };
    }

    /**
     * [NEW] Sets the active status filter.
     *
     * @param string $status
     */
    public function setStatusFilter(string $status)
    {
        $this->statusFilter = $status;
    }

    /**
     * [NEW] A helper function to check if a cell should be visible based on current filters.
     *
     * @param InventoryItem|null $item
     * @return bool
     */
    public function shouldShowCell(?InventoryItem $item): bool
    {
        // 1. Handle text search (highlighting is done in blade, this just shows/hides)
        if (strlen($this->highlightSearch) >= 2) {
            if (!$item) return false; // Hide empty cells during text search
            return str_contains(strtolower($item->name), strtolower($this->highlightSearch)) ||
                str_contains(strtolower($item->sku), strtolower($this->highlightSearch));
        }

        // 2. Handle status filters
        if ($this->statusFilter === 'all') {
            return true;
        }
        if ($this->statusFilter === 'empty') {
            return $item === null;
        }
        if ($item === null) {
            return false; // Hide empty cells for all other status filters
        }

        // Get the status string from the color class method
        $status = $this->getItemStatus($item);
        return $status === $this->statusFilter;
    }
    /**
     * [REFACTORED] This method now only returns a status string.
     *
     * @param InventoryItem $item
     * @return string
     */
    public function getItemStatus(InventoryItem $item): string
    {
        if ($item->expiration_date && Carbon::parse($item->expiration_date)->isPast()) return 'critical';
        if ($item->minimum_stock_level && $item->quantity <= $item->minimum_stock_level) return 'critical';
        if ($item->expiration_date && Carbon::parse($item->expiration_date)->isBefore(now()->addMonths(3))) return 'near_expiry';
        if ($item->minimum_stock_level && $item->quantity <= ($item->minimum_stock_level * 1.2)) return 'warning';
        return 'ok';
    }


    public function render()
    {
        return view('livewire.box-viewer')->layout('layouts.app');
    }
}
