<?php

namespace App\Livewire;

use App\Models\StorageLocation;
use Illuminate\Support\Facades\Auth;

// We are still extending BoxViewer to reuse modal logic, search, etc.
class CaneViewer extends BoxViewer
{
    /**
     * [FIXED] Override the mount method to manually fetch the location
     * and perform authorization checks.
     *
     * @param int $locationId The ID from the URL.
     */
    public function mount($locationId)
    {
        // Manually find the location or fail with a 404 if it doesn't exist at all.
        $location = StorageLocation::findOrFail($locationId);

        // Ensure the location is a cane.
        if ($location->location_type !== 'cane') {
            abort(403, 'این مکان یک کرایوکین نیست.');
        }

        // Get the teams the current user belongs to.
        $userTeamIds = Auth::user()->researchTeams()->pluck('research_teams.id')->all();

        // [SECURITY FIX] Check if the location's team is in the user's list of teams.
        // If not, throw a 403 Forbidden error, which is more accurate than a 404.
        if (!in_array($location->research_team_id, $userTeamIds)) {
            abort(403, 'شما اجازه دسترسی به این مکان را ندارید.');
        }

        // If all checks pass, set the location and build the grid.
        $this->location = $location;
        $this->buildGrid();
    }

    /**
     * Override the component view file to point to the cane viewer blade.
     *
     * @return string
     */
    public function getView(): string
    {
        return 'livewire.cane-viewer';
    }

    /**
     * [CORRECTED] Override the grid creation logic for a vertical cane.
     * This now correctly uses the 'inventoryItems' relationship.
     */
    protected function buildGrid()
    {
        $this->itemsInGrid = collect();
        if ($this->location) {
            // Eager load the items using the correct relationship name
            $this->location->load('inventoryItems');

            if ($this->location->inventoryItems) {
                // Key the collection by the 'position_in_location' attribute of the item.
                $this->itemsInGrid = $this->location->inventoryItems->keyBy('position_in_location');
            }
        }
    }

    /**
     * [FIXED] Override the available positions logic for a cane.
     * This now checks against the keys of the correctly built grid.
     */
    protected function calculateAvailablePositions()
    {
        $this->availablePositions = [];
        // Use the 'size' attribute from the database, which you added with the migration.
        $capacity = $this->location->size ?? 10; // Default to 10 if size is not set
        $occupiedPositions = $this->itemsInGrid->keys()->all();

        for ($i = 1; $i <= $capacity; $i++) {
            // Positions are strings from the database, so we do a loose comparison
            if (!in_array($i, $occupiedPositions)) {
                $this->availablePositions[] = (string)$i;
            }
        }
    }
}
