<?php

namespace App\Livewire;

use App\Models\ChatRoom;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class ChatManager extends Component
{
    public $chatRooms;
    public ?ChatRoom $activeChatRoom = null;
    public $chatRoomId;

    // [NEW] Properties for New Direct Chat
    public bool $confirmingNewChat = false;
    public string $userSearch = '';
    public $userSearchResults = [];
    protected $listeners = ['messagesRead'];


    public function mount($chatRoomId = null)
    {
        $this->loadChatRooms();

        if ($chatRoomId) {
            $this->activeChatRoom = ChatRoom::findOrFail($chatRoomId);
        } elseif ($this->chatRooms->isNotEmpty()) {
            $this->activeChatRoom = $this->chatRooms->first();
        }
    }

    /**
     * [NEW] A dedicated method to load and correctly format chat rooms.
     */
    /**
     * [CORRECTED] واکشی اتاق‌های گفتگو به همراه تعداد پیام‌های خوانده نشده (بدون احتساب پیام‌های خود کاربر).
     */
    public function loadChatRooms()
    {
        $user = Auth::user();
        $this->chatRooms = $user->chatRooms()
            ->with(['project', 'users'])
            ->withCount(['messages as unread_messages_count' => function ($query) use ($user) {
                $query->where('user_id', '!=', $user->id) // [FIX] پیام‌های خود کاربر را شمارش نکن
                ->whereDoesntHave('readByUsers', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                });
            }])
            ->get()
            ->values();
    }
    /**
     * [NEW] این متد اکنون لیست گفتگوها را رفرش می‌کند.
     */
    public function messagesRead()
    {
        $this->loadChatRooms();
    }

    /**
     * Show the modal for creating a new direct chat.
     */
    public function confirmNewChat()
    {
        $this->resetErrorBag();
        $this->reset(['userSearch', 'userSearchResults']);
        $this->confirmingNewChat = true;
    }

    /**
     * Live search for users to start a chat with.
     */
    public function updatedUserSearch($value)
    {
        if (strlen($this->userSearch) < 2) {
            $this->userSearchResults = [];
            return;
        }

        $this->userSearchResults = User::where('id', '!=', Auth::id())
            ->where(function ($query) {
                $query->where('username', 'like', '%' . $this->userSearch . '%')
                    ->orWhere('first_name', 'like', '%' . $this->userSearch . '%')
                    ->orWhere('last_name', 'like', '%' . $this->userSearch . '%');
            })
            ->take(5)
            ->get()
            ->values();
    }

    /**
     * Find an existing direct chat or create a new one.
     */
    /**
     * [REWRITTEN] Find an existing direct chat or create a new one using attach().
     */
    public function startDirectChat($userId)
    {
        $user = User::findOrFail($userId);
        $currentUser = Auth::user();

        $chatRoom = $currentUser->chatRooms()
            ->where('type', 'direct')
            ->with('users')
            ->get()
            ->first(function ($room) use ($user) {
                return $room->users->count() === 2 && $room->users->contains($user->id);
            });

        if (!$chatRoom) {
            $chatRoom = ChatRoom::create([
                'type' => 'direct',
                'name' => 'گفتگو با ' . $user->name,
            ]);
            // [FIX] Using attach() instead of sync() for new records.
            $chatRoom->users()->attach([$currentUser->id, $user->id]);
        }

        $this->confirmingNewChat = false;
        $this->selectChatRoom($chatRoom->id);
    }


    public function selectChatRoom($chatRoomId)
    {
        $this->activeChatRoom = ChatRoom::findOrFail($chatRoomId);
        $this->loadChatRooms(); // [UPDATED] Use the dedicated method to reload rooms

    }

    public function render()
    {
        // استفاده از layout اختصاصی که ساختیم
        return view('livewire.chat-manager')
            ->layout('layouts.chat');
    }
}
