<?php

namespace App\Livewire;

use App\Models\ChatRoom;
use App\Notifications\NewChatMessageNotification; // [NEW]
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification; // [NEW]
use Livewire\Component;

class ChatMessageManager extends Component
{
    use AuthorizesRequests;

    public ChatRoom $chatRoom;
    public $messages;
    public string $newMessage = '';

    public function mount(ChatRoom $chatRoom)
    {
        $this->chatRoom = $chatRoom;
        $this->markMessagesAsRead();
        $this->loadMessages();
    }
    /**
     * [NEW] Mark all unread messages in this room as read for the current user.
     */
    public function markMessagesAsRead()
    {
        $user = Auth::user();
        $unreadMessageIds = $this->chatRoom->messages()
            ->whereDoesntHave('readByUsers', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            })
            ->pluck('id');

        if ($unreadMessageIds->isNotEmpty()) {
            $user->readChatMessages()->attach($unreadMessageIds, ['read_at' => now()]);
            // Notify the parent component to refresh the unread counts
            $this->dispatch('messagesRead');
        }
    }

    /**
     * [CORRECTED] Load messages without reversing the collection in the backend.
     * The view will handle the display order using flex-col-reverse.
     */
    public function loadMessages()
    {
        $this->messages = $this->chatRoom->messages()
            ->with('user')
            ->latest()
            ->take(50)
            ->get()
            ->reverse()
            ->values() // Re-index the collection keys from 0
            ->all();   // Convert the collection to a plain PHP array
    }

    /**
     * Send a new message to the chat room.
     */
    /**
     * [UPDATED] Send a new message and notify other room members.
     */
    public function sendMessage()
    {
        $this->validate(['newMessage' => 'required|string']);

        $sender = Auth::user();

        $message = $this->chatRoom->messages()->create([
            'user_id' => $sender->id,
            'body' => $this->newMessage,
        ]);

        // [NEW] Send notification to other users in the room
        $recipients = $this->chatRoom->users()
            ->where('user_id', '!=', $sender->id)
            ->get();

        if ($recipients->isNotEmpty()) {
            Notification::send($recipients, new NewChatMessageNotification($message, $this->chatRoom, $sender));
        }

        $this->reset('newMessage');
        $this->loadMessages();
    }

    public function render()
    {
        return view('livewire.chat-message-manager');
    }
}
