<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\User;
use Livewire\Component;
use Illuminate\Support\Collection;
use Carbon\Carbon;

class CollaboratorProjectReport extends Component
{
    public Project $project;
    public $collaborators;
    public $selectedCollaboratorId;

    public function mount(Project $project)
    {
        $this->project = $project;
        $this->collaborators = $this->project->collaborators->push($this->project->owner)->unique('id');

        if ($this->collaborators->isNotEmpty()) {
            $this->selectedCollaboratorId = $this->collaborators->first()->id;
        }
    }

    public function render()
    {
        $selectedCollaborator = null;
        $tasks = collect();
        $chartData = [];
        $completionPercentage = 0; // [NEW]

        if ($this->selectedCollaboratorId) {
            $selectedCollaborator = User::find($this->selectedCollaboratorId);

            if ($selectedCollaborator) {
                $tasks = $selectedCollaborator->assignedTasks()
                    ->where('project_id', $this->project->id)
                    ->get();

                $totalTasks = $tasks->count();
                if ($totalTasks > 0) {
                    $completedTasks = $tasks->where('status', 'انجام شده')->count();
                    $completionPercentage = round(($completedTasks / $totalTasks) * 100); // [NEW]
                }

                // ... (منطق آماده‌سازی داده‌های نمودار بدون تغییر)
                $tasksByMonth = $tasks->whereNotNull('due_date')->groupBy(function ($task) {
                    return $task->due_date->format('Y-m');
                });
                $sortedTasksByMonth = $tasksByMonth->sortKeys();
                $labels = $sortedTasksByMonth->keys()->map(function ($dateKey) {
                    return verta(Carbon::createFromFormat('Y-m', $dateKey))->format('F Y');
                })->all();
                $dataDone = [];
                $dataInProgress = [];
                $dataTodo = [];
                foreach ($sortedTasksByMonth as $monthGroup) {
                    $dataDone[] = $monthGroup->where('status', 'انجام شده')->count();
                    $dataInProgress[] = $monthGroup->where('status', 'در حال انجام')->count();
                    $dataTodo[] = $monthGroup->where('status', 'برای انجام')->count();
                }
                $datasets = [
                    ['label' => 'انجام شده', 'data' => $dataDone, 'backgroundColor' => '#22c55e'],
                    ['label' => 'در حال انجام', 'data' => $dataInProgress, 'backgroundColor' => '#f59e0b'],
                    ['label' => 'برای انجام', 'data' => $dataTodo, 'backgroundColor' => '#3b82f6'],
                ];
                $chartData = ['labels' => $labels, 'datasets' => $datasets];
            }
        }

        $this->dispatch('update-collaborator-chart', data: $chartData);

        return view('livewire.collaborator-project-report', [
            'selectedCollaborator' => $selectedCollaborator,
            'tasks' => $tasks,
            'chartData' => $chartData,
            'completionPercentage' => $completionPercentage, // [NEW]
        ]);
    }
}
