<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Livewire\Component;

class CollaboratorTaskReport extends Component
{
    public Project $project;
    public $collaborators;
    public $selectedCollaboratorId;

    // Properties for the selected collaborator's stats
    public ?User $selectedCollaborator = null;
    public $tasks;
    public array $chartData = [];

    public function mount(Project $project)
    {
        $this->project = $project->load('collaborators', 'owner');
        $this->collaborators = $this->project->collaborators->push($this->project->owner)->unique('id');

        if ($this->collaborators->isNotEmpty()) {
            $this->selectedCollaboratorId = $this->collaborators->first()->id;
            $this->loadCollaboratorData();
        }
    }

    public function updatedSelectedCollaboratorId()
    {
        $this->loadCollaboratorData();
    }

    public function loadCollaboratorData()
    {
        if (!$this->selectedCollaboratorId) {
            $this->selectedCollaborator = null;
            return;
        }

        $this->selectedCollaborator = User::find($this->selectedCollaboratorId);

        if ($this->selectedCollaborator) {
            $this->tasks = $this->selectedCollaborator->assignedTasks()
                ->where('project_id', $this->project->id)
                ->get();

            // Prepare data for the line chart (tasks completed over time)
            $completedTasksOverTime = $this->tasks
                ->where('status', 'انجام شده')
                ->whereNotNull('completed_at')
                ->groupBy(function ($task) {
                    return $task->completed_at->format('Y-m'); // Group by year-month
                })
                ->map(function ($group) {
                    return $group->count();
                });

            $this->chartData = [
                'labels' => $completedTasksOverTime->keys()->all(),
                'data'   => $completedTasksOverTime->values()->all(),
            ];
        }
    }

    public function render()
    {
        return view('livewire.collaborator-task-report');
    }
}
