<?php

namespace App\Livewire;

use App\Models\InventoryItem;
use App\Models\Project;
use App\Models\LabNotebookEntry;
use App\Models\InventoryConsumptionRequest;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Livewire\WithFileUploads; // [NEW]
use Hekmatinasser\Verta\Verta; // [NEW]
use App\Models\LabNotebookTemplate;


class CreateLabNotebookEntry extends Component
{
    use AuthorizesRequests;
    use WithFileUploads; // [NEW]

    public Project $project;

    // Properties for the main form
    public $title;
    public $entry_date;
    public $body = '';
    public $generatedSku = '';
    // [NEW] Properties for Attachments
    public $newAttachments = [];
    public $attachments = [];
    // [NEW] Properties for Templates
    public $templates = [];
    public $selectedTemplateId = '';

    public array $attachmentCaptions = []; // [NEW]
    public bool $confirmingAttachments = false; // [NEW]

    // Properties for Materials
    public string $materialSearch = '';
    public $materialSearchResults = [];
    public array $usedMaterials = [];

    // Properties for Stock Error Modal
    public $confirmingStockError = false;
    public $stockErrorMessage = '';
    public bool $hasHardErrors = false;
    // [NEW] Properties for Entry Selector Modal
    public bool $showEntrySelectorModal = false;
    public $projectEntries = [];
    // [NEW] Properties for Content Preview
    public bool $showContentPreviewModal = false;
    public string $previewHtml = '';

    // [NEW] Properties for Linked Entry Preview
    public ?LabNotebookEntry $previewingEntry = null;
    public bool $showPreviewModal = false;

    protected $listeners = ['editorContentUpdated'];

    protected function rules()
    {
        return [
            'title' => 'required|string|max:255',
            'entry_date' => 'required|date', // [KEY CHANGE] We now validate a date object
            'body' => 'required|string|min:10',
            'usedMaterials.*.quantity_used' => 'required|numeric|min:0.0001',
            'attachments.*' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240', // 10MB Max per file
            'attachmentCaptions.*' => 'nullable|string|max:255', // [NEW]
        ];
    }
    public function editorContentUpdated($content)
    {
        $this->body = $content;
    }

    public function mount(Project $project)
    {
        $this->authorize('create', [LabNotebookEntry::class, $project]);
        $this->project = $project->load('budgetItems', 'researchTeam.labNotebookTemplates');
        $this->generateSku();
        // [NEW] Load available templates for the project's team
        if ($this->project->researchTeam) {
            $this->templates = $this->project->researchTeam->labNotebookTemplates;
        }
    }

    /**
     * [NEW] Applies the selected template's content to the form.
     */
    public function applyTemplate()
    {
        if (empty($this->selectedTemplateId)) {
            return;
        }

        $template = LabNotebookTemplate::find($this->selectedTemplateId);

        if ($template) {
            $this->title = $template->title;
            $this->body = $template->body;

            // Dispatch event to update the Jodit editor in the view
            $this->dispatch('update-editor-content', content: $this->body);
        }
    }
    /**
     * [NEW] This hook is called when files are selected/dropped.
     * It opens the caption editing modal.
     */
    public function updatedNewAttachments()
    {
        $this->validate(['newAttachments.*' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240']);

        // فایل‌های جدید را با فایل‌های قبلی ادغام می‌کنیم
        $this->attachments = array_merge($this->attachments, $this->newAttachments);

        // کپشن‌ها را برای فایل‌های جدید مقداردهی اولیه می‌کنیم
        $this->attachmentCaptions = array_pad($this->attachmentCaptions, count($this->attachments), '');

        $this->confirmingAttachments = true;
    }

    /**
     * [NEW] Re-open the attachments modal for editing.
     */
    public function editAttachments()
    {
        $this->confirmingAttachments = true;
    }
    /**
     * [NEW] Remove a temporary attachment from the list.
     */
    public function removeAttachment($index)
    {
        $attachments = $this->attachments;
        $captions = $this->attachmentCaptions;

        if (isset($attachments[$index])) {
            unset($attachments[$index]);
        }
        if (isset($captions[$index])) {
            unset($captions[$index]);
        }

        $this->attachments = array_values($attachments);
        $this->attachmentCaptions = array_values($captions);
    }

    /**
     * [REWRITTEN] Live search for inventory items only within the project's budget.
     */
    public function updatedMaterialSearch($value)
    {
        if (strlen($this->materialSearch) < 2) {
            $this->materialSearchResults = [];
            return;
        }

        $this->materialSearchResults = $this->project->budgetItems()
            ->where('name', 'like', '%' . $this->materialSearch . '%')
            ->take(5)
            ->get();
    }

    /**
     * Add a selected material to the temporary list.
     */
    public function addMaterial(InventoryItem $item)
    {
        if (!collect($this->usedMaterials)->contains('id', $item->id)) {
            $this->usedMaterials[] = [
                'id' => $item->id,
                'name' => $item->name,
                'sku' => $item->sku,
                'unit' => $item->unit,
                'quantity_used' => ''
            ];
        }
        $this->materialSearch = '';
        $this->materialSearchResults = [];
    }

    /**
     * Remove a material from the temporary list.
     */
    public function removeMaterial($itemId)
    {
        $this->usedMaterials = array_filter($this->usedMaterials, fn($material) => $material['id'] != $itemId);
    }

    public function generateSku()
    {
        $projectSkuPrefix = $this->project->sku ?? 'LNOTE';
        do {
            $randomPart = strtoupper(Str::random(10));
            $sku = $projectSkuPrefix . '-Note-' . $randomPart;
        } while (LabNotebookEntry::where('sku', $sku)->exists());
        $this->generatedSku = $sku;
    }

    /**
     * [REWRITTEN] Step 1: Validate inputs and show modals if necessary.
     */
    public function save()
    {
        $this->authorize('create', [LabNotebookEntry::class, $this->project]);
        // [KEY CHANGE] تبدیل تاریخ شمسی به میلادی قبل از هر کاری
        try {
            if (!empty($this->entry_date)) {
                $this->entry_date = Verta::parse($this->entry_date)->toCarbon();
            }
        } catch (\Exception $e) {
            $this->addError('entry_date', 'فرمت تاریخ نامعتبر است.');
            return;
        }

        // حالا داده‌های تبدیل شده را اعتبارسنجی می‌کنیم
        $validatedData = $this->validate();

         $this->validate([
            'title' => 'required|string|max:255',
            'entry_date' => 'required|date',
            'body' => 'required|string|min:10',
            'usedMaterials.*.quantity_used' => 'required|numeric|min:0.0001'
        ]);

        $hardErrors = [];
        $softWarnings = [];
        $allowedItems = $this->project->budgetItems->keyBy('id');

        foreach ($this->usedMaterials as $material) {
            $inventoryItem = InventoryItem::find($material['id']);
            if (!$inventoryItem) continue;

            // این اعتبارسنجی دیگر لازم نیست چون جستجو محدود به بودجه است، اما برای امنیت بیشتر باقی می‌ماند
            if (!$allowedItems->has($material['id'])) {
                $hardErrors[] = "ماده '{$material['name']}' در لیست مواد مجاز این طرح قرار ندارد.";
                continue;
            }

            $quantityToUse = (float)$material['quantity_used'];
            $budgetItem = $allowedItems->get($material['id']);
            $remainingBudget = $budgetItem->pivot->max_allowed_quantity - $budgetItem->pivot->total_consumed_quantity;

            // بررسی خطای سخت: آیا سقف بودجه رعایت شده است؟
            if ($quantityToUse > $remainingBudget) {
                $hardErrors[] = "مصرف ماده '{$material['name']}' از سقف مجاز بودجه (باقیمانده: {$remainingBudget} {$inventoryItem->unit}) فراتر می‌رود.";
            }

            // بررسی هشدار نرم: آیا موجودی انبار کافی است؟
            if ($inventoryItem->quantity < $quantityToUse) {
                $softWarnings[] = "موجودی انبار برای '{$material['name']}' (موجودی فعلی: {$inventoryItem->quantity} {$inventoryItem->unit}) کافی نیست.";
            }
        }

        if (!empty($hardErrors)) {
            $this->hasHardErrors = true;
            $this->stockErrorMessage = implode('<br>', $hardErrors);
            $this->confirmingStockError = true;
            return;
        }

        if (!empty($softWarnings)) {
            $this->hasHardErrors = false;
            $this->stockErrorMessage = implode('<br>', $softWarnings) . "<br><br>درخواست شما همچنان برای بررسی ارسال خواهد شد. آیا مایل به ادامه هستید؟";
            $this->confirmingStockError = true;
            return;
        }

        return  $this->proceedWithSave($validatedData);
    }

    /**
     * [NEW] Step 2: Proceed with saving the entry and creating requests.
     */
    public function proceedWithSave(array $validatedData)
    {
        $this->confirmingStockError = false;

        DB::beginTransaction();
        try {
            $newEntry = $this->project->labNotebookEntries()->create([
                'title' => $validatedData['title'],
                'sku' => $this->generatedSku,
                'entry_date' => $validatedData['entry_date'],
                'body' => $validatedData['body'],
                'user_id' => Auth::id(),
            ]);

            // Handle file uploads
            foreach ($this->attachments as $index => $attachmentFile) {
                $path = $attachmentFile->store('labnotes/' . $newEntry->sku, 'public');
                $newEntry->attachments()->create([
                    'user_id' => Auth::id(),
                    'file_path' => $path,
                    'file_name' => $attachmentFile->getClientOriginalName(),
                    'file_size' => $attachmentFile->getSize(),
                    'mime_type' => $attachmentFile->getMimeType(),
                    'caption' => $this->attachmentCaptions[$index] ?? null,
                ]);
            }

            // Handle material consumption requests
            foreach ($this->usedMaterials as $material) {
                InventoryConsumptionRequest::create([
                    'project_id' => $this->project->id,
                    'lab_notebook_entry_id' => $newEntry->id,
                    'requester_id' => Auth::id(),
                    'inventory_item_id' => $material['id'],
                    'type' => 'consumption',
                    'quantity' => $material['quantity_used'],
                    'status' => 'pending',
                ]);
                $newEntry->materials()->attach($material['id'], ['quantity_used' => $material['quantity_used']]);
            }

            DB::commit();

            session()->flash('message', 'یادداشت جدید با موفقیت ثبت شد.');
            return redirect()->route('entries.edit', $newEntry);

        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('error', 'خطایی در هنگام ذخیره رخ داد: ' . $e->getMessage());
            return;
        }
    }

    /**
     * [NEW] Opens the modal to select an entry for cross-linking.
     */
    public function openEntrySelector()
    {
        $this->projectEntries = $this->project
            ->labNotebookEntries()
            ->latest()
            ->get(['id', 'sku', 'title']);

        $this->showEntrySelectorModal = true;
    }

    /**
     * [NEW] Dispatches an event to insert the selected entry's SKU into the editor.
     *
     * @param string $sku The SKU of the selected entry.
     */
    public function insertEntrySku(string $sku)
    {
        $this->dispatch('insert-text-in-editor', text: "[[{$sku}]]");
        $this->showEntrySelectorModal = false;
    }

    /**
     * [NEW] Generates a preview of the current editor content.
     */
    public function generatePreview()
    {
        if (empty($this->body)) {
            $this->previewHtml = '<p class="text-center text-gray-500">محتوایی برای پیش‌نمایش وجود ندارد.</p>';
        } else {
            $this->previewHtml = preg_replace_callback(
                '/\[\[(.*?)\]\]/',
                function ($matches) {
                    $sku = trim($matches[1]);
                    if (empty($sku)) return $matches[0];
                    $linkedEntry = LabNotebookEntry::where('sku', $sku)->first();
                    if ($linkedEntry) {
                        return sprintf(
                            '<a href="#" wire:click.prevent="showEntryPreview(%d)" class="text-blue-600 dark:text-blue-400 font-semibold hover:underline" title="مشاهده پیش‌نمایش: %s">[[%s]]</a>',
                            $linkedEntry->id, htmlspecialchars($linkedEntry->title), htmlspecialchars($sku)
                        );
                    }
                    return sprintf(
                        '<span class="text-red-500 dark:text-red-400 line-through" title="یادداشت با شناسه %s یافت نشد">[[%s]]</span>',
                        htmlspecialchars($sku), htmlspecialchars($sku)
                    );
                },
                $this->body
            );
        }
        $this->showContentPreviewModal = true;
    }

    /**
     * [NEW] Fetches a linked entry and shows its preview modal.
     */
    public function showEntryPreview(int $entryId)
    {
        $this->previewingEntry = LabNotebookEntry::with('user')->find($entryId);
        $this->showPreviewModal = true;
    }

    public function render()
    {
        return view('livewire.create-lab-notebook-entry')
            ->layout('layouts.app');
    }
}
