<?php

namespace App\Livewire;

use App\Models\PersonalNote;
use App\Models\PersonalNoteAttachment;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Livewire\WithFileUploads;
use Hekmatinasser\Verta\Verta;
use Illuminate\Support\Collection;

class CreatePersonalNote extends Component
{
    use WithFileUploads;

    public ?PersonalNote $note = null;

    // Form properties
    public string $title = '';
    public string $body = '';
    public string $note_code = '';
    public $newAttachments = [];
    public Collection $existingAttachments;
    public string $note_date = '';

    protected function rules()
    {
        return [
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'note_date' => 'required|string',
            'newAttachments.*' => 'nullable|file|max:10240', // 10MB Max per file
        ];
    }

    public function mount(?PersonalNote $note)
    {
        if ($note && $note->exists) {
            $this->note = $note;
            $this->title = $note->title;
            $this->body = $note->body;
            $this->note_code = $note->note_code;
            $this->existingAttachments = $note->attachments;
            $this->note_date = $this->note->note_date ? Verta::instance($this->note->note_date)->format('Y/m/d') : '';
        } else {
            // Initialize an empty model for creation context
            $this->note = new PersonalNote();
            $this->note_code = $this->generateNoteCode();
            $this->existingAttachments = collect();
            $this->note_date = verta()->format('Y/m/d');
        }
    }

    private function generateNoteCode(): string
    {
        $user = Auth::user();
        $usernamePrefix = strtoupper($user->username);

        $latestNote = PersonalNote::where('user_id', $user->id)->latest('id')->first();
        $nextId = $latestNote ? (int)substr($latestNote->note_code, -6) + 1 : 1;

        do {
            $noteCode = $usernamePrefix . '-' . str_pad($nextId, 6, '0', STR_PAD_LEFT);
            $nextId++;
        } while (PersonalNote::where('note_code', $noteCode)->exists());

        return $noteCode;
    }

//    public function save()
//    {
//        $validatedData = $this->validate();
//
//        $data = [
//            'title' => $this->title,
//            'body' => $this->body,
//            'user_id' => Auth::id(),
//        ];
//
//        if ($this->note->exists) {
//            $this->note->update($data);
//            $noteInstance = $this->note;
//            session()->flash('message', 'یادداشت با موفقیت به‌روزرسانی شد.');
//        } else {
//            $data['note_code'] = $this->note_code;
//            $noteInstance = PersonalNote::create($data);
//            session()->flash('message', 'یادداشت جدید با موفقیت ایجاد شد.');
//        }
//
//        // Handle file uploads
//        foreach ($this->newAttachments as $attachmentFile) {
//            $path = $attachmentFile->store('personal_notes/' . $noteInstance->note_code, 'public');
//            $noteInstance->attachments()->create([
//                'user_id' => Auth::id(),
//                'file_path' => $path,
//                'file_name' => $attachmentFile->getClientOriginalName(),
//                'file_size' => $attachmentFile->getSize(),
//                'mime_type' => $attachmentFile->getMimeType(),
//            ]);
//        }
//
//        return redirect()->route('personal-notebook.index');
//    }

    public function save()
    {
        $this->validate();

        $this->note->title = $this->title;
        $this->note->body = $this->body;
        $this->note->user_id = Auth::id();

        // [NEW] Convert Jalali date to Gregorian before saving
        try {
            if (!empty($this->note_date)) {
                $this->note->note_date = Verta::parse($this->note_date)->toCarbon();
            }
        } catch (\Exception $e) {
            $this->addError('note_date', 'فرمت تاریخ نامعتبر است.');
            return;
        }

        if (!$this->note->exists) {
            $this->note->note_code = $this->note_code;
        }

        $this->note->save();

        // Handle file uploads
        foreach ($this->newAttachments as $attachmentFile) {
            $path = $attachmentFile->store('personal_notes/' . $this->note->id, 'public');
            $this->note->attachments()->create([
                'user_id' => Auth::id(),
                'file_path' => $path,
                'file_name' => $attachmentFile->getClientOriginalName(),
                'file_size' => $attachmentFile->getSize(),
                'mime_type' => $attachmentFile->getMimeType(),
            ]);
        }

        session()->flash('message', $this->note->wasRecentlyCreated ? 'یادداشت جدید با موفقیت ایجاد شد.' : 'یادداشت با موفقیت به‌روزرسانی شد.');

        return redirect()->route('personal-notebook.index');
    }

    public function removeAttachment(PersonalNoteAttachment $attachment)
    {
        Storage::disk('public')->delete($attachment->file_path);
        $attachment->delete();
        $this->existingAttachments = $this->note->attachments()->get(); // Refresh attachments
    }

    public function render()
    {
        return view('livewire.create-personal-note')->layout('layouts.app');
    }
}
