<?php

namespace App\Livewire;

use App\Models\ActivityRequest;
use App\Models\LabNotebookSignature;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;

class DashboardActivities extends Component
{
    // Properties for Signature Modal
    public ?ActivityRequest $requestToSign = null;
    public $confirmingSignature = false;
    public $password = '';

    /**
     * Show the password confirmation modal for signing.
     */
    public function confirmSignature(ActivityRequest $request)
    {
        // We only proceed if the request is for signing
        if ($request->activity_type !== 'امضا') {
            // Handle other activity types here in the future
            return;
        }

        $this->resetErrorBag();
        $this->password = '';
        $this->requestToSign = $request;
        $this->confirmingSignature = true;
    }

    /**
     * Sign the entry, lock it, and complete the activity request.
     */
    public function signEntry()
    {
        if (!$this->requestToSign) {
            return;
        }

        // 1. Validate the password
        if (!Hash::check($this->password, Auth::user()->password)) {
            $this->addError('password', 'رمز عبور وارد شده صحیح نیست.');
            return;
        }

        $entry = $this->requestToSign->labNotebookEntry;

        // 2. Create the signature record
        $entry->signatures()->create([
            'user_id' => Auth::id(),
            'signed_at' => now(),
        ]);

        // 3. Lock the entry
        $entry->update([
            'is_locked' => true,
            'locked_at' => now(),
            'locked_by' => Auth::id(),
        ]);

        // 4. Mark the activity request as completed
        $this->requestToSign->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);

        $this->confirmingSignature = false;
        session()->flash('message', 'یادداشت با موفقیت امضا و قفل شد.');
    }

    public function render()
    {
        // ما درخواست‌های فعالیتی را که به کاربر فعلی محول شده و هنوز در حالت "pending" هستند، می‌خوانیم
        // و اطلاعات مرتبط با آن‌ها (درخواست‌کننده، یادداشت و طرح) را نیز به صورت بهینه بارگذاری می‌کنیم
        $activityRequests = ActivityRequest::where('assignee_id', Auth::id())
            ->where('status', 'pending')
            ->with(['requester', 'labNotebookEntry.project'])
            ->latest()
            ->get();

        return view('livewire.dashboard-activities', [
            'requests' => $activityRequests,
        ]);
    }
}
