<?php

namespace App\Livewire;

use App\Models\Equipment;
use App\Models\Project;
use App\Models\ResearchTeam;
use App\Models\Task;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class DashboardStats extends Component
{
    public int $activeProjectsCount = 0;
    public int $remainingTasksCount = 0;
    public int $equipmentCount = 0;
    public int $teamMembersCount = 0;

    public bool $canViewEquipmentStats = false;
    public bool $canViewTeamStats = false;

    public function mount()
    {
        $user = Auth::user();

        // 1. Calculate Active Projects
        $this->activeProjectsCount = Project::where('status', 'در حال اجرا')
            ->where(function ($query) use ($user) {
                $query->where('owner_id', $user->id)
                    ->orWhereHas('collaborators', function ($q) use ($user) {
                        $q->where('users.id', $user->id);
                    });
            })
            ->count();

        // 2. Calculate Remaining Tasks
        $this->remainingTasksCount = Task::where('status', '!=', 'Done')
            ->whereHas('assignees', function ($query) use ($user) {
                $query->where('users.id', $user->id);
            })
            ->count();

        // 3. Calculate Equipment Count (with permission check)
        $managedTeamIds = $user->researchTeams()->wherePivot('role', 'admin')->pluck('research_teams.id');
        $calibrationManagedTeamIds = ResearchTeam::where('calibration_manager_id', $user->id)->pluck('id');
        $allVisibleTeamIds = $managedTeamIds->merge($calibrationManagedTeamIds)->unique();

        if ($allVisibleTeamIds->isNotEmpty()) {
            $this->canViewEquipmentStats = true;
            $this->equipmentCount = Equipment::whereIn('research_team_id', $allVisibleTeamIds)->count();
        }

        // 4. Calculate Team Members Count (with permission check)
        if ($managedTeamIds->isNotEmpty()) {
            $this->canViewTeamStats = true;
            // Eager load users and count unique members across all managed teams
            $teamsWithMembers = ResearchTeam::with('users')->whereIn('id', $managedTeamIds)->get();
            $this->teamMembersCount = $teamsWithMembers->pluck('users')->flatten()->unique('id')->count();
        }
    }

    public function render()
    {
        return view('livewire.dashboard-stats');
    }
}
