<?php

namespace App\Livewire;

use App\Models\ActivityRequest;
use App\Models\InventoryItem;
use App\Models\LabNotebookEntry;
use App\Models\User;
use App\Models\LabNotebookComment;
use App\Models\Attachment;
use App\Models\InventoryConsumptionRequest; // [NEW]
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;
use Illuminate\Support\Facades\Storage;
use App\Notifications\LowStockAlert; // کلاس جدید ما
use Hekmatinasser\Verta\Verta;
use Livewire\WithFileUploads;
use App\Services\AIService;
use Spatie\Tags\Tag;
use Illuminate\Support\Facades\Notification; // [NEW]
use App\Notifications\YouWereMentionedInComment; // [NEW]



class EditLabNotebookEntry extends Component
{
    use AuthorizesRequests;
    use WithFileUploads;

    public LabNotebookEntry $entry;

    // Form properties
    public $title;
    public $entry_date;
    public $body;
    // Attachments Properties
    public $newAttachments = [];
    public $existingAttachments = [];
    public array $attachmentCaptions = [];
    public bool $confirmingAttachments = false;

    // Materials Management Properties
    public string $materialSearch = '';
    public $materialSearchResults = [];
    public array $usedMaterials = [];

    // Activity Request Modal properties
    public $confirmingActivityRequest = false;
    public $activityType;
    public $activityAssigneeId;
    public string $activitySearch = '';
    public $activitySearchResults = [];

    // Properties for Return to Stock Modal
    public $itemToReturnId;
    public $itemToReturnName = ''; // مقدار اولیه اضافه شد
    public $quantityToReturn;
    public $confirmingReturnToStock = false;
    // Properties for Remove Material Modal
    public $confirmingMaterialRemoval = false;
    public $confirmingStockError = false;
    public $stockErrorMessage = '';

    // Properties for Signature
    public ?ActivityRequest $signatureRequest = null;
    public $confirmingSignature = false;
    public $password = '';
    // [NEW] Properties for conditional modal
    public bool $hasHardErrors = false;
    public array $versionsToCompare = [];

    // [NEW] Properties for Entry Selector Modal
    public bool $showEntrySelectorModal = false;
    public $projectEntries = [];
    // [NEW] Properties for Content Preview
    public bool $showContentPreviewModal = false;
    public string $previewHtml = '';

    // [NEW] Properties for Linked Entry Preview
    public ?LabNotebookEntry $previewingEntry = null;
    public bool $showPreviewModal = false;

    // [NEW] Properties for AI Analysis
    public bool $showAnalysisModal = false;
    public ?array $analysisResult = null;
    public bool $analyzing = false;
    public bool $showImageAnalysisModal = false;
    public ?array $imageAnalysisResult = null;
    public ?Attachment $analyzingAttachment = null;
    public bool $isAnalyzingImage = false;
    public $tags = [];

    // --- [NEW] Properties for Comments ---
    public $comments = [];
    public string $newComment = '';
    public ?int $replyingToCommentId = null; // To track which comment is being replied to
    public string $replyBody = ''; // The content of the reply
    public ?string $newCommentAnchorId = null;


    protected $listeners = ['editorContentUpdated', 'restoreVersionConfirmed' => 'restoreVersion'];



    protected function rules()
    {
        return [
            'title' => 'required|string|max:255',
            'entry_date' => 'required|date',
            'body' => 'required|string|min:10',
            'newAttachments.*' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240',
            'attachmentCaptions.*' => 'nullable|string|max:255',
        ];
    }

    public function editorContentUpdated($content)
    {
        $this->body = $content;
    }

    public function mount(LabNotebookEntry $entry)
    {
        $this->authorize('view', $entry->project);
//        $this->entry = $entry->load('project.collaborators', 'project.owner', 'materials');
        $this->entry = $entry->load('project.researchTeam', 'project.collaborators', 'project.owner', 'materials', 'project.budgetItems', 'consumptionRequests.requester', 'consumptionRequests.approver', 'consumptionRequests.inventoryItem', 'comments.user', 'comments.replies.user');
        $this->loadComments();
        $this->title = $entry->title;
        if ($entry->entry_date) {
            $this->entry_date = Verta::instance($entry->entry_date)->format('Y/m/d');
        }
        $this->body = $entry->body;
        // [NEW] Load existing tags into the tags array
        $this->tags = $entry->tags->pluck('name')->toArray();
        $this->loadExistingAttachments();
        // بررسی اینکه آیا یک درخواست امضای فعال برای این کاربر و این یادداشت وجود دارد یا خیر
        $this->signatureRequest = ActivityRequest::where('assignee_id', Auth::id())
            ->where('lab_notebook_entry_id', $this->entry->id)
            ->where('activity_type', 'امضا')
            ->where('status', 'pending')
            ->first();
        // واکشی ID آیتم‌هایی که برای آن‌ها درخواست ثبت شده است
        $itemIdsWithRequests = $this->entry->consumptionRequests->pluck('inventory_item_id')->unique();

        $this->usedMaterials = $this->entry->materials->map(function ($material) use ($itemIdsWithRequests) {
            return [
                'id' => $material->id,
                'name' => $material->name,
                'sku' => $material->sku,
                'unit' => $material->unit,
                'quantity_used' => $material->pivot->quantity_used,
                'is_locked_by_request' => $itemIdsWithRequests->contains($material->id), // [NEW] وضعیت قفل

            ];
        })->toArray();
    }

    public function loadExistingAttachments()
    {
        $this->existingAttachments = $this->entry->attachments->toArray();
    }

    public function updatedNewAttachments()
    {
        $this->validate(['newAttachments.*' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240']);
        $this->attachmentCaptions = array_pad($this->attachmentCaptions, count($this->newAttachments), '');
        $this->confirmingAttachments = true;
    }

    public function saveNewAttachments()
    {
        foreach ($this->newAttachments as $index => $attachmentFile) {
            $path = $attachmentFile->store('labnotes/' . $this->entry->sku, 'public');
            $this->entry->attachments()->create([
                'user_id' => Auth::id(),
                'file_path' => $path,
                'file_name' => $attachmentFile->getClientOriginalName(),
                'file_size' => $attachmentFile->getSize(),
                'mime_type' => $attachmentFile->getMimeType(),
                'caption' => $this->attachmentCaptions[$index] ?? null,
            ]);
        }
        $this->newAttachments = [];
        $this->attachmentCaptions = [];
        $this->confirmingAttachments = false;
        $this->loadExistingAttachments();
    }

    public function deleteAttachment($attachmentId)
    {
        $attachment = Attachment::findOrFail($attachmentId);
        // $this->authorize('delete', $attachment);

        Storage::disk('public')->delete($attachment->file_path);
        $attachment->delete();
        $this->loadExistingAttachments();
    }

    /**
     * Show the password confirmation modal for signing.
     */
    public function confirmSignature()
    {
        // ما قبل از هر کاری، از Policy اجازه می‌گیریم
        $this->authorize('sign', $this->entry);
        $this->resetErrorBag();
        $this->password = '';
        $this->confirmingSignature = true;
    }

    /**
     * Sign the entry, lock it, and complete the activity request.
     */
    public function signEntry()
    {
        $this->authorize('sign', $this->entry);

        if (!Hash::check($this->password, Auth::user()->password)) {
            $this->addError('password', 'رمز عبور وارد شده صحیح نیست.');
            return;
        }

        // 1. Create the signature record
        $this->entry->signatures()->create([
            'user_id' => Auth::id(),
            'signed_at' => now(),
        ]);

        // 2. Lock the entry
        $this->entry->update([
            'is_locked' => true,
            'locked_at' => now(),
            'locked_by' => Auth::id(),
        ]);

        // 3. Mark the activity request as completed
//        $this->signatureRequest->update([
//            'status' => 'completed',
//            'completed_at' => now(),
//        ]);

        // 3. If a signature request exists for this user, mark it as completed
        $signatureRequest = ActivityRequest::where('assignee_id', Auth::id())
            ->where('lab_notebook_entry_id', $this->entry->id)
            ->where('activity_type', 'امضا')
            ->where('status', 'pending')
            ->first();

        if ($signatureRequest) {
            $signatureRequest->update([
                'status' => 'completed',
                'completed_at' => now(),
            ]);
        }

        $this->confirmingSignature = false;
        $this->entry->refresh(); // Refresh entry data
        $this->signatureRequest = null; // Clear the completed request
        session()->flash('message', 'یادداشت با موفقیت امضا و قفل شد.');
    }

    /**
     * Show the modal for returning an item to stock.
     */
    public function confirmReturnToStock($itemId)
    {
        $this->authorize('manageMaterials', $this->entry);
        $this->resetErrorBag();
        $material = collect($this->usedMaterials)->firstWhere('id', $itemId);
        $this->itemToReturnId = $itemId;
        $this->itemToReturnName = $material['name'];
        $this->quantityToReturn = ''; // Reset the input field
        $this->confirmingReturnToStock = true;
    }

    /**
     * Process the return to stock action.
     */
    public function returnToStock()
    {
        $this->authorize('manageMaterials', $this->entry);
        $materialData = collect($this->usedMaterials)->firstWhere('id', $this->itemToReturnId);
        $this->validate([
            'quantityToReturn' => 'required|numeric|min:0.0001|max:' . $materialData['quantity_used']
        ]);

        DB::transaction(function () use ($materialData) {
            $inventoryItem = InventoryItem::find($this->itemToReturnId);
            $inventoryItem->increment('quantity', $this->quantityToReturn);
            $this->entry->project->budgetItems()->where('inventory_item_id', $this->itemToReturnId)->decrement('total_consumed_quantity', $this->quantityToReturn);

            $newQuantityUsed = $materialData['quantity_used'] - $this->quantityToReturn;

            if ($newQuantityUsed > 0) {
                $this->entry->materials()->updateExistingPivot($this->itemToReturnId, ['quantity_used' => $newQuantityUsed]);
            } else {
                $this->entry->materials()->detach($this->itemToReturnId);
            }
        });

        $this->confirmingReturnToStock = false;
        $this->entry->refresh()->load('materials', 'project.budgetItems');
        $this->mount($this->entry); // Remount to refresh the usedMaterials array
        session()->flash('message', 'مقدار مورد نظر با موفقیت به انبار بازگردانده شد.');
    }

    /**
     * Show the confirmation modal for removing a material.
     */
    public function confirmMaterialRemoval($itemId)
    {
        $this->authorize('manageMaterials', $this->entry);
        $this->itemToReturnId = $itemId;
        $this->confirmingMaterialRemoval = true;
    }

    /**
     * Remove the material from the list and return its full quantity to stock.
     */
    public function removeMaterialAndReturnStock()
    {
        $this->authorize('manageMaterials', $this->entry);
        DB::transaction(function () {
            $materialData = collect($this->usedMaterials)->firstWhere('id', $this->itemToReturnId);
            $inventoryItem = InventoryItem::find($this->itemToReturnId);

            if ($inventoryItem && $materialData) {
                // 1. بازگرداندن موجودی به انبار اصلی
                $inventoryItem->increment('quantity', $materialData['quantity_used']);
                // 2. کسر کردن کل مقدار از مصرف بودجه
                $this->entry->project->budgetItems()->where('inventory_item_id', $this->itemToReturnId)->decrement('total_consumed_quantity', $materialData['quantity_used']);
            }
            $this->entry->materials()->detach($this->itemToReturnId);
        });
        $this->confirmingMaterialRemoval = false;
        $this->entry->refresh()->load('materials', 'project.budgetItems');
        $this->mount($this->entry);
        session()->flash('message', 'آیتم حذف و موجودی آن به انبار و بودجه بازگردانده شد.');
    }



    public function updatedMaterialSearch($value)
    {
        if (strlen($this->materialSearch) < 2) {
            $this->materialSearchResults = [];
            return;
        }
        $this->materialSearchResults = $this->entry->project->budgetItems()
            ->where('name', 'like', '%' . $this->materialSearch . '%')
            ->take(5)
            ->get();
    }

    public function addMaterial(InventoryItem $item)
    {
        $this->authorize('manageMaterials', $this->entry); // <-- بررسی دسترسی
        if (!collect($this->usedMaterials)->contains('id', $item->id)) {
            $this->usedMaterials[] = [
                'id' => $item->id,
                'name' => $item->name,
                'sku' => $item->sku,
                'unit' => $item->unit,
                'quantity_used' => '',
                'is_locked_by_request' => false, // [FIX] افزودن کلید با مقدار پیش‌فرض
            ];
        }
        $this->materialSearch = '';
        $this->materialSearchResults = [];
    }

    public function removeMaterial($itemId)
    {
        $this->authorize('manageMaterials', $this->entry); // <-- بررسی دسترسی
        $this->usedMaterials = array_filter($this->usedMaterials, fn($material) => $material['id'] != $itemId);
    }

    public function updatedActivitySearch($value)
    {
        if (strlen($this->activitySearch) < 1) {
            $this->activitySearchResults = [];
            return;
        }
        $teamMembers = $this->entry->project->collaborators->push($this->entry->project->owner)->unique('id');
        $this->activitySearchResults = $teamMembers->filter(function ($user) {
            return str_contains(strtolower($user->first_name), strtolower($this->activitySearch)) ||
                str_contains(strtolower($user->last_name), strtolower($this->activitySearch)) ||
                str_contains(strtolower($user->username), strtolower($this->activitySearch));
        })->take(5);
    }

    public function selectActivityAssignee(User $user)
    {
        $this->activityAssigneeId = $user->id;
        $this->activitySearch = $user->first_name . ' ' . $user->last_name;
        $this->activitySearchResults = [];
    }


    /**
     * [REWRITTEN] Creates a new version of the current lab note entry,
     * including a snapshot of its materials and attachments.
     */
    private function createVersion()
    {
        $latestVersion = $this->entry->versions()->latest('version')->first();
        $newVersionNumber = $latestVersion ? $latestVersion->version + 1 : 1;

        // Create a log of the materials used in the original entry state
        $materialsLog = $this->entry->materials->map(function ($material) {
            return [
                'name' => $material->name,
                'sku' => $material->sku,
                'quantity_used' => $material->pivot->quantity_used,
                'unit' => $material->unit,
            ];
        });

        // [NEW] Create a log of the attachments from the original entry state
        $attachmentsLog = $this->entry->attachments->map(function ($attachment) {
            return [
                'id' => $attachment->id,
                'file_name' => $attachment->file_name,
                'file_path' => $attachment->file_path,
                'caption' => $attachment->caption,
            ];
        });

        $this->entry->versions()->create([
            'user_id'         => Auth::id(),
            'version'         => $newVersionNumber,
            'sku'             => $this->entry->getOriginal('sku'),
            'title'           => $this->entry->getOriginal('title'),
            'body'            => $this->entry->getOriginal('body'),
            'entry_date'      => $this->entry->getOriginal('entry_date'),
            'materials_log'   => $materialsLog,
            'attachments_log' => $attachmentsLog, // [NEW] Save the attachments log
        ]);
    }
    /**
     * Checks if there are any changes in the form.
     */
    private function hasChanges(): bool
    {
        // 1. مقایسه فیلدهای متنی ساده
        if ($this->entry->getOriginal('title') != $this->title) return true;
        if ($this->entry->getOriginal('body') != $this->body) return true;

        // 2. مقایسه دقیق تاریخ‌ها
        $originalDate = $this->entry->getOriginal('entry_date');
        if ($originalDate && !empty($this->entry_date)) {
            // اگر هر دو تاریخ وجود دارند، تاریخ اصلی را به شمسی تبدیل کرده و با رشته فعلی مقایسه می‌کنیم
            if (Verta::instance($originalDate)->format('Y/m/d') != $this->entry_date) return true;
        } elseif ($originalDate || !empty($this->entry_date)) {
            // اگر یکی از تاریخ‌ها خالی و دیگری پر باشد
            return true;
        }

        // 3. مقایسه مواد مصرفی
        $originalMaterials = $this->entry->materials->map(fn($m) => ['id' => $m->id, 'quantity_used' => (string)$m->pivot->quantity_used])->keyBy('id')->all();
        $currentMaterials = collect($this->usedMaterials)->map(fn($m) => ['id' => $m['id'], 'quantity_used' => (string)$m['quantity_used']])->keyBy('id')->all();
        if ($originalMaterials != $currentMaterials) return true;

        // 4. بررسی اینکه آیا فایل جدیدی برای آپلود انتخاب شده است یا خیر
        if (!empty($this->newAttachments)) return true;

        return false;
    }



    /**
     * [REWRITTEN] Step 1: Validate inputs and show modals if necessary.
     */
    public function update()
    {
        $this->authorize('update', $this->entry);
        if (!$this->hasChanges()) {
            session()->flash('message', 'هیچ تغییری برای ذخیره وجود ندارد.');
            return;
        }
        // 1. تبدیل تاریخ شمسی به میلادی قبل از هر کاری
        try {
            if (!empty($this->entry_date)) {
                $this->entry_date = Verta::parse($this->entry_date)->toCarbon();
            }
        } catch (\Exception $e) {
            $this->addError('entry_date', 'فرمت تاریخ نامعتبر است.');
            return;
        }

        // 2. اعتبارسنجی تمام داده‌های فرم
        $validatedData = $this->validate([
            'title' => 'required|string|max:255',
            'entry_date' => 'required|date',
            'body' => 'required|string|min:10',
            'usedMaterials.*.quantity_used' => 'required|numeric|min:0',
            'newAttachments.*' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240',
            'attachmentCaptions.*' => 'nullable|string|max:255',
        ]);

        // 3. اعتبارسنجی‌های پیشرفته مربوط به مواد مصرفی
        if (Auth::user()->can('manageMaterials', $this->entry)) {
            $hardErrors = [];
            $softWarnings = [];
            $allowedItems = $this->entry->project->budgetItems->keyBy('id');
            $originalMaterials = $this->entry->materials->keyBy('id');

            foreach ($this->usedMaterials as $material) {
                $inventoryItem = InventoryItem::find($material['id']);
                if (!$inventoryItem) continue;

                $originalQtyUsed = $originalMaterials->get($material['id'])?->pivot->quantity_used ?? 0;
                $newQtyUsed = (float) $material['quantity_used'];
                $qtyDifference = $newQtyUsed - $originalQtyUsed;

                // بررسی وجود درخواست فعال
                if ($qtyDifference != 0) {
                    if (InventoryConsumptionRequest::where('lab_notebook_entry_id', $this->entry->id)->where('inventory_item_id', $material['id'])->where('status', 'pending')->exists()) {
                        $hardErrors[] = "یک درخواست فعال برای ماده '{$material['name']}' وجود دارد.";
                        continue;
                    }
                }

                // بررسی بودجه و موجودی
                if (!$allowedItems->has($material['id'])) {
                    $hardErrors[] = "ماده '{$material['name']}' در لیست مواد مجاز این طرح قرار ندارد.";
                    continue;
                }
                if ($qtyDifference > 0) {
                    $budgetItem = $allowedItems->get($material['id']);
                    $remainingBudget = $budgetItem->pivot->max_allowed_quantity - $budgetItem->pivot->total_consumed_quantity;
                    if ($qtyDifference > $remainingBudget) {
                        $hardErrors[] = "مصرف ماده '{$material['name']}' از سقف مجاز بودجه فراتر می‌رود.";
                    }
                    if ($inventoryItem->quantity < $qtyDifference) {
                        $softWarnings[] = "موجودی انبار برای '{$material['name']}' کافی نیست.";
                    }
                }
            }

            if (!empty($hardErrors)) {
                $this->hasHardErrors = true;
                $this->stockErrorMessage = implode('<br>', $hardErrors);
                $this->confirmingStockError = true;
                return;
            }

            if (!empty($softWarnings)) {
                $this->hasHardErrors = false;
                $this->stockErrorMessage = implode('<br>', $softWarnings) . "<br><br>درخواست شما همچنان برای بررسی ارسال خواهد شد. آیا مایل به ادامه هستید؟";
                $this->confirmingStockError = true;
                return;
            }
        }

        // 4. اگر تمام اعتبارسنجی‌ها موفق بود، عملیات دیتابیس را در یک تراکنش انجام بده
        DB::transaction(function () use ($validatedData) {
            $this->entry->update([
                'title' => $validatedData['title'],
                'entry_date' => $validatedData['entry_date'],
                'body' => $validatedData['body'],
            ]);
// [NEW] Sync the tags with the entry
            $this->entry->syncTags($this->tags);
            // ذخیره پیوست‌های جدید
            foreach ($this->newAttachments as $index => $attachmentFile) {
                $path = $attachmentFile->store('labnotes/' . $this->entry->sku, 'public');
                $this->entry->attachments()->create([
                    'user_id' => Auth::id(),
                    'file_path' => $path,
                    'file_name' => $attachmentFile->getClientOriginalName(),
                    'file_size' => $attachmentFile->getSize(),
                    'mime_type' => $attachmentFile->getMimeType(),
                    'caption' => $this->attachmentCaptions[$index] ?? null,
                ]);
            }
            $this->createVersion();
            // ایجاد درخواست‌های مصرف مواد

            if (Auth::user()->can('manageMaterials', $this->entry)) {
                $originalMaterials = $this->entry->materials->keyBy('id');
                $materialsToSync = [];

                // ... (منطق ایجاد درخواست مصرف/بازگشت، دقیقاً مانند قبل)
                foreach ($this->usedMaterials as $material) {
                    $originalQtyUsed = $originalMaterials->get($material['id'])?->pivot->quantity_used ?? 0;
                    $newQtyUsed = (float) $material['quantity_used'];
                    $qtyDifference = $newQtyUsed - $originalQtyUsed;

                    if ($qtyDifference > 0) {
                        InventoryConsumptionRequest::create([
                            'project_id' => $this->entry->project_id, 'lab_notebook_entry_id' => $this->entry->id,
                            'requester_id' => Auth::id(), 'inventory_item_id' => $material['id'],
                            'type' => 'consumption', 'quantity' => $qtyDifference, 'status' => 'pending',
                        ]);
                    } elseif ($qtyDifference < 0) {
                        InventoryConsumptionRequest::create([
                            'project_id' => $this->entry->project_id, 'lab_notebook_entry_id' => $this->entry->id,
                            'requester_id' => Auth::id(), 'inventory_item_id' => $material['id'],
                            'type' => 'return', 'quantity' => abs($qtyDifference), 'status' => 'pending',
                        ]);
                    }
                    $materialsToSync[$material['id']] = ['quantity_used' => $newQtyUsed];
                }

                $newMaterialIds = collect($this->usedMaterials)->pluck('id');
                foreach($originalMaterials as $originalMaterial) {
                    if (!$newMaterialIds->contains($originalMaterial->id)) {
                        InventoryConsumptionRequest::create([
                            'project_id' => $this->entry->project_id, 'lab_notebook_entry_id' => $this->entry->id,
                            'requester_id' => Auth::id(), 'inventory_item_id' => $originalMaterial->id,
                            'type' => 'return', 'quantity' => $originalMaterial->pivot->quantity_used, 'status' => 'pending',
                        ]);
                    }
                }
                $this->entry->materials()->sync($materialsToSync);
            }

        });

        session()->flash('message', 'یادداشت با موفقیت به‌روزرسانی شد.');
        $this->mount($this->entry->fresh());
    }
    /**
     * [NEW] Step 2: Proceed with the update after confirmation (if needed).
     */
    public function confirmActivityRequest()
    {
        $this->authorize('update', $this->entry);
        $this->resetErrorBag();
        $this->reset(['activityType', 'activityAssigneeId', 'activitySearch', 'activitySearchResults']);
        $this->confirmingActivityRequest = true;
    }

    public function sendActivityRequest()
    {
        $this->authorize('update', $this->entry);
        $this->validate([
            'activityType' => 'required|string|in:بررسی,آرشیوسازی,امضا,قفل یادداشت',
            'activityAssigneeId' => 'required|exists:users,id',
        ]);

        ActivityRequest::create([
            'requester_id' => Auth::id(),
            'assignee_id' => $this->activityAssigneeId,
            'lab_notebook_entry_id' => $this->entry->id,
            'activity_type' => $this->activityType,
        ]);

        $this->confirmingActivityRequest = false;
        session()->flash('message', 'درخواست فعالیت با موفقیت ارسال شد.');
    }

    public function toggleLock()
    {
        $this->authorize('toggleLock', $this->entry);

        $this->entry->update([
            'is_locked' => !$this->entry->is_locked,
            'locked_at' => $this->entry->is_locked ? null : now(),
            'locked_by' => $this->entry->is_locked ? null : Auth::id(),
        ]);

        $this->entry->refresh();
        session()->flash('message', 'وضعیت قفل یادداشت با موفقیت تغییر کرد.');
    }


    /**
     * Save the valid parts of the form, create a version, and then redirect.
     */
    public function saveAndRedirectToInventory()
    {
        $this->authorize('update', $this->entry);
        $validatedData = $this->validate([
            'title' => 'required|string|max:255',
            'entry_date' => 'required|date',
            'body' => 'required|string|min:10',
        ]);

        DB::transaction(function() use ($validatedData) {
            $this->createVersion();
            $this->entry->update($validatedData);
        });

        session()->flash('inventory_redirect_message', 'تغییرات شما در یادداشت ذخیره شد. حالا می‌توانید انبار را مدیریت کنید.');
        return redirect()->route('inventory.index');
    }
    /**
     * [NEW] Restores the form state from a previous version.
     * This is the core functionality of Phase 3.
     */
    public function restoreVersion($versionId)
    {
        $this->authorize('update', $this->entry);

        $versionToRestore = \App\Models\LabNotebookEntryVersion::find($versionId);

        if (!$versionToRestore) {
            session()->flash('error', 'نسخه مورد نظر برای بازگردانی یافت نشد.');
            return;
        }

        // 1. Create a backup version of the current state before overwriting
        $this->createVersion();

        // 2. Overwrite the component's public properties with the version's data
        $this->title = $versionToRestore->title;
        $this->body = $versionToRestore->body;
        $this->entry_date = \Hekmatinasser\Verta\Verta::instance($versionToRestore->entry_date)->format('Y/m/d');

        // 3. Rebuild the usedMaterials array from the version's log
        $this->usedMaterials = collect($versionToRestore->materials_log ?? [])->map(function ($materialLog) {
            // We need to find the original inventory item to get its current ID and SKU
            $inventoryItem = \App\Models\InventoryItem::where('sku', $materialLog['sku'])->first();
            return [
                'id' => $inventoryItem->id ?? null, // Use current ID
                'name' => $materialLog['name'],
                'sku' => $materialLog['sku'],
                'unit' => $materialLog['unit'],
                'quantity_used' => $materialLog['quantity_used'],
                'is_locked_by_request' => false, // Restored items are not locked by default
            ];
        })->filter(fn($item) => $item['id'] !== null)->toArray(); // Filter out any materials that no longer exist

        // 4. Dispatch a browser event to update the Jodit editor's content
        $this->dispatch('updateEditorContent', content: $this->body);

        // 5. Inform the user
        session()->flash('message', 'اطلاعات با موفقیت از نسخه ' . $versionToRestore->version . ' بازگردانی شد. لطفاً تغییرات را بررسی کرده و برای نهایی شدن، دکمه "ذخیره تغییرات" را بزنید.');
    }


    /**
     * [NEW] Analyzes the notebook entry's body using the AIService.
     *
     * @param AIService $aiService The AI service injected by Laravel's service container.
     */
    public function analyzeWithAI(AIService $aiService)
    {
        $this->analyzing = true;
        $this->analysisResult = null; // Reset previous results

        // Call the service to get the analysis
        $result = $aiService->analyzeNotebookEntry($this->body);

        if ($result && isset($result['summary']) && isset($result['entities'])) {
            $this->analysisResult = $result;
        } else {
            // If the analysis fails, flash an error message to the user
            session()->flash('error', 'تحلیل با هوش مصنوعی با خطا مواجه شد. لطفاً دوباره تلاش کنید یا از صحت کلید API خود اطمینان حاصل نمایید.');
        }

        $this->analyzing = false;
        $this->showAnalysisModal = true;
    }
    /**
     * [NEW] Analyzes an attached image using the AIService.
     *
     * @param int $attachmentId The ID of the attachment to analyze.
     * @param AIService $aiService The AI service.
     */
    public function analyzeAttachment(int $attachmentId, AIService $aiService)
    {
        $this->isAnalyzingImage = true;
        $this->imageAnalysisResult = null;
        $this->analyzingAttachment = Attachment::find($attachmentId);

        if (!$this->analyzingAttachment) {
            $this->isAnalyzingImage = false;
            return;
        }

        $result = $aiService->analyzeImageForCellCount($this->analyzingAttachment->file_path);

        if ($result && isset($result['cell_count'])) {
            $this->imageAnalysisResult = $result;
        } else {
            session()->flash('error', 'تحلیل تصویر با خطا مواجه شد. لطفاً دوباره تلاش کنید.');
        }

        $this->isAnalyzingImage = false;
        $this->showImageAnalysisModal = true;
    }

    /**
     * [NEW] Adds a tag to the local array.
     * This can be called from Alpine.js in the view.
     */
    public function addTag($tag)
    {
        $tagName = trim($tag);
        if ($tagName && !in_array($tagName, $this->tags)) {
            $this->tags[] = $tagName;
        }
    }

    /**
     * [NEW] Removes a tag from the local array.
     */
    public function removeTag($tag)
    {
        $this->tags = array_filter($this->tags, fn($t) => $t !== $tag);
    }

    /**
     * [NEW] Opens the modal to select an entry for cross-linking.
     */
    public function openEntrySelector()
    {
        // Load all entries from the same project, excluding the current one.
        $this->projectEntries = $this->entry->project
            ->labNotebookEntries()
            ->where('id', '!=', $this->entry->id)
            ->latest()
            ->get(['id', 'sku', 'title']);

        $this->showEntrySelectorModal = true;
    }

    /**
     * [NEW] Dispatches an event to insert the selected entry's SKU into the editor.
     *
     * @param string $sku The SKU of the selected entry.
     */
    public function insertEntrySku(string $sku)
    {
        $this->dispatch('insert-text-in-editor', text: "[[{$sku}]]");
        $this->showEntrySelectorModal = false;
    }

    /**
     * [NEW] Generates a preview of the current editor content with processed cross-links.
     */
    public function generatePreview()
    {
        if (empty($this->body)) {
            $this->previewHtml = '<p class="text-center text-gray-500">محتوایی برای پیش‌نمایش وجود ندارد.</p>';
        } else {
            // Replicate the logic from the 'processed_body' accessor
            $this->previewHtml = preg_replace_callback(
                '/\[\[(.*?)\]\]/',
                function ($matches) {
                    $sku = trim($matches[1]);
                    if (empty($sku)) return $matches[0];

                    $linkedEntry = LabNotebookEntry::where('sku', $sku)->where('id', '!=', $this->entry->id)->first();

                    if ($linkedEntry) {
                        return sprintf(
                            '<a href="#" wire:click.prevent="showEntryPreview(%d)" class="text-blue-600 dark:text-blue-400 font-semibold hover:underline" title="مشاهده پیش‌نمایش: %s">[[%s]]</a>',
                            $linkedEntry->id, htmlspecialchars($linkedEntry->title), htmlspecialchars($sku)
                        );
                    }
                    return sprintf(
                        '<span class="text-red-500 dark:text-red-400 line-through" title="یادداشت با شناسه %s یافت نشد">[[%s]]</span>',
                        htmlspecialchars($sku), htmlspecialchars($sku)
                    );
                },
                $this->body
            );
        }
        $this->showContentPreviewModal = true;
    }

    /**
     * [NEW] Fetches a linked entry and shows its preview modal.
     */
    public function showEntryPreview(int $entryId)
    {
        $this->previewingEntry = LabNotebookEntry::with('user')->find($entryId);
        $this->showPreviewModal = true;
    }
    /**
     * [NEW] Loads top-level comments and their replies for the current entry.
     */
    public function loadComments()
    {
        $this->comments = $this->entry->comments()
            ->with('user', 'replies.user') // Eager load relationships for performance
            ->whereNull('parent_id') // Fetch only top-level comments
            ->latest()
            ->get();
    }

    /**
     * [NEW] Adds a new top-level comment to the entry.
     */
    public function addComment()
    {
        $this->validate(['newComment' => 'required|string|min:1']);

        $comment = $this->entry->comments()->create([
            'user_id' => Auth::id(),
            'body' => $this->newComment,
            'comment_anchor_id' => $this->newCommentAnchorId, // [NEW] Save the anchor ID
        ]);

        $this->processMentions($comment);

        // Reset state
        $this->newComment = '';
        $this->newCommentAnchorId = null;
        $this->loadComments();
    }

    /**
     * [NEW] Deletes a comment after authorization.
     *
     * @param int $commentId The ID of the comment to delete.
     */
    public function deleteComment(int $commentId)
    {
        $comment = LabNotebookComment::findOrFail($commentId);

        // Authorize that the current user can delete this comment
        // You can define a 'delete' policy for LabNotebookComment later
        if (Auth::id() !== $comment->user_id) {
            abort(403);
        }

        $comment->delete();
        $this->loadComments(); // Refresh the list
    }
    /**
     * [NEW] Sets the component state to start replying to a specific comment.
     *
     * @param int $commentId The ID of the parent comment.
     */
    public function startReply(int $commentId)
    {
        $this->replyingToCommentId = $commentId;
        $this->replyBody = ''; // Clear previous reply text
    }

    /**
     * [NEW] Resets the reply state.
     */
    public function cancelReply()
    {
        $this->replyingToCommentId = null;
        $this->replyBody = '';
    }

    /**
     * [NEW] Adds a new reply to a parent comment and processes mentions.
     */
    public function addReply()
    {
        $this->validate(['replyBody' => 'required|string|min:1']);

        $parentComment = LabNotebookComment::find($this->replyingToCommentId);
        if (!$parentComment) {
            return; // Or show an error
        }

        $reply = $parentComment->replies()->create([
            'user_id' => Auth::id(),
            'lab_notebook_entry_id' => $this->entry->id,
            'body' => $this->replyBody,
        ]);

        $this->processMentions($reply); // [NEW] Process mentions for the reply

        $this->cancelReply(); // Reset the form
        $this->loadComments(); // Refresh the list
    }

    /**
     * [NEW] Parses a comment's body for @username mentions and sends notifications.
     *
     * @param LabNotebookComment $comment The newly created comment or reply.
     */
    private function processMentions(LabNotebookComment $comment)
    {
        preg_match_all('/@([a-zA-Z0-9_]+)/', $comment->body, $matches);
        $mentionedUsernames = array_unique($matches[1]);

        if (empty($mentionedUsernames)) {
            return;
        }

        $mentionedUsers = User::whereIn('username', $mentionedUsernames)
            ->where('id', '!=', Auth::id()) // Don't notify the user who wrote the comment
            ->get();


         Notification::send($mentionedUsers, new YouWereMentionedInComment($comment, $this->entry));
    }

    /**
     * [NEW] Prepares to add a comment anchored to a specific text selection.
     * This method is called from Alpine.js in the view.
     *
     * @param string $anchorId The unique ID generated for the <span> tag.
     */
    public function startCommentOnSelection(string $anchorId)
    {
        $this->newCommentAnchorId = $anchorId;
        // Optionally, you can dispatch an event to focus the comment input field
        $this->dispatch('focus-comment-input');
    }


    public function render()
    {
        // [NEW] Fetch all existing tags for autocomplete suggestions
        $allTags = Tag::all()->pluck('name');

        return view('livewire.edit-lab-notebook-entry', [
            'allTags' => $allTags
        ])->layout('layouts.app');
    }
}
