<?php
namespace App\Livewire;

use App\Models\Equipment;
use App\Models\ResearchTeam;
use App\Models\User;
use App\Models\CalibrationLog;
use App\Models\TemperatureLog;
use Mccarlosen\LaravelMpdf\Facades\LaravelMpdf as PDF;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Hekmatinasser\Verta\Verta;
use Endroid\QrCode\Builder\Builder;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\Color\Color; // برای تغییر رنگ
use Endroid\QrCode\Label\Label; // برای اضافه کردن متن زیر QR Code
use Endroid\QrCode\Label\LabelAlignment;
use Endroid\QrCode\Label\Font\OpenSans;
use Endroid\QrCode\RoundBlockSizeMode;
use App\Models\Auditor;
use App\Models\StorageLocation;
use Illuminate\Support\Collection;


class EquipmentManager extends Component
{
    use AuthorizesRequests;
    use WithPagination;
    use WithFileUploads;

    // General Properties
    public string $activeTab = 'equipment';
    public string $search = '';
    protected $paginationTheme = 'bootstrap';


    // --- MERGED PROPERTIES ---

    // Calibration & Original Equipment Properties
    public $teamFilter = '';
    public bool $confirmingEquipmentAction = false;
    public ?Equipment $editingEquipment = null;
    public array $state = []; // This will now hold both calibration and temp-log fields
    public Collection $managedTeams;
    public ?User $responsibleUserForForm = null;
    public ?Equipment $equipmentToDelete = null;
    public bool $confirmingEquipmentDeletion = false;
    public ?Equipment $equipmentForLog = null;
    public bool $confirmingLogCreation = false;
    public array $logState = [];
    public $logAttachment;
    public ?Equipment $equipmentForQrCode = null;
    public bool $showingQrCodeModal = false;
    public ?string $qrCodeDataUri = null;
    public array $selectedForPrinting = [];
    // Properties for data scoping
    public $managedTeamIds;
    // Location Properties
    public $locations;
    public ?StorageLocation $parentLocation = null;
    public bool $confirmingLocationCreation = false;
    public string $newLocationName = '';
    public string $newLocationType = '';
    public array $allowedLocationTypes = [];
    public $locationTeamId = '';
    // [NEW] Properties for Location QR Code
    public ?StorageLocation $locationForQrCode = null;
    public bool $showingLocationQrCodeModal = false;
    public ?string $locationQrCodeDataUri = null;


    // Auditor Properties
    public ?Auditor $editingAuditor = null;
    public bool $confirmingAuditorCreation = false;
    public array $auditorState = [];


    public bool $selectAll = false;

    protected function rules()
    {
        if ($this->activeTab === 'equipment' && $this->confirmingEquipmentAction) {
            return [
                'state.research_team_id' => 'required|exists:research_teams,id',
                'state.name' => 'required|string|max:255',
                'state.model' => 'nullable|string|max:255',
                'state.serial_number' => 'nullable|string|max:255',
                'state.user_defined_id' => 'nullable|string|max:255',
                'state.calibration_interval_days' => 'nullable|integer|min:1',
                'state.last_calibration_date' => 'nullable|string',
                // New Temp Log Rules
                'state.storage_location_id' => 'required|exists:storage_locations,id',
                'state.requires_temp_log' => 'boolean',
                'state.min_temp' => 'nullable|required_if:state.requires_temp_log,true|numeric',
                'state.max_temp' => 'nullable|required_if:state.requires_temp_log,true|numeric|gte:state.min_temp',
            ];
        }
        if ($this->activeTab === 'locations') {
            return [
                'newLocationName' => 'required|string|max:255',
                'newLocationType' => 'required|in:building,room',
            ];
        }
        if ($this->activeTab === 'auditors') {
            $rules = [
                'auditorState.name' => 'required|string|max:255',
                'auditorState.is_active' => 'boolean',
            ];
            $rules['auditorState.pin'] = $this->editingAuditor ? 'nullable|digits:4' : 'required|digits:4';
            return $rules;
        }
        return [];
    }
    /**
     * [NEW & REFACTORED] A helper function to get the full team models the user is authorized to manage for this module.
     */
    private function getAuthorizedTeamIds(): Collection
    {
        $user = Auth::user();

        if ($user->hasRole('super_admin') || $user->hasRole('institution_admin')) {
            return ResearchTeam::where('institution_id', $user->institutionMemberships->first()->institution_id)->get();
        }

        $teamIds = collect();
        foreach ($user->researchTeams as $team) {
            // Check for admin role in team pivot table OR team-specific calibration manager role
            if ($team->pivot->role === 'admin' || $user->hasRole("calibration_manager-Team-{$team->id}")) {
                $teamIds->push($team->id);
            }
        }

        return ResearchTeam::whereIn('id', $teamIds->unique())->get();
    }

    public function mount()
    {
        $this->managedTeams = $this->getAuthorizedTeamIds();
        $this->loadLocations();
    }

    // [NEW] This hook is called whenever the teamFilter property is updated.
    public function updatedTeamFilter()
    {
        $this->resetPage(); // Reset pagination for all tabs
        $this->loadLocations(); // Reload locations based on the new filter
    }

    public function switchTab($tab)
    {
        $this->activeTab = $tab;
        $this->resetErrorBag();
        $this->resetPage();
    }

    public function confirmEquipmentAction(?Equipment $equipment = null)
    {
        $this->resetErrorBag();
        if ($equipment && $equipment->exists) {
            $this->authorize('update', $equipment);
            $this->editingEquipment = $equipment;
            $this->state = $equipment->toArray();
            if ($equipment->last_calibration_date) {
                $this->state['last_calibration_date'] = Verta::instance($equipment->last_calibration_date)->format('Y/m/d');
            }
            $this->responsibleUserForForm = $equipment->responsibleUser;
        } else {
            $this->authorize('create', Equipment::class);
            $this->editingEquipment = null;
            $this->state = [
                'requires_temp_log' => false,
                'min_temp' => '',
                'max_temp' => '',
                'last_calibration_date' => '',
                'calibration_interval_days' => '',
            ];
            $this->responsibleUserForForm = null;
        }
        $this->confirmingEquipmentAction = true;
    }
    /**
     * [NEW] Live update for the last calibration date field.
     */
    public function updatedStateLastCalibrationDate()
    {
        $this->calculateNextCalibrationDate();
    }

    /**
     * [NEW] Live update for the calibration interval field.
     */
    public function updatedStateCalibrationIntervalDays()
    {
        $this->calculateNextCalibrationDate();
    }

    /**
     * [NEW] Calculates the next calibration date based on form inputs.
     */
    private function calculateNextCalibrationDate()
    {
        $lastDateStr = $this->state['last_calibration_date'] ?? null;
        $interval = $this->state['calibration_interval_days'] ?? null;

        if ($lastDateStr && is_numeric($interval) && $interval > 0) {
            try {
                $nextDate = Verta::parse($lastDateStr)->addDays((int)$interval);
                $this->state['next_calibration_date'] = $nextDate->format('Y/m/d');
            } catch (\Exception $e) {
                $this->state['next_calibration_date'] = 'تاریخ نامعتبر';
            }
        } else {
            $this->state['next_calibration_date'] = '';
        }
    }

    /**
     * [NEW] This method is called when the team dropdown changes in the form.
     */
    public function updatedStateResearchTeamId($teamId)
    {
        if ($teamId) {
            $team = ResearchTeam::find($teamId);
            $this->responsibleUserForForm = $team->calibrationManager ?? $team->managers()->first();
        } else {
            $this->responsibleUserForForm = null;
        }
    }

    public function saveEquipment()
    {
        $validatedData = $this->validate();
        $equipmentData = $validatedData['state'];
        $user = Auth::user();

        if (!empty($equipmentData['last_calibration_date']) && !empty($equipmentData['calibration_interval_days'])) {
            $lastDate = Verta::parse($equipmentData['last_calibration_date'])->toCarbon();
            $equipmentData['last_calibration_date'] = $lastDate;
            $equipmentData['next_calibration_date'] = $lastDate->copy()->addDays((int)$equipmentData['calibration_interval_days']);
        } else {
            $equipmentData['last_calibration_date'] = null;
            $equipmentData['next_calibration_date'] = null;
        }

        $equipmentData['last_modified_by'] = $user->id;

        if ($this->editingEquipment) {
            $this->authorize('update', $this->editingEquipment);
            $this->editingEquipment->update($equipmentData);
            session()->flash('message', 'تجهیز با موفقیت به‌روزرسانی شد.');
        } else {
            $this->authorize('create', Equipment::class);

            $manager = ResearchTeam::where('calibration_manager_id', $user->id)->first()?->managers()->first() ?? $user;

            $equipmentData['system_sku'] = $this->generateSku();
            $equipmentData['manager_id'] = $manager->id;

            // [UPDATED] منطق ساده‌تر شده چون انتخاب تیم اجباری است
            $team = ResearchTeam::find($equipmentData['research_team_id']);
            $equipmentData['responsible_user_id'] = $team->calibrationManager?->id ?? $manager->id;

            Equipment::create($equipmentData);
            session()->flash('message', 'تجهیز جدید با موفقیت ایجاد شد.');
        }

        $this->confirmingEquipmentAction = false;
    }

    private function generateSku(): string
    {
        do {
            $sku = 'EQUIP-' . strtoupper(Str::random(8));
        } while (Equipment::where('system_sku', $sku)->exists());
        return $sku;
    }
    /**
     * [NEW] Show the modal for creating a new calibration log.
     */
    public function confirmLogCreation(Equipment $equipment)
    {
        $this->resetErrorBag();
        $this->reset('logAttachment'); // پاک کردن فایل آپلودی قبلی
        $this->equipmentForLog = $equipment;
        $this->logState = [];
        $this->confirmingLogCreation = true;
    }

    /**
     * [NEW] Save the new calibration log and update the equipment's dates.
     */
    public function saveCalibrationLog()
    {
        if (!$this->equipmentForLog) return;

        // [CORRECTED] Validation for the log is now self-contained here.
        $validatedData = $this->validate([
            'logState.calibration_date' => 'required|string',
            'logState.notes' => 'nullable|string',
            'logAttachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240',
        ]);

        $logData = $validatedData['logState'];

        try {
            $calibrationDate = Verta::parse($logData['calibration_date'])->toCarbon();
        } catch (\Exception $e) {
            $this->addError('logState.calibration_date', 'فرمت تاریخ نامعتبر است.');
            return;
        }

        DB::transaction(function () use ($logData, $calibrationDate) {
            $log = $this->equipmentForLog->calibrationLogs()->create([
                'user_id' => Auth::id(),
                'calibration_date' => $calibrationDate,
                'notes' => $logData['notes'],
            ]);

            if ($this->logAttachment) {
                $path = $this->logAttachment->store(
                    'calibration/' . $this->equipmentForLog->system_sku . '/' . $calibrationDate->format('Y-m-d'),
                    'public'
                );
                $log->attachments()->create([
                    'user_id' => Auth::id(),
                    'file_path' => $path,
                    'file_name' => $this->logAttachment->getClientOriginalName(),
                    'file_size' => $this->logAttachment->getSize(),
                    'mime_type' => $this->logAttachment->getMimeType(),
                ]);
            }

            $this->equipmentForLog->last_calibration_date = $calibrationDate;

            if ($this->equipmentForLog->calibration_interval_days) {
                $this->equipmentForLog->next_calibration_date = $calibrationDate->copy()->addDays($this->equipmentForLog->calibration_interval_days);
            }

            $this->equipmentForLog->save();
        });

        $this->confirmingLogCreation = false;
        session()->flash('message', 'سابقه‌ی کالیبراسیون با موفقیت ثبت شد.');
    }
    /**
     * [NEW] Generate a QR code for the selected equipment and show the modal.
     */
    public function showQrCode(Equipment $equipment)
    {
        $this->equipmentForQrCode = $equipment;

        $qrCode = new QrCode(
            data: route('equipment.public.show', $equipment),
            encoding: new Encoding('UTF-8'),
            errorCorrectionLevel: ErrorCorrectionLevel::High,
            size: 300,
            margin: 10,
            roundBlockSizeMode: RoundBlockSizeMode::Margin,
            foregroundColor: new Color(0, 0, 0),
            backgroundColor: new Color(255, 255, 255)
        );

        $writer = new PngWriter();
        $result = $writer->write($qrCode);

        $this->qrCodeDataUri = $result->getDataUri();
        $this->showingQrCodeModal = true;
    }
    /**
     * [NEW] Generate QR codes for all selected equipment and dispatch an event to print them.
     */
    public function printSelectedQrCodes()
    {
        $selectedIds = array_keys(array_filter($this->selectedForPrinting));
        if (empty($selectedIds)) {
            return;
        }

        $equipmentToPrint = Equipment::whereIn('id', $selectedIds)->get();
        $qrCodesData = [];

        foreach ($equipmentToPrint as $equipment) {
            $qrCode = new QrCode(
                data: route('equipment.public.show', $equipment),
                encoding: new Encoding('UTF-8'),
                errorCorrectionLevel: ErrorCorrectionLevel::High,
                size: 150, // Smaller size for printing on a sheet
                margin: 5
            );
            $writer = new PngWriter();
            $result = $writer->write($qrCode);

            $qrCodesData[] = [
                'uri' => $result->getDataUri(),
                'name' => $equipment->name,
                'sku' => $equipment->system_sku,
            ];
        }

        $this->dispatch('print-qr-codes', qrCodes: $qrCodesData);
    }
// New Temp Log Feature Methods
    public function importFromStorageLocations()
    {
        // Scoped to managed teams
//        $storageItems = StorageLocation::whereIn('research_team_id', $this->managedTeamIds)
//            ->whereIn('location_type', ['refrigerator', 'freezer'])->get();
//        $importedCount = 0;
//        foreach ($storageItems as $item) {
//            if (!$item->parent_id) continue;
//            $exists = Equipment::where('name', $item->name)->where('storage_location_id', $item->parent_id)->exists();
//            if (!$exists) {
//                Equipment::create([
//                    'name' => $item->name,
//                    'system_sku' => 'EQUIP-' . strtoupper(Str::random(8)),
//                    'storage_location_id' => $item->parent_id,
//                    'research_team_id' => $item->research_team_id,
//                    'requires_temp_log' => true,
//                    'min_temp' => $item->location_type === 'freezer' ? -20 : 2,
//                    'max_temp' => $item->location_type === 'freezer' ? -10 : 8,
//                ]);
//                $importedCount++;
//            }
//        }
        $authorizedTeamIds = $this->getAuthorizedTeamIds()->pluck('id');
        $storageItems = StorageLocation::whereIn('research_team_id', $authorizedTeamIds)
            ->whereIn('location_type', ['refrigerator', 'freezer'])->get();
        $importedCount = 0;
        foreach ($storageItems as $item) {
            if (!$item->parent_id) continue;
            $exists = Equipment::where('name', $item->name)->where('storage_location_id', $item->parent_id)->exists();
            if (!$exists) {
                Equipment::create([
                    'name' => $item->name,
                    'system_sku' => 'EQUIP-' . strtoupper(Str::random(8)),
                    'storage_location_id' => $item->parent_id,
                    'research_team_id' => $item->research_team_id,
                    'requires_temp_log' => true,
                    'min_temp' => $item->location_type === 'freezer' ? -20 : 2,
                    'max_temp' => $item->location_type === 'freezer' ? -10 : 8,
                ]);
                $importedCount++;
            }
        }
        session()->flash('message', "$importedCount تجهیز جدید با موفقیت از انبار وارد شد.");
    }

    // New Location Management Methods
    public function loadLocations()
    {
//        $this->locations = StorageLocation::whereIn('research_team_id', $this->managedTeamIds)
//            ->whereNull('parent_id')
//            ->with('children')
//            ->get();
        $authorizedTeamIds = $this->getAuthorizedTeamIds()->pluck('id');
        $this->locations = StorageLocation::whereIn('research_team_id', $authorizedTeamIds)
            ->whereNull('parent_id')
            ->with('children')
            ->get();
    }
    public function confirmLocationCreation($parentId = null)
    {
        $this->resetErrorBag();
        $this->reset(['newLocationName', 'newLocationType', 'parentLocation', 'allowedLocationTypes', 'locationTeamId']);

        if ($parentId) {
            $this->parentLocation = StorageLocation::find($parentId);
            $this->allowedLocationTypes = ['room'];
            $this->newLocationType = 'room'; // [FIXED] Explicitly set the type
        } else {
            $this->allowedLocationTypes = ['building'];
            $this->newLocationType = 'building';
        }
        $this->confirmingLocationCreation = true;
    }

    /**
     * [NEW] Generates a unique location code based on the team's prefix.
     */
    private function generateLocationCode(ResearchTeam $team): string
    {
        do {
            $randomNumber = str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT);
            $code = ($team->research_teams_code ?? 'LOC') . '-' . $randomNumber;
        } while (StorageLocation::where('location_code', $code)->exists());

        return $code;
    }
    public function saveLocation()
    {
//        $this->validate([
//            'newLocationName' => 'required|string|max:255',
//            'newLocationType' => 'required|in:' . implode(',', $this->allowedLocationTypes),
//        ]);
//
//        StorageLocation::create([
//            'name' => $this->newLocationName,
//            'location_type' => $this->newLocationType,
//            'parent_id' => $this->parentLocation?->id,
//            'uuid' => Str::uuid(),
//        ]);
//
//        $this->confirmingLocationCreation = false;
//        $this->loadLocations();
//        session()->flash('message', 'مکان جدید با موفقیت ایجاد شد.');
//        $validatedData = $this->validate();
//        $teamId = $this->parentLocation ? $this->parentLocation->research_team_id : $validatedData['locationTeamId'];
//
//        StorageLocation::create([
//            'name' => $validatedData['newLocationName'],
//            'location_type' => $validatedData['newLocationType'],
//            'parent_id' => $this->parentLocation?->id,
//            'research_team_id' => $teamId,
//            'uuid' => Str::uuid(),
//        ]);
//        $this->confirmingLocationCreation = false;
//        $this->loadLocations();
//        session()->flash('message', 'مکان جدید با موفقیت ایجاد شد.');
        $this->validate(); // Run validation. It will throw an exception on failure.

        // [FIXED] Use the public property directly after validation passes.
        $teamId = $this->parentLocation
            ? $this->parentLocation->research_team_id
            : $this->locationTeamId;
        $team = ResearchTeam::find($teamId);
        StorageLocation::create([
            'name' => $this->newLocationName,
            'location_type' => $this->newLocationType,
            'parent_id' => $this->parentLocation?->id,
            'research_team_id' => $teamId,
            'uuid' => Str::uuid(),
            'location_code' => $this->generateLocationCode($team),
        ]);

        $this->confirmingLocationCreation = false;
        $this->loadLocations();
        session()->flash('message', 'مکان جدید با موفقیت ایجاد شد.');
    }

    /**
     * [NEW] Generate a QR code for the selected location and show the modal.
     */
    public function showLocationQrCode(StorageLocation $location)
    {
        $this->locationForQrCode = $location;
        $qrCode = new QrCode(
            data: route('public.temp-log', $location->uuid),
            size: 300,
            margin: 10
        );
        $writer = new PngWriter();
        $result = $writer->write($qrCode);
        $this->locationQrCodeDataUri = $result->getDataUri();
        $this->showingLocationQrCodeModal = true;
    }

    /**
     * [NEW] Prepares a single location QR code for printing.
     */
    public function printLocationQrCode()
    {
        if (!$this->locationForQrCode) return;

        $qrCode = new QrCode(
            data: route('public.temp-log', $this->locationForQrCode->uuid),
            size: 150,
            margin: 5
        );
        $writer = new PngWriter();
        $result = $writer->write($qrCode);
        $qrCodeData = [
            'uri' => $result->getDataUri(),
            'name' => $this->locationForQrCode->parent->name . ' / ' . $this->locationForQrCode->name,
            'sku' => $this->locationForQrCode->location_code,
        ];
        $this->dispatch('print-qr-codes', qrCodes: [$qrCodeData]);
    }

    // --- AUDITOR METHODS ---
    public function confirmAuditorCreation(?Auditor $auditor = null)
    {
        $this->resetErrorBag();
        if ($auditor && $auditor->exists) {
            $this->editingAuditor = $auditor;
            $this->auditorState = $auditor->toArray();
            $this->auditorState['pin'] = '';
        } else {
            $this->editingAuditor = null;
            $this->auditorState = ['is_active' => true, 'research_team_id' => ''];
        }
        $this->confirmingAuditorCreation = true;
    }

    public function saveAuditor()
    {
        $validatedData = $this->validate();
//        $auditorData = $validatedData['auditorState'];
        $auditorData = $this->auditorState;

        if (empty($auditorData['pin'])) {
            unset($auditorData['pin']);
        }

        if ($this->editingAuditor) {
            $this->editingAuditor->update($auditorData);
            session()->flash('message', 'بازرس با موفقیت به‌روزرسانی شد.');
        } else {
            Auditor::create($auditorData);
            session()->flash('message', 'بازرس جدید با موفقیت ایجاد شد.');
        }

        $this->confirmingAuditorCreation = false;
    }
    /**
     * [NEW] Prepares a single QR code for printing by dispatching an event.
     */
    public function printSingleQrCode()
    {
        if (!$this->equipmentForQrCode) {
            return;
        }

        $qrCode = new QrCode(
            data: route('equipment.public.show', $this->equipmentForQrCode),
            size: 150,
            margin: 5
        );
        $writer = new PngWriter();
        $result = $writer->write($qrCode);

        $qrCodeData = [
            'uri' => $result->getDataUri(),
            'name' => $this->equipmentForQrCode->name,
            'sku' => $this->equipmentForQrCode->system_sku,
        ];

        // We reuse the same event as the bulk print for simplicity
        $this->dispatch('print-qr-codes', qrCodes: [$qrCodeData]);
    }
    /**
     * [NEW] A private helper method to get the base query for equipment.
     * This avoids repeating the query logic.
     */
    private function getEquipmentQuery()
    {
//        $query = Equipment::whereIn('research_team_id', $this->managedTeamIds);
//
//        if (!empty($this->search)) {
//            $query->where(fn($q) => $q->where('name', 'like', '%' . $this->search . '%'));
//        }
//        if (!empty($this->teamFilter)) {
//            $query->where('research_team_id', $this->teamFilter);
//        }
//        return $query;
        $authorizedTeamIds = $this->getAuthorizedTeamIds()->pluck('id');

        $query = Equipment::whereIn('research_team_id', $authorizedTeamIds);

        if (!empty($this->search)) {
            $query->where(fn($q) => $q->where('name', 'like', '%' . $this->search . '%'));
        }
        if (!empty($this->teamFilter)) {
            if (in_array($this->teamFilter, $authorizedTeamIds->toArray())) {
                $query->where('research_team_id', $this->teamFilter);
            }
        }
        return $query;
    }

    /**
     * [NEW] This is a Livewire lifecycle hook that runs when the "selectAll" property is updated.
     */
    public function updatedSelectAll($value)
    {
        if ($value) {
            // Get IDs of all equipment that match the current filters
            $allEquipmentIds = $this->getEquipmentQuery()->pluck('id')->map(fn ($id) => (string) $id);
            // Select all of them
            foreach ($allEquipmentIds as $id) {
                $this->selectedForPrinting[$id] = true;
            }
        } else {
            // Deselect all
            $this->selectedForPrinting = [];
        }
    }

    /**
     * [NEW] Show the confirmation modal for deleting an equipment.
     */
    public function confirmEquipmentDeletion(Equipment $equipment)
    {
        $this->authorize('delete', $equipment);
        $this->equipmentToDelete = $equipment;
        $this->confirmingEquipmentDeletion = true;
    }

    /**
     * [NEW] Delete the selected equipment.
     */
    public function deleteEquipment()
    {
        if (!$this->equipmentToDelete) {
            return;
        }

        $this->authorize('delete', $this->equipmentToDelete);
        $this->equipmentToDelete->delete();

        $this->confirmingEquipmentDeletion = false;
        session()->flash('message', 'تجهیز با موفقیت حذف شد.');
    }


    public function render()
    {
        $this->authorize('viewAny', Equipment::class);

        $authorizedTeamIds = $this->getAuthorizedTeamIds()->pluck('id');

        // Scoped Equipment Query
        $equipmentQuery = Equipment::whereIn('research_team_id', $authorizedTeamIds);
        if (!empty($this->search)) {
            $equipmentQuery->where(fn($q) => $q->where('name', 'like', '%' . $this->search . '%'));
        }
        if (!empty($this->teamFilter)) {
            if (in_array($this->teamFilter, $authorizedTeamIds->toArray())) {
                $equipmentQuery->where('research_team_id', $this->teamFilter);
            }
        }

        // Scoped Auditors Query
        $auditorsQuery = Auditor::whereIn('research_team_id', $authorizedTeamIds);
        if (!empty($this->teamFilter)) {
            if (in_array($this->teamFilter, $authorizedTeamIds->toArray())) {
                $auditorsQuery->where('research_team_id', $this->teamFilter);
            }
        }
        $auditors = $auditorsQuery->latest()->paginate(10);


        // Scoped Rooms Query
        $rooms = StorageLocation::where('location_type', 'room')
            ->whereIn('research_team_id', $authorizedTeamIds)
            ->with('parent')
            ->get();

//        $auditors = Auditor::whereIn('research_team_id', $authorizedTeamIds)->latest()->paginate(10);
//
//        $rooms = StorageLocation::where('location_type', 'room')
//            ->whereIn('research_team_id', $authorizedTeamIds)
//            ->with('parent')
//            ->get();

        return view('livewire.equipment-manager', [
            'equipmentList' => $equipmentQuery->with(['researchTeam', 'responsibleUser', 'location.parent'])->latest()->paginate(10),
            'auditors' => $auditors,
            'rooms' => $rooms,
            'managedTeams' => $this->managedTeams,
        ])->layout('layouts.app');
    }
}
