<?php

namespace App\Livewire;

use App\Models\PurchaseRequest;
use App\Models\SupplierResponse;
use App\Models\Supplier;
use App\Models\Payment;
use App\Models\PaymentNote;
use App\Services\PaymentService;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Livewire\Component;
use Livewire\WithPagination;
use Hekmatinasser\Verta\Verta;

class FinancialManager extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    // Properties for History Filters
    public $filterSupplier = '';
    public $filterStatus = '';
    public $filterStartDate = '';
    public $filterEndDate = '';

    public string $activeTab = 'pending';

    // Properties for Invoice Details Modal
    public ?SupplierResponse $viewingInvoice = null;
    public bool $confirmingInvoiceDetails = false;
    public $winningItemsForInvoice = [];

    // Properties for Budget Check & Confirmation
    public ?SupplierResponse $invoiceForPayment = null;
    public bool $confirmingOverBudgetPayment = false;
    public array $budgetDetails = [];
    // Properties for Notes
    public string $newNote = '';


    public function switchTab($tab)
    {
        $this->resetPage();
        $this->activeTab = $tab;
    }

    /**
     * Show the modal with full details of the invoice.
     */
    public function showInvoiceDetails(SupplierResponse $response)
    {
        $this->viewingInvoice = $response->load([
            'supplier',
            'purchaseRequest.project',
            'payments.notes.user',
            'payments.logs.user',
        ]);

        // Fetch ONLY the items from this response that were marked as the final choice.
        $this->winningItemsForInvoice = $this->viewingInvoice->purchaseRequest->items()
            ->where('status', 'offer_accepted')
            ->whereHas('finalResponseItem.response', function ($query) use ($response) {
                $query->where('id', $response->id);
            })
            ->with('finalResponseItem')
            ->get();

        // Calculate and prepare budget details for the modal
        $this->budgetDetails = [];
        $project = $this->viewingInvoice->purchaseRequest->project;
        if ($project && $project->budget) {
            $spentAmount = Payment::where('status', 'successful')
                ->whereHasMorph('payable', [SupplierResponse::class], function ($query) use ($project) {
                    $query->whereHas('purchaseRequest', fn($q) => $q->where('project_id', $project->id));
                })->sum('amount');

            $this->budgetDetails = [
                'total' => $project->budget,
                'spent' => $spentAmount,
                'remaining' => $project->budget - $spentAmount,
            ];
        }

        $this->confirmingInvoiceDetails = true;
    }

    /**
     * Initiates the payment process for a selected supplier response.
     */
    public function initiatePayment(SupplierResponse $response)
    {
        $this->invoiceForPayment = $response; // Store the response for payment

        $invoiceAmount = $this->calculateWinningItemsAmount($response);

        $project = $response->purchaseRequest->project;

        // Check for budget overrun if a project with a budget is associated
        if ($project && $project->budget) {
            $spentAmount = Payment::where('status', 'successful')
                ->whereHasMorph('payable', [SupplierResponse::class], fn($q) => $q->whereHas('purchaseRequest', fn($sq) => $sq->where('project_id', $project->id)))
                ->sum('amount');

            $remainingBudget = $project->budget - $spentAmount;

            if ($invoiceAmount > $remainingBudget) {
                $this->confirmingOverBudgetPayment = true; // Show confirmation modal
                return;
            }
        }

        // If no budget overrun, proceed directly to payment
        $this->proceedWithPayment();
    }

    /**
     * Retries a pending or failed payment from the history tab.
     */
    public function retryPayment(Payment $payment, PaymentService $paymentService)
    {
        if (!in_array($payment->status, ['pending', 'failed'])) {
            session()->flash('error', 'فقط تراکنش‌های ناموفق یا در انتظار را می‌توان مجددا پرداخت کرد.');
            return;
        }

        $this->authorize('confirmResponses', $payment->payable->purchaseRequest);

        $redirectUrl = $paymentService->requestPayment($payment);

        if ($redirectUrl) {
            return $this->redirect($redirectUrl, navigate: false);
        } else {
            $payment->update(['status' => 'failed']);
            session()->flash('error', 'ارتباط با درگاه پرداخت برقرار نشد. لطفاً دوباره تلاش کنید.');
        }
    }

    /**
     * This method is called after the user confirms the over-budget warning.
     */
    public function forcePayment()
    {
        $this->confirmingOverBudgetPayment = false;
        $this->proceedWithPayment();
    }

    /**
     * Contains the actual payment logic.
     */
    public function proceedWithPayment()
    {
        if (!$this->invoiceForPayment) return;

        $response = $this->invoiceForPayment;
        $paymentService = app(PaymentService::class);

        $totalAmount = $this->calculateWinningItemsAmount($response);

        if ($totalAmount <= 0) {
            session()->flash('error', 'مبلغ قابل پرداخت صفر است.');
            return;
        }

        $payment = Payment::create([
            'uuid' => Str::uuid(),
            'user_id' => Auth::id(),
            'payable_id' => $response->id,
            'payable_type' => SupplierResponse::class,
            'amount' => $totalAmount,
            'gateway_name' => 'zarinpal',
            'status' => 'pending',
        ]);

        $redirectUrl = $paymentService->requestPayment($payment);

        if ($redirectUrl) {
            return $this->redirect($redirectUrl, navigate: false);
        } else {
            $payment->update(['status' => 'failed']);
            session()->flash('error', 'ارتباط با درگاه پرداخت برقرار نشد.');
        }
    }

    /**
     * Helper method to calculate the total amount of winning items for a specific supplier response.
     */
    private function calculateWinningItemsAmount(SupplierResponse $response): float
    {
        $winningItems = $response->purchaseRequest->items()
            ->where('status', 'offer_accepted')
            ->whereHas('finalResponseItem.response', function ($query) use ($response) {
                $query->where('id', $response->id);
            })
            ->with('finalResponseItem')
            ->get();

        return $winningItems->sum(function ($item) {
            if ($item->finalResponseItem) {
                return $item->finalResponseItem->supplied_quantity * $item->finalResponseItem->price_per_unit;
            }
            return 0;
        });
    }


    /**
     * A helper method to get the filtered query for payment history.
     */
    private function getFilteredHistoryQuery()
    {
        $user = Auth::user();
        $managedTeamIds = $user->researchTeams()->wherePivot('role', 'admin')->pluck('research_teams.id');

        $historyQuery = Payment::query()->with([
            'user',
            'payable.supplier',
            'payable.purchaseRequest.project'
        ]);

        if (!$user->hasRole('purchasing_manager')) {
            // [BUGFIX] Passed the $user variable to the closure's scope.
            $historyQuery->whereHasMorph('payable', [SupplierResponse::class], function ($query) use ($managedTeamIds, $user) {
                $query->whereHas('purchaseRequest', function (Builder $subQuery) use ($managedTeamIds, $user) {
                    $subQuery->where(function (Builder $q) use ($managedTeamIds, $user) {
                        $q->whereHas('project.researchTeam', function ($teamQuery) use ($managedTeamIds) {
                            $teamQuery->whereIn('research_teams.id', $managedTeamIds);
                        })->orWhere(function (Builder $orQ) use ($user) {
                            $orQ->whereNull('project_id')->where('institution_id', $user->currentTeam->id);
                        });
                    });
                });
            });
        }

        // Apply filters
        $historyQuery->when($this->filterSupplier, fn($q) => $q->whereHasMorph('payable', [SupplierResponse::class], fn($sq) => $sq->where('supplier_id', $this->filterSupplier)));
        $historyQuery->when($this->filterStatus, fn($q) => $q->where('status', $this->filterStatus));
        $historyQuery->when($this->filterStartDate, fn($q) => $q->where('created_at', '>=', Verta::parse($this->filterStartDate)->toCarbon()->startOfDay()));
        $historyQuery->when($this->filterEndDate, fn($q) => $q->where('created_at', '<=', Verta::parse($this->filterEndDate)->toCarbon()->endOfDay()));

        return $historyQuery;
    }

    /**
     * Export the filtered history to a CSV file.
     */
    public function exportHistoryCsv()
    {
        $logs = $this->getFilteredHistoryQuery()->get();
        $fileName = 'financial-history.csv';

        $headers = ["Content-type" => "text/csv", "Content-Disposition" => "attachment; filename=$fileName"];

        $callback = function() use ($logs) {
            $file = fopen('php://output', 'w');
            fputcsv($file, ['کد درخواست', 'تامین‌کننده', 'مبلغ (تومان)', 'وضعیت', 'تاریخ پرداخت', 'کد رهگیری']);
            foreach ($logs as $log) {
                fputcsv($file, [
                    $log->payable?->purchaseRequest?->request_code ?? '-',
                    $log->payable?->supplier?->company_name ?? 'N/A',
                    $log->amount,
                    $log->status,
                    $log->paid_at ? verta($log->paid_at)->format('Y/m/d H:i') : '-',
                    $log->meta['ref_id'] ?? $log->transaction_id ?? '-'
                ]);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Add a new internal note to a payment.
     */
    public function addNote()
    {
        $this->validate(['newNote' => 'required|string']);

        $payment = $this->viewingInvoice->payments()->first();

        if ($payment) {
            $payment->notes()->create([
                'user_id' => Auth::id(),
                'body' => $this->newNote,
            ]);
            $this->newNote = '';
            $this->showInvoiceDetails($this->viewingInvoice->fresh());
        }
    }

    public function render()
    {
        $user = Auth::user();
        $managedTeamIds = $user->researchTeams()->wherePivot('role', 'admin')->pluck('research_teams.id');

        $pendingInvoices = collect();
        $paymentHistory = collect();
        $suppliers = collect();


        if ($this->activeTab === 'pending') {
            $query = PurchaseRequest::where('status', 'final_approval');

            // Correctly handle permissions for non-purchasing managers
            // to include both their project-specific requests AND general institutional requests.
            if (!$user->hasRole('purchasing_manager')) {
                $query->where(function (Builder $subQuery) use ($managedTeamIds, $user) {
                    // Condition 1: Request is part of a project managed by the user's teams.
                    $subQuery->whereHas('project.researchTeam', function ($teamQuery) use ($managedTeamIds) {
                        $teamQuery->whereIn('research_teams.id', $managedTeamIds);
                    })
                        // Condition 2: Request is a general request (no project) from the user's institution.
                        ->orWhere(function (Builder $orSubQuery) use ($user) {
                            $orSubQuery->whereNull('project_id')
                                ->where('institution_id', $user->currentTeam->id);
                        });
                });
            }

            // Eager load all necessary relationships.
            $approvedRequests = $query->with([
                'items' => fn($q) => $q->where('status', 'offer_accepted'),
                'items.finalResponseItem.response.supplier',
                'items.finalResponseItem.response.payments',
                'project.researchTeam'
            ])->get();

            // Get all unique winning invoices from the approved requests.
            $allWinningInvoices = $approvedRequests->flatMap(function ($request) {
                return $request->items->pluck('finalResponseItem.response');
            })->filter()->unique('id');

            // Filter the collection to exclude invoices that have a successful payment.
            $pendingInvoices = $allWinningInvoices->filter(function ($response) {
                return !$response->payments->where('status', 'successful')->count();
            });
        }

        if ($this->activeTab === 'history') {
            $paymentHistory = $this->getFilteredHistoryQuery()->latest()->paginate(15);
            $suppliers = Supplier::whereHas('responses.payments')->get();
        }

        return view('livewire.financial-manager', [
            'pendingInvoices' => $pendingInvoices,
            'paymentHistory' => $paymentHistory,
            'suppliers' => $suppliers,
        ])->layout('layouts.app');
    }
}
