<?php

namespace App\Livewire;

use App\Models\Payment;
use App\Models\PurchaseRequest;
use App\Models\SupplierResponse;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class FinancialSummaryWidget extends Component
{
    public int $pendingInvoicesCount = 0;
    public float $pendingInvoicesAmount = 0;
    public float $paidLast30DaysAmount = 0;
    public bool $canView = false;

    public function mount()
    {
        $user = Auth::user();

        // Check if the user is authorized to see this widget at all
        if (!$user->hasRole('purchasing_manager') && !$user->researchTeams()->wherePivot('role', 'admin')->exists()) {
            $this->canView = false;
            return;
        }
        $this->canView = true;

        $managedTeamIds = $user->researchTeams()->wherePivot('role', 'admin')->pluck('research_teams.id');

        // --- Calculate Pending Invoices ---
        $pendingQuery = PurchaseRequest::where('status', 'final_approval');
        if (!$user->hasRole('purchasing_manager')) {
            $pendingQuery->whereHas('project.researchTeam', function ($q) use ($managedTeamIds) {
                $q->whereIn('research_teams.id', $managedTeamIds);
            });
        }
        $approvedRequests = $pendingQuery->with(['items.finalResponseItem.response'])->get();
        $pendingInvoices = $approvedRequests->flatMap(function ($request) {
            return $request->items->pluck('finalResponseItem.response')->filter()->unique('id');
        });

        $this->pendingInvoicesCount = $pendingInvoices->count();
        $this->pendingInvoicesAmount = $pendingInvoices->sum(function ($response) {
            return $response->items->sum(fn($item) => $item->supplied_quantity * $item->price_per_unit);
        });


        // --- Calculate Amount Paid in Last 30 Days ---
        $paidQuery = Payment::where('status', 'successful')->where('paid_at', '>=', now()->subDays(30));
        if (!$user->hasRole('purchasing_manager')) {
            $paidQuery->whereHasMorph('payable', [SupplierResponse::class], function ($query) use ($managedTeamIds) {
                $query->whereHas('purchaseRequest.project', function ($subQuery) use ($managedTeamIds) {
                    $subQuery->whereIn('research_team_id', $managedTeamIds);
                });
            });
        }
        $this->paidLast30DaysAmount = $paidQuery->sum('amount');
    }


    public function render()
    {
        return view('livewire.financial-summary-widget');
    }
}
