<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\Task;
use App\Models\Equipment;
use App\Models\InventoryItem;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class GlobalSearch extends Component
{
    public string $searchQuery = '';
    public array $searchResults = [];
    public bool $showResults = false;

    /**
     * This is a Livewire lifecycle hook that runs whenever the searchQuery property is updated.
     */
    public function updatedSearchQuery()
    {
        // Only start searching after 3 characters have been typed
        if (strlen($this->searchQuery) < 3) {
            $this->searchResults = [];
            $this->showResults = false;
            return;
        }

        $this->showResults = true;
        $this->searchResults = [];

        $user = Auth::user();
        $teamIds = $user->researchTeams()->pluck('research_teams.id');

        // --- Search Projects ---
        $projects = Project::whereIn('research_team_id', $teamIds)
            ->where(function ($query) {
                $query->where('title', 'like', '%' . $this->searchQuery . '%')
                    ->orWhere('sku', 'like', '%' . $this->searchQuery . '%');
            })
            ->take(5)
            ->get();

        if ($projects->isNotEmpty()) {
            $this->searchResults['طرح‌ها'] = $projects->map(function ($project) {
                return [
                    'title' => $project->title,
                    'url' => route('projects.show', $project),
                    'subtitle' => $project->sku
                ];
            });
        }

        // --- Search Tasks ---
        // Note: This assumes Tasks are linked to Projects that have a team.
        $tasks = Task::whereHas('project', function ($query) use ($teamIds) {
            $query->whereIn('research_team_id', $teamIds);
        })
            ->where('title', 'like', '%' . $this->searchQuery . '%')
            ->with('project')
            ->take(5)
            ->get();

        if ($tasks->isNotEmpty()) {
            $this->searchResults['وظایف'] = $tasks->map(function ($task) {
                return [
                    'title' => $task->title,
                    'url' => route('projects.show', $task->project), // Links to the project detail page
                    'subtitle' => 'در طرح: ' . $task->project->title
                ];
            });
        }

        // --- Search Equipment ---
        $equipment = Equipment::whereIn('research_team_id', $teamIds)
            ->where(function ($query) {
                $query->where('name', 'like', '%' . $this->searchQuery . '%')
                    ->orWhere('system_sku', 'like', '%' . $this->searchQuery . '%');
            })
            ->take(5)
            ->get();

        if ($equipment->isNotEmpty()) {
            $this->searchResults['تجهیزات'] = $equipment->map(function ($item) {
                return [
                    'title' => $item->name,
                    'url' => route('equipment.index'), // Assumes you have an equipment index route
                    'subtitle' => $item->system_sku
                ];
            });
        }

        // --- Search Inventory Items ---
        $inventoryItems = InventoryItem::whereIn('research_team_id', $teamIds)
            ->where(function ($query) {
                $query->where('name', 'like', '%' . $this->searchQuery . '%')
                    ->orWhere('sku', 'like', '%' . $this->searchQuery . '%');
            })
            ->take(5)
            ->get();

        if ($inventoryItems->isNotEmpty()) {
            $this->searchResults['مواد انبار'] = $inventoryItems->map(function ($item) {
                return [
                    'title' => $item->name,
                    'url' => route('inventory.index'), // Assumes you have an inventory index route
                    'subtitle' => 'موجودی: ' . $item->quantity . ' ' . $item->unit
                ];
            });
        }
    }

    /**
     * Resets the search state, typically after a result is clicked.
     */
    public function resetSearch()
    {
        $this->searchQuery = '';
        $this->searchResults = [];
        $this->showResults = false;
    }

    public function render()
    {
        return view('livewire.global-search');
    }
}
