<?php

namespace App\Livewire;

use App\Models\Institution;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str; // برای تولید رشته تصادفی
use Livewire\Component;

class InstitutionManager extends Component
{
    // Properties for the "Create" modal form
    public $name, $type, $address, $phone, $website;
    public $confirmingInstitutionAddition = false;

    // Properties for the "Edit" modal form
    public ?Institution $editingInstitution;
    public array $editingState = [];
    public $confirmingInstitutionEdit = false;

    // Properties for the "Delete" modal form
    public $institutionIdToDelete;
    public $confirmingInstitutionDeletion = false;


    protected function rules()
    {
        // ما اعتبارسنجی کد را از اینجا حذف می‌کنیم چون دیگر توسط کاربر وارد نمی‌شود
        return [
            'editingState.name' => 'required|string|max:255',
            'editingState.type' => 'nullable|string|max:255',
            'editingState.address' => 'nullable|string',
            'editingState.phone' => 'nullable|string|max:50',
            'editingState.website' => 'nullable|url|max:255',
        ];
    }

    public function confirmInstitutionAddition()
    {
        $this->resetErrorBag();
        $this->reset(['name', 'type', 'address', 'phone', 'website']);
        $this->confirmingInstitutionAddition = true;
    }

    public function saveInstitution()
    {
        $this->validate([
            'name' => 'required|string|max:255|unique:institutions',
            'type' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'phone' => 'nullable|string|max:50',
            'website' => 'nullable|url|max:255',
        ]);

        // تولید کد منحصر به فرد
        $generatedCode = '';
        do {
            $generatedCode = strtoupper(Str::random(3));
        } while (Institution::where('code', $generatedCode)->exists());


        Institution::create([
            'name' => $this->name,
            'code' => $generatedCode, // استفاده از کد تولید شده
            'type' => $this->type,
            'address' => $this->address,
            'phone' => $this->phone,
            'website' => $this->website,
            'contact_person_id' => Auth::id(),
        ]);

        $this->confirmingInstitutionAddition = false;
        session()->flash('message', 'نهاد جدید با کد ' . $generatedCode . ' با موفقیت ایجاد شد.');
    }

    public function confirmInstitutionEdit(Institution $institution)
    {
        $this->resetErrorBag();
        $this->editingInstitution = $institution;
        $this->editingState = $institution->toArray();
        $this->confirmingInstitutionEdit = true;
    }

    public function updateInstitution()
    {
        $this->validate();

        $this->editingInstitution->update($this->editingState);

        $this->confirmingInstitutionEdit = false;
        session()->flash('message', 'نهاد با موفقیت به‌روزرسانی شد.');
    }

    public function confirmInstitutionDeletion($id)
    {
        $this->institutionIdToDelete = $id;
        $this->confirmingInstitutionDeletion = true;
    }

    public function deleteInstitution()
    {
        Institution::findOrFail($this->institutionIdToDelete)->delete();

        $this->confirmingInstitutionDeletion = false;
        session()->flash('message', 'نهاد با موفقیت حذف شد.');
    }


    public function render()
    {
        $institutions = Institution::all();

        return view('livewire.institution-manager', [
            'institutions' => $institutions,
        ])->layout('layouts.app');
    }
}
