<?php

namespace App\Livewire;

use App\Models\Institution;
use App\Models\InstitutionRequest;
use App\Models\User;
use App\Models\ResearchTeam; // [NEW]
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB; // [NEW]
use Illuminate\Support\Str;
use Livewire\Component;

class InstitutionOnboarding extends Component
{
    // Mode can be 'create' or 'join'
    public $mode = 'create';

    // Properties for creating a new institution
    // Properties for the multi-step creation modal
    public $confirmingInstitutionCreation = false;
    public $creationStep = 1;
    public $create_name = '';
    public $create_type = '';
    public $create_address = '';
    public $create_phone = '';
    public $create_website = '';
    public $personnel_count;
    public $field_of_work;
    public $desired_teams_count;
    public $contact_person_name;

    // Properties for joining an existing institution
    public string $join_search = '';
    public $join_searchResults = [];
    public ?Institution $selectedInstitution = null;
    public $requestSent = false;

    // Properties to manage user state
    public $hasPendingRequest = false;
    public $pendingRequestTrackingCode = '';

    public function mount()
    {
        // Check if the user already has a pending or approved request/membership
        $user = Auth::user();
        $pendingRequest = $user->institutionRequests()->where('status', 'pending')->first();
        if ($pendingRequest) {
            $this->hasPendingRequest = true;
            $this->pendingRequestTrackingCode = $pendingRequest->tracking_code;
        }
    }
    /**
     * Validate the first step (institution name) and open the multi-step modal.
     */
    public function confirmInstitutionCreation()
    {
        $this->validate(['create_name' => 'required|string|max:255|unique:institutions,name|unique:institution_requests,name']);
        $this->creationStep = 1;
        $this->confirmingInstitutionCreation = true;
    }

    /**
     * Validate the current step and move to the next one.
     */
    /**
     * Validate the current step and move to the next one.
     */
    public function nextStep()
    {
        if ($this->creationStep === 1) {
            $this->validate([
                'create_type' => 'required|string|max:255',
                'field_of_work' => 'required|string|max:255',
            ]);
        } elseif ($this->creationStep === 2) {
            $this->validate([
                'personnel_count' => 'required|integer|min:1',
                'desired_teams_count' => 'required|integer|min:1',
            ]);
        }
        $this->creationStep++;
    }

    /**
     * Go back to the previous step.
     */
    public function previousStep()
    {
        $this->creationStep--;
    }

    /**
     * Final step: validate all data and submit the creation request.
     */
    /**
     * Final step: validate all data and submit the creation request.
     */
    public function submitInstitutionCreation()
    {
        $validatedData = $this->validate([
            'create_name' => 'required|string|max:255|unique:institutions,name|unique:institution_requests,name',
            'create_type' => 'required|string|max:255',
            'field_of_work' => 'required|string|max:255',
            'personnel_count' => 'required|integer|min:1',
            'desired_teams_count' => 'required|integer|min:1',
            'contact_person_name' => 'required|string|max:255',
            'create_phone' => 'required|string|max:50',
            'create_address' => 'required|string',
            'create_website' => 'nullable|url|max:255',
        ]);

        do {
            $trackingCode = 'BENCH-' . str_pad(rand(0, 9999999999), 10, '0', STR_PAD_LEFT);
        } while (InstitutionRequest::where('tracking_code', $trackingCode)->exists());

        $request = InstitutionRequest::create([
            'user_id' => Auth::id(),
            'tracking_code' => $trackingCode,
            'name' => $validatedData['create_name'],
            'type' => $validatedData['create_type'],
            'address' => $validatedData['create_address'],
            'phone' => $validatedData['create_phone'],
            'website' => $validatedData['create_website'],
            'personnel_count' => $validatedData['personnel_count'],
            'field_of_work' => $validatedData['field_of_work'],
            'desired_teams_count' => $validatedData['desired_teams_count'],
            'contact_person_name' => $validatedData['contact_person_name'],
            'status' => 'pending',
        ]);

        $this->confirmingInstitutionCreation = false;
        $this->hasPendingRequest = true;
        $this->pendingRequestTrackingCode = $request->tracking_code;
        session()->flash('onboarding_message', 'درخواست ایجاد سازمان شما با موفقیت ثبت شد.');
    }
    /**
     * Live search for existing institutions.
     */
    public function updatedJoinSearch()
    {
        if (strlen($this->join_search) < 2) {
            $this->join_searchResults = [];
            return;
        }
        // --- منطق جدید: ما تعداد اعضای تایید شده را نیز شمارش می‌کنیم ---
        $this->join_searchResults = Institution::where('status', 'approved') // فقط سازمان‌های تایید شده
        ->where(function ($query) {
            $query->where('name', 'like', '%' . $this->join_search . '%')
                ->orWhere('code', 'like', '%' . $this->join_search . '%');
        })
            ->withCount(['memberships' => function ($query) {
                $query->where('status', 'approved');
            }])
            ->take(5)
            ->get();
    }

    /**
     * Select an institution from the search results.
     */
    public function selectInstitution(Institution $institution)
    {
        $this->selectedInstitution = $institution;
        $this->join_search = $institution->name;
        $this->join_searchResults = [];
    }

    /**
     * Create a new institution and make the current user its admin.
     */
    public function createInstitution()
    {
        $validatedData = $this->validate([
            'create_name' => 'required|string|max:255|unique:institutions,name',
            'create_type' => 'nullable|string|max:255',
            'create_address' => 'nullable|string',
            'create_phone' => 'nullable|string|max:50',
            'create_website' => 'nullable|url|max:255',
        ]);

        // Generate a unique code for the new institution
        do {
            $generatedCode = strtoupper(Str::random(3));
        } while (Institution::where('code', $generatedCode)->exists());

        $institution = Institution::create([
            'name' => $validatedData['create_name'],
            'code' => $generatedCode,
            'type' => $validatedData['create_type'],
            'address' => $validatedData['create_address'],
            'phone' => $validatedData['create_phone'],
            'website' => $validatedData['create_website'],
            'status' => 'pending_approval', // وضعیت جدید
        ]);

        // Add the user as the admin of the new institution
        $institution->memberships()->create([
            'user_id' => Auth::id(),
            'role' => 'admin',
            'status' => 'pending', // عضویت نیز در انتظار تایید سازمان است
        ]);

        // در آینده، یک اعلان برای super_admin ارسال خواهد شد

        session()->flash('onboarding_message', 'درخواست ایجاد سازمان شما با موفقیت ثبت شد و پس از بررسی توسط بنچسینک، نتیجه به شما اطلاع داده خواهد شد.');
        $this->mode = 'request_sent'; // یک حالت جدید برای نمایش پیام    }
    }

    /**
     * Send a request to join an existing institution.
     */
    public function requestToJoin()
    {
        $this->validate([
            'selectedInstitution' => 'required',
        ], ['selectedInstitution.required' => 'لطفاً یک سازمان را از لیست جستجو انتخاب کنید.']);

        $user = Auth::user();

        // Check if a membership request already exists
        $existingMembership = $user->institutionMemberships()
            ->where('institution_id', $this->selectedInstitution->id)
            ->exists();

        if ($existingMembership) {
            $this->addError('join_search', 'شما قبلاً برای این سازمان درخواست عضویت داده‌اید یا عضو آن هستید.');
            return;
        }

        // Create a pending membership request
        $this->selectedInstitution->memberships()->create([
            'user_id' => $user->id,
            'role' => 'member', // Default role
            'status' => 'pending',
        ]);

        $this->requestSent = true;
    }
    /**
     * [NEW] Create a personal institution and research team for a solo user.
     */
    public function createPersonalTeam()
    {
        $user = Auth::user();

        DB::transaction(function () use ($user) {
            // 1. Create a personal "Institution"
            $institutionName = 'سازمان شخصی ' . $user->name;
            do {
                $institutionCode = strtoupper(Str::random(3));
            } while (Institution::where('code', $institutionCode)->exists());
            $institution = Institution::create([
                'name' => $institutionName,
                'code' => $institutionCode,
                'status' => 'approved',
            ]);

            // 2. Create a personal "Research Team"
            $teamName = 'تیم شخصی ' . $user->name;
            do {
                $teamCode = strtoupper(Str::random(4));
            } while (ResearchTeam::where('research_teams_code', 'like', '%-' . $teamCode)->exists());
            $fullTeamCode = $institutionCode . '-' . $teamCode;
            $team = $institution->researchTeams()->create([
                'name' => $teamName,
                'research_teams_code' => $fullTeamCode,
            ]);

            // 3. Add the user to the new team as an admin
            $team->users()->attach($user->id, ['role' => 'admin']);

            // 4. Create an approved membership for the user
            $institution->memberships()->create([
                'user_id' => $user->id,
                'role' => 'admin',
                'status' => 'approved',
            ]);

            // 5. [NEW] Assign system-wide manager roles to the user
            $user->assignRole(['institute_manager', 'team_manager']);
        });

        return redirect()->route('research-teams.dashboard');
    }



    public function render()
    {
        return view('livewire.institution-onboarding')
            ->layout('layouts.app');
    }
}
