<?php

namespace App\Livewire;

use App\Models\Institution;
use App\Models\InventoryItem;
use App\Models\StorageLocation;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests; // این خط برای استفاده از authorize اضافه می‌شود
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Illuminate\Support\Str;
use App\Models\ResearchTeam;
use Illuminate\Support\Collection;



class InventoryManager extends Component
{
    use AuthorizesRequests; // این Trait قابلیت authorize را به کامپوننت اضافه می‌کند

    // Properties for the "Create" modal form
    public $name, $item_type, $quantity, $unit, $location_id, $research_team_id;
    public $received_date, $expiration_date, $minimum_stock_level;
    public $generatedSku = '---';
    public $confirmingItemAddition = false;

    // Properties for the "Edit" modal form
    public ?InventoryItem $editingItem;
    public array $editingState = [];
    public $confirmingItemEdit = false;

    // Properties for the "Delete" modal form
    public $itemIdToDelete;
    public $confirmingItemDeletion = false;

    // Properties for Hierarchical Locations
    public $selected_team_id;
    public $selected_level_1, $selected_level_2, $selected_level_3, $selected_level_4, $selected_level_5;
    public $locations_level_1 = [], $locations_level_2 = [], $locations_level_3 = [], $locations_level_4 = [], $locations_level_5 = [];
// Properties for Smart & Manual Placement
    public $placement_mode = 'auto'; // 'auto' or 'manual'
    public ?StorageLocation $selectedBox = null;
    public $occupied_positions = [];
    public $selected_position = null;

    // Properties for Advanced Search
    public $searchQuery = '';
    public $searchTeamId = '';

    // Properties for History Modal
    public ?InventoryItem $historyItem = null;
    public $confirmingHistoryView = false;
    // Properties for Box Full Error Modal
    public $confirmingBoxFullError = false;
    public $boxFullErrorMessage = '';

    // Properties for Deletion Error Modal
    public $confirmingDeleteError = false;
    public $deleteErrorMessage = '';

    // Properties for Bulk Actions
    public $selectedItems = [];
    public $selectAll = false;
    public $price, $brand, $lot_number;

    // [NEW] To store the teams the user is authorized to view
    public Collection $authorizedTeams;

    // Listener for date selection from the datepicker
    protected $listeners = ['dateSelected'];


    public function mount()
    {
        $this->authorizedTeams = $this->getAuthorizedTeams();
    }

    /**
     * [NEW] A helper function to get the IDs of the teams the current user is authorized to manage.
     */
    private function getAuthorizedTeamIds(): Collection
    {
        $user = Auth::user();
        $teamIds = collect();

        // Super admins and institution admins can see all teams in their institution
        if ($user->hasRole('super_admin') || $user->hasRole('institution_admin')) {
            // Assuming a user belongs to one institution, adjust if needed
            return ResearchTeam::where('institution_id', $user->institutionMemberships->first()->institution_id)->pluck('id');
        }

        // Team managers and warehouse managers can see their specific teams
        foreach ($user->researchTeams as $team) {
            if ($team->pivot->role === 'admin' || $user->hasRole("Warehouse_Manager-Team-{$team->id}")) {
                $teamIds->push($team->id);
            }
        }

        return $teamIds->unique();
    }

    /**
     * [NEW] A helper function to get the full team models the user is authorized to manage.
     */
    private function getAuthorizedTeams(): Collection
    {
        $teamIds = $this->getAuthorizedTeamIds();
        return ResearchTeam::whereIn('id', $teamIds)->get();
    }

    /**
     * This method runs when the user selects a research team in the form.
     */

    public function updatedSelectedTeamId($teamId)
    {
        $this->resetLocationLevels();
        if ($teamId) {
            $this->locations_level_1 = StorageLocation::where('research_team_id', $teamId)->whereNull('parent_id')->get();
            $team = ResearchTeam::find($teamId);
            if ($team && $team->research_teams_code) {
                $this->generateSku($team->research_teams_code);
            }
        } else {
            $this->generatedSku = '---';
        }
    }

    public function updatedSelectedLevel1($parentId)
    {
        $this->selected_level_2 = null;
        $this->selected_level_3 = null;
        $this->selected_level_4 = null;
        $this->selectedBox = null;
        $this->locations_level_2 = StorageLocation::where('parent_id', $parentId)->get();
        $this->locations_level_3 = [];
        $this->locations_level_4 = [];
    }

    public function updatedSelectedLevel2($parentId)
    {
        $this->selected_level_3 = null;
        $this->selected_level_4 = null;
        $this->selectedBox = null;
        $this->locations_level_3 = StorageLocation::where('parent_id', $parentId)->get();
        $this->locations_level_4 = [];
    }

    public function updatedSelectedLevel3($parentId)
    {
        $this->selected_level_4 = null;
        $this->selectedBox = null;
        $this->locations_level_4 = StorageLocation::where('parent_id', $parentId)->get();
    }
    public function updatedSelectedLevel4($parentId)
    {
        $this->selected_level_5 = null;
        $this->selectedBox = null;
        $this->locations_level_5 = StorageLocation::where('parent_id', $parentId)->get();
    }

    public function updatedSelectedLevel5($locationId)
    {
        $this->selectedBox = StorageLocation::with('inventoryItems')->find($locationId);
        if ($this->selectedBox) {
            $this->occupied_positions = $this->selectedBox->inventoryItems->pluck('position_in_location')->all();
        }
    }

    private function resetLocationLevels()
    {
        $this->selected_level_1 = null;
        $this->selected_level_2 = null;
        $this->selected_level_3 = null;
        $this->selected_level_4 = null;
        $this->selected_level_5 = null;
        $this->selectedBox = null;
        $this->locations_level_1 = [];
        $this->locations_level_2 = [];
        $this->locations_level_3 = [];
        $this->locations_level_4 = [];
        $this->locations_level_5 = [];
    }


    private function generateSku($teamCode)
    {
        do {
            $randomNumber = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
            $sku = $teamCode . '-' . $randomNumber;
        } while (InventoryItem::where('sku', $sku)->exists());
        $this->generatedSku = $sku;
    }

    public function dateSelected($date, $property)
    {
        $this->$property = $date;
    }


    public function confirmItemAddition()
    {
        $this->authorize('create', InventoryItem::class);
        $this->resetErrorBag();
        $this->reset(['name', 'item_type', 'quantity', 'unit', 'location_id', 'research_team_id',
            'received_date', 'expiration_date', 'selected_team_id', 'minimum_stock_level',
            'price', 'brand', 'lot_number']);
        $this->generatedSku = '---';
        $this->confirmingItemAddition = true;
    }

    public function saveItem()
    {
        $locationId = $this->selected_level_5 ?? $this->selected_level_4 ??
            $this->selected_level_3 ?? $this->selected_level_2 ??
            $this->selected_level_1;

        $rules = [
            'name' => 'required|string|max:255',
            'selected_team_id' => 'required|exists:research_teams,id',
            'item_type' => 'required|in:reagent,consumable,biological_sample,kit',
            'quantity' => 'required|numeric|min:0',
            'unit' => 'required|string|max:50',
            'minimum_stock_level' => 'nullable|numeric|min:0',
            'price' => 'nullable|numeric|min:0',
            'brand' => 'nullable|string|max:255',
            'lot_number' => 'nullable|string|max:255',
        ];

        // --- منطق جدید و هوشمندانه ---
        // اگر حالت جانمایی دستی باشد، انتخاب یک موقعیت الزامی است
        if ($this->placement_mode === 'manual' && $this->selectedBox) {
            $rules['selected_position'] = 'required';
        }

        $this->validate($rules, [
            'selected_position.required' => 'لطفاً یک موقعیت را از نمای گرافیکی انتخاب کنید.'
        ]);

        $this->validate($rules);

        $position = null;
        if ($this->selectedBox) {
            if ($this->placement_mode === 'auto') {
                $position = $this->findFirstAvailablePosition($this->selectedBox);
                if (is_null($position)) {
                    // --- منطق جدید: نمایش مودال خطا ---
                    $this->boxFullErrorMessage = "ظرفیت محفظه '{$this->selectedBox->name}' کامل است. لطفاً یک محفظه دیگر انتخاب کنید یا یک آیتم را به صورت دستی جانمایی نمایید.";
                    $this->confirmingBoxFullError = true;
                    return;
                }
            } else {
                $position = $this->selected_position;
            }
        }

        InventoryItem::create([
            'name' => $this->name,
            'sku' => $this->generatedSku,
            'item_type' => $this->item_type,
            'quantity' => $this->quantity,
            'unit' => $this->unit,
            'price' => $this->price,
            'brand' => $this->brand,
            'lot_number' => $this->lot_number,
            'minimum_stock_level' => $this->minimum_stock_level,
            'storage_location_id' => $locationId,
            'location_id' => $locationId,
            'position_in_location' => $this->selectedBox ? $position : null,
            'research_team_id' => $this->selected_team_id,
            'owner_id' => Auth::id(),
            'received_date' => $this->received_date,
            'expiration_date' => $this->expiration_date,
        ]);

        $this->confirmingItemAddition = false;
        session()->flash('message', 'آیتم جدید با موفقیت به انبار اضافه شد.');
    }

    public function confirmItemEdit(InventoryItem $item)
    {
        $this->authorize('update', $item);
        $this->resetErrorBag();
        $this->editingItem = $item;
        $this->editingState = $item->toArray();
        $this->confirmingItemEdit = true;
    }

    public function updateItem()
    {
        $this->authorize('update', $this->editingItem);

        $this->validate([
            'editingState.name' => 'required|string|max:255',
            'editingState.item_type' => 'required|string|in:reagent,consumable,biological_sample,kit',
            'editingState.quantity' => 'required|numeric|min:0',
            'editingState.unit' => 'required|string|max:50',
            'editingState.price' => 'nullable|numeric|min:0',
            'editingState.brand' => 'nullable|string|max:255',
            'editingState.lot_number' => 'nullable|string|max:255',
            'editingState.minimum_stock_level' => 'nullable|numeric|min:0',
            'editingState.location_id' => 'nullable|exists:storage_locations,id',
            'editingState.research_team_id' => 'required|exists:research_teams,id',
            'editingState.received_date' => 'nullable|date',
            'editingState.expiration_date' => 'nullable|date|after_or_equal:editingState.received_date',
        ]);

        unset($this->editingState['sku']);

        $this->editingItem->update($this->editingState);

        $this->confirmingItemEdit = false;
        session()->flash('message', 'آیتم با موفقیت به‌روزرسانی شد.');
    }

    public function confirmItemDeletion($id)
    {
        $this->itemIdToDelete = $id;
        $this->confirmingItemDeletion = true;
    }



    /**
     * Delete the inventory item, only if it's not part of an active transfer.
     */

    public function deleteItem()
    {
        $item = InventoryItem::with('transferItems.transfer')->findOrFail($this->itemIdToDelete);
        $this->authorize('delete', $item);

        // --- منطق جدید: بررسی حواله‌های فعال ---
        // ما بررسی می‌کنیم که آیا این آیتم در هیچ حواله‌ای با وضعیت "در انتظار دریافت" وجود دارد یا خیر.
        $hasActiveTransfer = $item->transferItems->contains(function ($transferItem) {
            // --- اصلاح کلیدی: ابتدا وجود رابطه را بررسی می‌کنیم ---
            return $transferItem->transfer->status === 'pending_receipt';

        });

        if ($hasActiveTransfer) {
            // به جای فلش کردن پیام، مودال خطا را فعال می‌کنیم
            $this->deleteErrorMessage = 'این آیتم قابل حذف نیست زیرا در یک حواله انبار فعال (در انتظار تایید) قرار دارد.';
            $this->confirmingDeleteError = true;
            $this->confirmingItemDeletion = false; // مودال تایید حذف را می‌بندیم
            return;
        }

        $item->delete();

        $this->confirmingItemDeletion = false;
        session()->flash('message', 'آیتم با موفقیت حذف شد.');
    }

    private function findFirstAvailablePosition(StorageLocation $box): ?string
    {
        $occupied = $box->inventoryItems()->pluck('position_in_location');
        for ($row = 0; $row < $box->rows; $row++) {
            for ($col = 1; $col <= $box->columns; $col++) {
                $position = chr(65 + $row) . $col;
                if (!$occupied->contains($position)) {
                    return $position;
                }
            }
        }
        return null; // Box is full
    }
    /**
     * Show the modal with the item's usage history.
     */
    public function viewHistory(InventoryItem $item)
    {
        // We load the relationship to see which lab notes the item was used in.
        $this->historyItem = $item->load('notebookEntries.project');
        $this->confirmingHistoryView = true;
    }

    /**
     * This method runs when the "Select All" checkbox is toggled.
     */
    public function updatedSelectAll($value)
    {
//        if ($value) {
//            // Get all visible item IDs and add them to the selected list
//            $this->selectedItems = $this->getFilteredItems()->pluck('id')->map(fn($id) => (string) $id)->toArray();
//        } else {
//            $this->selectedItems = [];
//        }
        if ($value) {
            $this->selectedItems = $this->getFilteredItemsQuery()->pluck('id')->map(fn($id) => (string) $id)->toArray();
        } else {
            $this->selectedItems = [];
        }
    }

    /**
     * A helper method to get the currently filtered items.
     */
//    private function getFilteredItems()
//    {
//        $user = Auth::user();
//        $userTeamIds = $user->researchTeams()->pluck('research_teams.id');
////        $itemsQuery = InventoryItem::whereIn('research_team_id', $userTeamIds);
//        $itemsQuery = InventoryItem::query()
//            ->where('is_biological', false)
//            ->whereIn('research_team_id', $userTeamIds);
//
//
//        if (!empty($this->searchQuery)) {
//            $itemsQuery->where(function ($query) {
//                $query->where('name', 'like', '%' . $this->searchQuery . '%')
//                    ->orWhere('sku', 'like', '%' . $this->searchQuery . '%');
//            });
//        }
//        if (!empty($this->searchTeamId)) {
//            $itemsQuery->where('research_team_id', $this->searchTeamId);
//        }
//        return $itemsQuery->get();
//    }

    private function getFilteredItemsQuery()
    {
        $authorizedTeamIds = $this->getAuthorizedTeamIds();

        $itemsQuery = InventoryItem::query()
            ->where('is_biological', false)
            ->whereIn('research_team_id', $authorizedTeamIds);


        if (!empty($this->searchQuery)) {
            $itemsQuery->where(function ($query) {
                $query->where('name', 'like', '%' . $this->searchQuery . '%')
                    ->orWhere('sku', 'like', '%' . $this->searchQuery . '%');
            });
        }
        if (!empty($this->searchTeamId)) {
            // Ensure the user is authorized to view the selected team
            if($authorizedTeamIds->contains($this->searchTeamId)){
                $itemsQuery->where('research_team_id', $this->searchTeamId);
            }
        }
        return $itemsQuery;
    }
    /**
     * Store selected item IDs in session and redirect to the bulk print route.
     */
    public function printSelectedBarcodes()
    {
        if (empty($this->selectedItems)) {
            return;
        }

        // We store the IDs in the session to avoid a very long URL.
        session()->put('selected_barcode_items', $this->selectedItems);

        // We need to return a redirect response that opens in a new tab.
        // This is done by emitting an event to be handled by AlpineJS.
        $this->dispatch('open-new-tab', url: route('inventory.items.bulk-barcode'));
    }

    public function showBoxViewer(int $locationId)
    {
        return redirect()->route('inventory.locations.view', $locationId);
    }

    public function render()
    {
//        $user = Auth::user();
//        $researchTeams = $user->researchTeams()->get();
//
//        $items = $this->getFilteredItems();
//
//        // Update the "Select All" checkbox state
//        $this->selectAll = count($this->selectedItems) === $items->count() && $items->count() > 0;
//
//        return view('livewire.inventory-manager', [
//            'items' => $items,
//            'researchTeams' => $researchTeams,
//        ])->layout('layouts.app');
        $itemsQuery = $this->getFilteredItemsQuery();
        $items = $itemsQuery->paginate(15);
        $this->selectAll = count($this->selectedItems) === $items->total() && $items->total() > 0;

        return view('livewire.inventory-manager', [
            'items' => $items,
            'researchTeams' => $this->authorizedTeams,
        ])->layout('layouts.app');
    }
}
