<?php

namespace App\Livewire;

use App\Models\InventoryItem;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use App\Models\InventoryConsumptionRequest; // [NEW]
use App\Models\Project; // [NEW]
use Illuminate\Support\Facades\DB; // [NEW]
use App\Models\ResearchTeam;

class InventoryReport extends Component
{
    public string $activeReport = 'expiring_soon';
    public $results = [];
    public $projects = [];
    public float $totalCost = 0;


    // Filters for the 'expiring_soon' report
    public int $expiryMonths = 3;
    // [NEW] Filters for 'consumption_rate' report
    public int $consumptionTimeframe = 30; // Default to last 30 days
    public $consumptionProjectId = 'all'; // Default to all projects

    /**
     * A helper method to get the IDs of teams managed by the current user.
     *
     * @return \Illuminate\Support\Collection
     */
    private function getManagedTeamIds()
    {
        return Auth::user()->researchTeams()
            ->wherePivot('role', 'admin') // [FIXED] Check the role in the pivot table
            ->pluck('research_teams.id');
    }
    /**
     * Runs when the component is first mounted.
     * Loads the default report.
     */
    public function mount()
    {
        $userManagedTeamIds = $this->getManagedTeamIds();
        $this->projects = Project::whereIn('research_team_id', $userManagedTeamIds)->get();

        $this->generateReport();
    }

    /**
     * Main method to generate the currently active report.
     */
    public function generateReport()
    {
        $this->results = []; // Reset results before generating
        $this->totalCost = 0; // Reset total cost

        // A switch statement to handle different report types in the future
        switch ($this->activeReport) {
            case 'expiring_soon':
                $this->runExpiringSoonReport();
                break;
            case 'consumption_rate': // [NEW]
                $this->runConsumptionRateReport();
                break;
            case 'cost_report': // [NEW]
                $this->runCostReport();
                break;
        }
    }

    /**
     * Sets the active report type and regenerates the report.
     *
     * @param string $reportName
     */
    public function setActiveReport(string $reportName)
    {
        $this->activeReport = $reportName;
        $this->results = []; // Clear previous results
        $this->generateReport();
    }

    /**
     * Fetches inventory items that are expiring within the selected timeframe.
     */
    private function runExpiringSoonReport()
    {
        $this->validate(['expiryMonths' => 'required|integer|min:1']);

        $userManagedTeamIds = $this->getManagedTeamIds();
        $targetDate = Carbon::now()->addMonths($this->expiryMonths);

        $this->results = InventoryItem::whereIn('research_team_id', $userManagedTeamIds)
            ->whereNotNull('expiration_date')
            ->where('expiration_date', '<=', $targetDate)
            ->where('expiration_date', '>=', Carbon::now()) // Only include items not yet expired
            ->orderBy('expiration_date', 'asc')
            ->with('location', 'researchTeam')
            ->get();
    }
    /**
     * [NEW] Fetches and aggregates consumption data based on selected filters.
     */
    private function runConsumptionRateReport()
    {
        $this->validate([
            'consumptionTimeframe' => 'required|integer',
            'consumptionProjectId' => 'required',
        ]);

        $userManagedTeamIds = $this->getManagedTeamIds();
        $startDate = Carbon::now()->subDays($this->consumptionTimeframe);

        $query = InventoryConsumptionRequest::query()
            ->join('inventory_items', 'inventory_consumption_requests.inventory_item_id', '=', 'inventory_items.id')
            ->whereIn('inventory_items.research_team_id', $userManagedTeamIds)
            ->where('inventory_consumption_requests.type', 'consumption')
            ->where('inventory_consumption_requests.status', 'approved')
            ->where('inventory_consumption_requests.processed_at', '>=', $startDate)
            ->select(
                'inventory_items.id',
                'inventory_items.name',
                'inventory_items.sku',
                'inventory_items.unit',
                DB::raw('SUM(inventory_consumption_requests.quantity) as total_consumed')
            )
            ->groupBy('inventory_items.id', 'inventory_items.name', 'inventory_items.sku', 'inventory_items.unit')
            ->orderBy('total_consumed', 'desc');

        if ($this->consumptionProjectId !== 'all') {
            $query->where('inventory_consumption_requests.project_id', $this->consumptionProjectId);
        }

        $this->results = $query->get();
    }

    /**
     * [NEW] Fetches consumption data and calculates the cost for each item.
     */
    private function runCostReport()
    {
        $this->validate([
            'consumptionTimeframe' => 'required|integer',
            'consumptionProjectId' => 'required',
        ]);

        $userManagedTeamIds = $this->getManagedTeamIds();
        $startDate = Carbon::now()->subDays($this->consumptionTimeframe);

        $query = InventoryConsumptionRequest::query()
            ->join('inventory_items', 'inventory_consumption_requests.inventory_item_id', '=', 'inventory_items.id')
            ->whereIn('inventory_items.research_team_id',  $userManagedTeamIds)
            ->where('inventory_consumption_requests.type', 'consumption')
            ->where('inventory_consumption_requests.status', 'approved')
            ->where('inventory_consumption_requests.processed_at', '>=', $startDate)
            ->select(
                'inventory_items.name',
                'inventory_items.sku',
                'inventory_items.unit',
                DB::raw('SUM(inventory_consumption_requests.quantity) as total_consumed'),
                // Calculate the total cost for each item (quantity * price)
                DB::raw('SUM(inventory_consumption_requests.quantity * inventory_items.price) as total_item_cost')
            )
            ->groupBy('inventory_items.id', 'inventory_items.name', 'inventory_items.sku', 'inventory_items.unit')
            ->orderBy('total_item_cost', 'desc');

        if ($this->consumptionProjectId !== 'all') {
            $query->where('inventory_consumption_requests.project_id', $this->consumptionProjectId);
        }

        $this->results = $query->get();

        // Calculate the grand total cost of all items in the report
        $this->totalCost = $this->results->sum('total_item_cost');
    }


    public function render()
    {
        return view('livewire.inventory-report')->layout('layouts.app');
    }
}
