<?php

namespace App\Livewire;

use App\Models\InventoryItem;
use App\Models\InventoryTransfer;
use App\Models\ResearchTeam;
use App\Notifications\InventoryTransferNotification; // کلاس جدید ما
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Livewire\Component;

class InventoryTransferManager extends Component
{
    // Properties for the "Create Transfer" modal
    public $confirmingTransferCreation = false;
    public $source_team_id, $destination_team_id;
    public $transfer_date, $notes;
    public $transferItems = [];

    // Properties for live item search
    public string $itemSearch = '';
    public $itemSearchResults = [];
    // Properties for live destination team search
    public string $destinationTeamSearch = '';
    public $destinationTeamSearchResults = [];

    // Properties for Retraction
    public $transferIdToRetract;
    public $confirmingRetraction = false;

    public function confirmTransferCreation()
    {
        $this->resetErrorBag();
        $this->reset(['source_team_id', 'destination_team_id', 'transfer_date', 'notes', 'transferItems', 'itemSearch', 'itemSearchResults', 'destinationTeamSearch', 'destinationTeamSearchResults']);
        $this->confirmingTransferCreation = true;
    }

    /**
     * Live search for inventory items in the selected source team.
     */
    public function updatedItemSearch()
    {
        if (!$this->source_team_id || strlen($this->itemSearch) < 2) {
            $this->itemSearchResults = [];
            return;
        }

        $this->itemSearchResults = InventoryItem::where('research_team_id', $this->source_team_id)
            ->where(function ($query) {
                $query->where('name', 'like', '%' . $this->itemSearch . '%')
                    ->orWhere('sku', 'like', '%' . $this->itemSearch . '%');
            })
            ->take(5)
            ->get();
    }

    /**
     * Live search for destination teams.
     */
    public function updatedDestinationTeamSearch()
    {
        if (strlen($this->destinationTeamSearch) < 2) {
            $this->destinationTeamSearchResults = [];
            return;
        }
        // Search all teams, excluding the source team
        $this->destinationTeamSearchResults = ResearchTeam::where('name', 'like', '%' . $this->destinationTeamSearch . '%')
            ->where('id', '!=', $this->source_team_id)
            ->take(5)
            ->get();
    }

    /**
     * Add a selected item to the transfer list.
     */
    public function addItemToTransfer(InventoryItem $item)
    {
        if (!collect($this->transferItems)->contains('id', $item->id)) {
            $this->transferItems[] = [
                'id' => $item->id,
                'name' => $item->name,
                'sku' => $item->sku,
                'unit' => $item->unit,
                'available_quantity' => $item->quantity,
                'quantity_transferred' => 1,
            ];
        }
        $this->itemSearch = '';
        $this->itemSearchResults = [];
    }
    public function selectDestinationTeam(ResearchTeam $team)
    {
        $this->destination_team_id = $team->id;
        $this->destinationTeamSearch = $team->name;
        $this->destinationTeamSearchResults = [];
    }

    /**
     * Remove an item from the transfer list.
     */
    public function removeItem($itemId)
    {
        $this->transferItems = array_filter($this->transferItems, fn($item) => $item['id'] != $itemId);
    }

    /**
     * Save the new inventory transfer.
     */
    public function saveTransfer()
    {
        $validatedData = $this->validate([
            'source_team_id' => 'required|exists:research_teams,id',
            'destination_team_id' => 'required|exists:research_teams,id|different:source_team_id',
            'transfer_date' => 'required|date',
            'notes' => 'nullable|string',
            'transferItems' => 'required|array|min:1',
            'transferItems.*.quantity_transferred' => 'required|numeric|min:0.0001',
        ], [
            'destination_team_id.different' => 'تیم مبدأ و مقصد نمی‌توانند یکسان باشند.'
        ]);
        // بررسی موجودی قبل از شروع تراکنش
        $hasError = false;
        foreach ($this->transferItems as $index => $itemData) {
            $sourceItem = InventoryItem::find($itemData['id']);

            if (!$sourceItem) {
                $this->addError('transferItems.' . $index, 'آیتم مورد نظر یافت نشد');
                $hasError = true;
                continue;
            }

            if ($sourceItem->quantity < $itemData['quantity_transferred']) {
                $this->addError(
                    'transferItems.' . $index . '.quantity_transferred',
                    "موجودی کافی نیست! (حداکثر: $sourceItem->quantity $sourceItem->unit)"
                );
                $hasError = true;
            }
        }

        if ($hasError) {
            return;
        }

        DB::transaction(function () use ($validatedData) {
            // Step 1: Create the main transfer record
            $transfer = InventoryTransfer::create([
                'transfer_code' => 'TR-'.strtoupper(Str::random(10)),
                'source_team_id' => $validatedData['source_team_id'],
                'destination_team_id' => $validatedData['destination_team_id'],
                'transfer_date' => $validatedData['transfer_date'],
                'notes' => $validatedData['notes'],
                'created_by_user_id' => Auth::id(),
                'status' => 'pending_receipt', // وضعیت جدید
            ]);

            // Step 2: Process each item
            foreach ($this->transferItems as $itemData) {
                $sourceItem = InventoryItem::find($itemData['id']);

                // Step 2a: Deduct from source inventory
                $sourceItem->decrement('quantity', $itemData['quantity_transferred']);

//                // Step 2b: Create a new item in the destination inventory
//
//                $destinationTeam = ResearchTeam::with('users')->find($validatedData['destination_team_id']);
//
//                // --- منطق جدید و اصلاح‌شده ---
//                // ما مدیر تیم مقصد را از طریق جدول واسط پیدا می‌کنیم
//                $destinationAdmin = $destinationTeam->users()->wherePivot('role', 'admin')->first();
//
//                // اگر تیمی مدیری نداشت، آیتم به نام اولین عضو تیم ثبت می‌شود
//                $newOwnerId = $destinationAdmin ? $destinationAdmin->id : $destinationTeam->users()->first()->id;
//
//                $newSku = ($destinationTeam->research_teams_code ?? 'TEAM') . '-' . str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
//
//                $newItem = InventoryItem::create([
//                    'name' => $sourceItem->name,
//                    'sku' => $newSku,
//                    'item_type' => $sourceItem->item_type,
//                    'quantity' => $itemData['quantity_transferred'],
//                    'unit' => $sourceItem->unit,
//                    'research_team_id' => $validatedData['destination_team_id'],
//                    'owner_id' => $newOwnerId,
//                    'received_date' => now(),
//                ]);
//
//                // Step 2c: Record the transferred item
//                $transfer->items()->create([
//                    'inventory_item_id' => $sourceItem->id, // Link to the original item
//                    'quantity_transferred' => $itemData['quantity_transferred'],
//                ]);
//            }
//        });
//
//        $this->confirmingTransferCreation = false;
//        session()->flash('message', 'حواله انبار با موفقیت ثبت شد.');
                // Step 2b: Record the transferred item
                $transfer->items()->create([
                    'inventory_item_id' => $sourceItem->id,
                    'quantity_transferred' => $itemData['quantity_transferred'],
                ]);
            }

            // Step 3: Notify the admins of the destination team
            $destinationTeam = ResearchTeam::with('users')->find($validatedData['destination_team_id']);
            $admins = $destinationTeam->users()->wherePivot('role', 'admin')->get();
            foreach ($admins as $admin) {
                $admin->notify(new InventoryTransferNotification($transfer));
            }
        });

        $this->confirmingTransferCreation = false;
        session()->flash('message', 'حواله انبار با موفقیت ثبت و برای تایید تیم مقصد ارسال شد.');
    }

    /**
     * Show the confirmation modal for retracting a transfer.
     */
    public function confirmRetraction($transferId)
    {
        $this->transferIdToRetract = $transferId;
        $this->confirmingRetraction = true;
    }

    /**
     * Retract (delete) the transfer and return the stock to the source inventory.
     */
    public function retractTransfer()
    {
        $transfer = InventoryTransfer::with('items.inventoryItem')->findOrFail($this->transferIdToRetract);

        // We can add a policy check here later if needed.
        if ($transfer->status !== 'pending_receipt') {
            session()->flash('error', 'این حواله قابل بازپس‌گیری نیست زیرا توسط تیم مقصد اقدام شده است.');
            $this->confirmingRetraction = false;
            return;
        }

        DB::transaction(function () use ($transfer) {
            // Return the quantity back to the source inventory
            foreach ($transfer->items as $transferItem) {
                $sourceItem = $transferItem->inventoryItem;
                if ($sourceItem) {
                    $sourceItem->increment('quantity', $transferItem->quantity_transferred);
                }
            }
            // Delete the transfer record, which will cascade delete the transfer items.
            $transfer->delete();
        });

        $this->confirmingRetraction = false;
        session()->flash('message', 'حواله با موفقیت بازپس گرفته شد و موجودی به انبار مبدأ بازگردانده شد.');
    }

    public function render()
    {
        $user = Auth::user();
        $userTeamIds = $user->researchTeams()->pluck('research_teams.id');

        $transfers = InventoryTransfer::whereIn('source_team_id', $userTeamIds)
            ->orWhereIn('destination_team_id', $userTeamIds)
            ->with(['sourceTeam', 'destinationTeam', 'creator'])
            ->withCount('items') // شمارش تعداد آیتم‌های هر حواله
            ->latest()
            ->get();

        $researchTeams = $user->researchTeams()->get();

        return view('livewire.inventory-transfer-manager', [
            'transfers' => $transfers,
            'researchTeams' => $researchTeams,
        ])->layout('layouts.app');
    }
}
