<?php

namespace App\Livewire;

use App\Models\LabNotebookEntry;
use App\Models\Project;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpWord\IOFactory;
use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\Shared\Html;
use PhpOffice\PhpWord\Style\Language;
use Mccarlosen\LaravelMpdf\Facades\LaravelMpdf as PDF;
use ZipArchive;


class LabNotebook extends Component
{
    use AuthorizesRequests;

    public Project $project;
    public $selectedEntries = [];

    // Properties for the "Edit Entry" modal form
    public ?LabNotebookEntry $editingEntry;
    public array $editingState = [];
    public $confirmingEntryEdit = false;

    // Properties for the "Delete Entry" modal form
    public $entryIdToDelete;
    public $confirmingEntryDeletion = false;
    // --- [NEW] Properties for Entry Preview Modal ---
    public ?LabNotebookEntry $previewingEntry = null;
    public bool $showPreviewModal = false;

    protected $listeners = ['editorContentUpdated'];

    public function editorContentUpdated($content)
    {
        // This listener now only applies to the edit modal
        if ($this->confirmingEntryEdit) {
            $this->editingState['body'] = $content;
        }
    }

    public function mount(Project $project)
    {
        $this->project = $project;
    }

    public function confirmEntryEdit(LabNotebookEntry $entry)
    {
        // Add authorization check later if needed
        $this->resetErrorBag();
        $this->editingEntry = $entry;
        $this->editingState = $entry->toArray();
        $this->confirmingEntryEdit = true;
        $this->dispatch('init-editor', ['content' => $entry->body, 'target' => 'entry-body-edit']);
    }

    public function updateEntry()
    {
        // Add authorization check later if needed
        $this->validate([
            'editingState.title' => 'required|string|max:255',
            'editingState.entry_date' => 'required|date',
            'editingState.body' => 'required|string|min:10',
        ]);

        $this->editingEntry->update($this->editingState);
        $this->confirmingEntryEdit = false;
        session()->flash('message', 'یادداشت با موفقیت به‌روزرسانی شد.');
    }

    public function confirmEntryDeletion($entryId)
    {
        // Add authorization check later if needed
        $this->entryIdToDelete = $entryId;
        $this->confirmingEntryDeletion = true;
    }

    public function deleteEntry()
    {
        $entry = LabNotebookEntry::findOrFail($this->entryIdToDelete);
        // Add authorization check later if needed

        $entry->delete();
        $this->confirmingEntryDeletion = false;
        session()->flash('message', 'یادداشت با موفقیت حذف شد.');
        $this->selectedEntries = array_diff($this->selectedEntries, [$this->entryIdToDelete]);

    }

    /**
     * Handles the export request for selected entries based on the specified format.
     *
     * @param string $format The desired export format ('html', 'word', 'zip').
     * @return \Symfony\Component\HttpFoundation\StreamedResponse|void
     */
    public function exportSelected($format)
    {
        // 1. Validate that at least one entry is selected
        if (empty($this->selectedEntries)) {
            // Optionally, dispatch a browser event to show an error notification
            return;
        }

        // 2. Fetch the selected entries from the database
        $entries = LabNotebookEntry::with('user')
            ->whereIn('id', $this->selectedEntries)
            ->get();

        // 3. Check authorization for all selected entries
//        foreach ($entries as $entry) {
//            Gate::authorize('view', $entry);
//        }

        // 4. Generate a filename
        $date = verta()->format('Y-m-d');
        $filename = "benchsync-export-{$this->project->id}-{$date}.{$format}";

        // 5. Handle export based on the format
        switch ($format) {
            case 'html':
                // Render the Blade view with the necessary data
                $htmlContent = view('livewire.exports.lab-notebook-entries', [
                    'entries' => $entries,
                    'project' => $this->project,
                ])->render();

                // Return the rendered HTML as a downloadable file
                return response()->streamDownload(function () use ($htmlContent) {
                    echo $htmlContent;
                }, $filename);

            case 'word':
                $phpWord = new PhpWord();
                $phpWord->getSettings()->setThemeFontLang(new Language('fa-IR'));
                $phpWord->setDefaultFontName('Tahoma');
                $phpWord->setDefaultFontSize(11);

                // ** FIX: Define styles as arrays and use addText instead of addTitle **
                // --- Define Font and Paragraph Styles ---
                $title1Style = ['name' => 'Tahoma', 'size' => 18, 'bold' => true, 'color' => '2F4F4F'];
                $title2Style = ['name' => 'Tahoma', 'size' => 14, 'bold' => true, 'color' => '1E90FF'];
                $metaStyle = ['name' => 'Tahoma', 'size' => 9, 'color' => '708090'];
                $centerParagraphStyle = ['align' => 'center', 'spaceAfter' => 100];
                $rightParagraphStyle = ['align' => 'right', 'spaceAfter' => 100];

                $section = $phpWord->addSection(['bidi' => true]);

                // --- Add Header and Footer ---
                $header = $section->addHeader();
                $header->addText('Benchsync - ELN Export', ['name' => 'Tahoma', 'size' => 8, 'bold' => true], ['align' => 'left']);
                $header->addText("پروژه: {$this->project->title}", ['name' => 'Tahoma', 'size' => 8], ['align' => 'left']);

                $footer = $section->addFooter();
                $footer->addPreserveText('صفحه {PAGE} از {NUMPAGES}', null, ['align' => 'center']);
                $footer->addText("گزارش تولید شده در تاریخ " . verta()->format('Y/m/d H:i'), ['name' => 'Tahoma', 'size' => 8], ['align' => 'center']);

                // --- Document Body ---
                $section->addText('خروجی یادداشت‌های آزمایشگاه', $title1Style, $centerParagraphStyle);
                $section->addTextBreak(1);

                foreach ($entries as $index => $entry) {
                    if ($index > 0) {
                        $section->addPageBreak();
                    }

                    $section->addText($entry->title, $title2Style, $rightParagraphStyle);
                    $metaText = "ثبت توسط:  {$entry->user->first_name} {$entry->user->last_name} | تاریخ ثبت:  " . verta($entry->entry_date)->format('Y/m/d');
                    $section->addText($metaText, $metaStyle, $rightParagraphStyle);
                    $section->addLine(['weight' => 1, 'width' => 450, 'height' => 0, 'color' => '1E90FF']);
                    $section->addTextBreak(1);

                    if (!empty($entry->body)) {
                        try {
                            $styledBody = '<div style="direction: rtl; text-align: right;">' . $entry->body . '</div>';
                            libxml_use_internal_errors(true);
                            Html::addHtml($section, $styledBody, false, false);
                            libxml_clear_errors();
                        } catch (\Exception $e) {
                            $section->addText(
                                '[خطا: فرمت‌بندی این بخش پیچیده بود و به صورت متن ساده نمایش داده شده است.]',
                                ['italic' => true, 'color' => 'ff0000'], $rightParagraphStyle
                            );
                            $section->addTextBreak(1);
                            $text = strip_tags(preg_replace('/<(p|h[1-6])[^>]*>/', "\n\n", $entry->body));
                            $section->addText(trim(html_entity_decode($text, ENT_QUOTES, 'UTF-8')), [], $rightParagraphStyle);
                        }
                    }
                }

                $objWriter = IOFactory::createWriter($phpWord, 'Word2007');
                $tempFile = tempnam(sys_get_temp_dir(), 'benchsync_word_export');
                $objWriter->save($tempFile);

                return response()->download($tempFile, $filename)->deleteFileAfterSend(true);


            case 'zip':
                $zip = new ZipArchive();
                $tempZipFile = tempnam(sys_get_temp_dir(), 'benchsync_zip_export');

                if ($zip->open($tempZipFile, ZipArchive::CREATE) !== TRUE) {
                    throw new \Exception('Cannot create ZIP file.');
                }

                foreach ($entries as $entry) {
                    // Sanitize entry title for use as a folder name
                    $folderName = "Entry_{$entry->sku}_" . preg_replace('/[^A-Za-z0-9\-_\.]/', '_', $entry->entry_date);

                    // 1. Generate and add PDF of the entry
                    $pdf = PDF::loadView('livewire.exports.lab-notebook-single-entry-pdf', ['entry' => $entry]);
                    $zip->addFromString("{$folderName}/{$entry->title}.pdf", $pdf->output());

                    // 2. Add all attachments for the entry
                    foreach ($entry->attachments as $attachment) {
                        // IMPORTANT: Assumes 'path' column in attachments table stores the file path in your default storage disk.
                        if (Storage::disk('public')->exists($attachment->file_path)) {
                            $zip->addFile(
                                Storage::disk('public')->path($attachment->file_path),
                                "{$folderName}/Attachments/{$attachment->file_name}"
                            );
                        }
                    }
                }

                $zip->close();

                return response()->download($tempZipFile, $filename)->deleteFileAfterSend(true);

        }
    }

    /**
     * [NEW] Fetches an entry by its ID and displays it in a preview modal.
     * This method is called by the cross-links generated in the model.
     *
     * @param int $entryId The ID of the lab notebook entry to preview.
     */
    public function showEntryPreview(int $entryId)
    {
        $entry = LabNotebookEntry::with('user')->find($entryId);
        if (!$entry) {
            return; // Or show a notification
        }

        // Optional: You can add an authorization check here if needed
        // Gate::authorize('view', $entry->project);

        $this->previewingEntry = $entry;
        $this->showPreviewModal = true;
    }

    public function render()
    {
        // We only fetch the entries here.
        $entries = $this->project->labNotebookEntries()->with('user')->latest()->get();

        return view('livewire.lab-notebook', [
            'entries' => $entries,
        ]);
    }
}
