<?php

namespace App\Livewire;

use App\Models\InventoryItem;
use App\Models\LowStockAcknowledgement;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class LowStockDashboardWidget extends Component
{
    /**
     * A computed property to get the low stock items.
     */
    public function getLowStockItemsProperty()
    {
        $user = Auth::user();
        if (!$user) {
            return collect();
        }

        // 1. Get IDs of items already acknowledged by the user.
        $acknowledgedItemIds = $user->lowStockAcknowledgements()->pluck('inventory_item_id');

        // 2. Get IDs of all teams the user is a member of.
        $userTeamIds = $user->researchTeams()->pluck('research_teams.id');

        // 3. Find items that are below the minimum stock level in the user's teams,
        //    and have NOT been acknowledged by this user.
        return InventoryItem::whereIn('research_team_id', $userTeamIds)
            ->whereNotNull('minimum_stock_level')
            ->whereColumn('quantity', '<=', 'minimum_stock_level')
            ->whereNotIn('id', $acknowledgedItemIds) // The smart filter
            ->with('researchTeam')
            ->get();
    }

    /**
     * Acknowledge a low stock alert for a specific item, hiding it from the widget.
     */
    public function dismissItem($itemId)
    {
        $user = Auth::user();
        if (!$user) {
            return;
        }

        // Create an acknowledgement record for this user and item.
        LowStockAcknowledgement::create([
            'user_id' => $user->id,
            'inventory_item_id' => $itemId,
        ]);
    }

    public function render()
    {
        return view('livewire.low-stock-dashboard-widget');
    }
}
