<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\ProjectAnnouncement;
use App\Models\ProjectAnnouncementAttachment;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Livewire\WithFileUploads;
use App\Notifications\NewProjectAnnouncement;
use Illuminate\Support\Facades\Notification;



class ProjectAnnouncementBoard extends Component
{
    use AuthorizesRequests;
    use WithFileUploads;

    public Project $project;

    // Properties for Create/Edit Modal
    public bool $confirmingAnnouncementAction = false;
    public ?ProjectAnnouncement $editingAnnouncement = null;
    public array $state = [];

    // Properties for Delete Modal
    public ?ProjectAnnouncement $announcementToDelete = null;
    public bool $confirmingAnnouncementDeletion = false;
    // [NEW] Properties for Attachments
    public $newAttachments = [];
    public $existingAttachments = [];

    protected function rules()
    {
        return [
            'state.title' => 'required|string|max:255',
            'state.body' => 'required|string',
            'state.level' => 'required|in:info,warning,danger',
            'state.is_pinned' => 'nullable|boolean',
            'newAttachments.*' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240', // 10MB Max per file

        ];
    }

    public function mount(Project $project)
    {
        $this->project = $project;
    }

    public function confirmAnnouncementAction(?ProjectAnnouncement $announcement = null)
    {
        $this->resetErrorBag();
        $this->newAttachments = []; // Reset temporary files

        if ($announcement && $announcement->exists) {
            $this->authorize('update', $announcement);
            $this->editingAnnouncement = $announcement->load('attachments');
            $this->state = $announcement->toArray();
            $this->existingAttachments = $announcement->attachments->toArray();
        } else {
            $this->authorize('create', [ProjectAnnouncement::class, $this->project]);
            $this->editingAnnouncement = null;
            $this->state = ['level' => 'info', 'is_pinned' => false];
            $this->existingAttachments = [];
        }
        $this->confirmingAnnouncementAction = true;
    }

    public function saveAnnouncement()
    {
        $validatedData = $this->validate();
        $creator = Auth::user();

        $announcement = DB::transaction(function () use ($validatedData, $creator) {
            if ($this->editingAnnouncement) {
                $this->authorize('update', $this->editingAnnouncement);
                $this->editingAnnouncement->update($validatedData['state']);
                return $this->editingAnnouncement;
            } else {
                $this->authorize('create', [ProjectAnnouncement::class, $this->project]);
                $newAnnouncement = $this->project->announcements()->create($validatedData['state'] + ['user_id' => $creator->id]);

                // Handle file uploads
                foreach ($this->newAttachments as $attachmentFile) {
                    $path = $attachmentFile->store('announcements', 'public');
                    $newAnnouncement->attachments()->create([
                        'user_id' => $creator->id,
                        'file_path' => $path,
                        'file_name' => $attachmentFile->getClientOriginalName(),
                        'file_size' => $attachmentFile->getSize(),
                        'mime_type' => $attachmentFile->getMimeType(),
                    ]);
                }

                return $newAnnouncement;
            }
        });

        // Send notification only for new announcements
        if (!$this->editingAnnouncement) {
            $recipients = $this->project->collaborators->push($this->project->owner)
                ->unique('id')
                ->where('id', '!=', $creator->id);

            if ($recipients->isNotEmpty()) {
                Notification::send($recipients, new NewProjectAnnouncement($announcement, $creator));
            }
            session()->flash('announcement_message', 'اطلاعیه جدید با موفقیت ایجاد و برای اعضا ارسال شد.');
        } else {
            session()->flash('announcement_message', 'اطلاعیه با موفقیت به‌روزرسانی شد.');
        }

        $this->confirmingAnnouncementAction = false;
    }

    public function confirmAnnouncementDeletion(ProjectAnnouncement $announcement)
    {
        // $this->authorize('delete', $announcement);
        $this->announcementToDelete = $announcement;
        $this->confirmingAnnouncementDeletion = true;
    }

    public function deleteAnnouncement($attachmentId)
    {
        $attachment = ProjectAnnouncementAttachment::findOrFail($attachmentId);
        // $this->authorize('delete', $attachment);

        Storage::disk('public')->delete($attachment->file_path);
        $attachment->delete();

        // Refresh the attachments list in the modal
        $this->existingAttachments = $this->editingAnnouncement->attachments()->get()->toArray();
    }

    public function togglePin(ProjectAnnouncement $announcement)
    {
        // $this->authorize('update', $announcement);
        $announcement->update(['is_pinned' => !$announcement->is_pinned]);
    }

    public function render()
    {
        $announcements = $this->project->announcements()
            ->with('user')
            ->orderBy('is_pinned', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('livewire.project-announcement-board', [
            'announcements' => $announcements
        ]);
    }
}
