<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\protocols;
use App\Models\InventoryItem;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Component;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;


class ProjectDetail extends Component
{
    use AuthorizesRequests;

    public Project $project;

    // Properties for Collaborators
    public $userIdToAdd;
    public $roleIdToAdd; // [UPDATED] به جای نام نقش، ID آن را ذخیره می‌کنیم

    public string $search = '';
    public $searchResults = [];
    public $confirmingCollaboratorAddition = false;
    public $collaboratorIdToRemove;
    public $confirmingCollaboratorRemoval = false;
    public $assignableRoles = [];
    public $projectRole;

    // Properties for Protocols
    public $protocolIdToAdd;
    public $confirmingProtocolAddition = false;
    public $protocolIdToRemove;
    public $confirmingProtocolRemoval = false;

    // Properties for Project Edit Modal
    public array $editingState = [];
    public $confirmingProjectEdit = false;

    // ==[ NEW ]== Properties for Budget Management
    public $confirmingBudgetManagement = false;
    public $inventoryItemSearch = '';
    public $inventorySearchResults = [];
    public $selectedInventoryItemId;
    public $max_allowed_quantity;
    public $budgetItemToRemoveId;
    public $confirmingBudgetItemRemoval = false;
    public ?InventoryItem $selectedInventoryItem = null; // برای نگهداری اطلاعات آیتم انتخاب شده

    // Properties for Editing a budget item
    public $editingBudgetItemId = null;
    public $editingMaxAllowedQuantity = '';
    public $confirmingBudgetEdit = false;

    // [NEW] Properties for Custom Permissions
    public $permissionsForSelectedRole = [];
    public array $customPermissions = [];

    // [NEW] Properties for Editing Collaborator Permissions
    public ?User $editingCollaborator = null;
    public $confirmingPermissionEdit = false;
    // [NEW] Properties for Warehouse Manager
    public $warehouseManagerId;
    // [NEW] پراپرتی برای مدیریت تب فعال
    public string $activeTab = 'tasks';


    public function mount(Project $project)
    {
        $this->authorize('view', $project);
        $this->project = $project;

        // [FIXED] We now explicitly load the pivot data for the collaborators relationship.
        $this->project->load([
            'researchTeam.institution',
            'owner',
            'protocols',
            'budgetItems',
            'warehouseManager',
            'collaborators' => function ($query) {
                $query->where('users.id', '!=', $this->project->owner_id) // Exclude owner
                ->withPivot('project_role')
                    ->with(['projectPermissions' => function ($permissionQuery) {
                        $permissionQuery->where('project_user_permission.project_id', $this->project->id);
                    }]);
            }
        ]);
        $this->warehouseManagerId = $this->project->warehouse_manager_id;
    }
    // Collaborator Methods...
    public function updatedSearch($value)
    {
        if (strlen($this->search) < 2) {
            $this->searchResults = [];
            return;
        }
        $existingUserIds = $this->project->collaborators->pluck('id')->push($this->project->owner_id);
        $this->searchResults = User::with('institutionMemberships.institution')
            ->whereNotIn('id', $existingUserIds)
            ->where(function ($query) {
                $query->where('username', 'like', '%' . $this->search . '%')
                    ->orWhere('first_name', 'like', '%' . $this->search . '%')
                    ->orWhere('last_name', 'like', '%' . $this->search . '%')
                    ->orWhere('email', 'like', '%' . $this->search . '%');
            })
            ->take(5)
            ->get();
    }

    public function selectUser(User $user)
    {
        $this->userIdToAdd = $user->id;
        $this->search = $user->first_name . ' ' . $user->last_name;
        $this->searchResults = [];
    }

    public function confirmCollaboratorAddition()
    {
        $this->authorize('update', $this->project);
        $this->resetErrorBag();
        $this->reset(['userIdToAdd', 'roleIdToAdd', 'search', 'searchResults', 'permissionsForSelectedRole', 'customPermissions']);
        $this->assignableRoles = Role::whereIn('name', ['project_owner', 'adviser', 'operator', 'student', 'researcher'])->get();
        $this->confirmingCollaboratorAddition = true;
    }

    public function updatedRoleIdToAdd($roleId)
    {
        if ($roleId) {
            $role = Role::findById($roleId, 'web');
            $this->permissionsForSelectedRole = $role->permissions;
            $this->customPermissions = $role->permissions->pluck('id')->map(fn($id) => (string) $id)->toArray();
        } else {
            $this->permissionsForSelectedRole = [];
            $this->customPermissions = [];
        }
    }

    public function addCollaborator()
    {
        $this->authorize('manageCollaborators', $this->project);
        $this->validate([
            'userIdToAdd' => 'required|exists:users,id',
            'roleIdToAdd' => 'required|exists:permission_roles,id',
            'customPermissions' => 'nullable|array',
            'customPermissions.*' => 'exists:bench_permissions,id',
        ]);
        $user = User::find($this->userIdToAdd);
        $role = Role::findById($this->roleIdToAdd, 'web');
        $this->project->collaborators()->attach($user->id, ['project_role' => $role->name]);
        $permissionsToSync = collect($this->customPermissions)->mapWithKeys(fn($permissionId) => [$permissionId => ['project_id' => $this->project->id]]);
        $user->projectPermissions()->syncWithoutDetaching($permissionsToSync);
        $this->confirmingCollaboratorAddition = false;
        $this->project->load('collaborators');
        session()->flash('message', 'همکار جدید با موفقیت به طرح اضافه شد.');
    }

    public function confirmCollaboratorRemoval($userId)
    {
        $this->authorize('update', $this->project);
        $this->collaboratorIdToRemove = $userId;
        $this->confirmingCollaboratorRemoval = true;
    }

    public function removeCollaborator()
    {
        $this->authorize('manageCollaborators', $this->project);
        $user = User::find($this->collaboratorIdToRemove);
        if ($user) {
            $user->projectPermissions()->wherePivot('project_id', $this->project->id)->detach();
        }
        $this->project->collaborators()->detach($this->collaboratorIdToRemove);
        $this->confirmingCollaboratorRemoval = false;
        $this->project->load('collaborators');
        session()->flash('message', 'همکار با موفقیت از طرح حذف شد.');
    }


    // Protocol Methods...
    public function confirmProtocolAddition()
    {
        $this->authorize('update', $this->project);
        $this->resetErrorBag();
        $this->reset('protocolIdToAdd');
        $this->confirmingProtocolAddition = true;
    }

    public function addProtocol()
    {
        $this->authorize('update', $this->project);
        $this->validate([
            'protocolIdToAdd' => 'required|exists:protocols,id',
        ]);
        if ($this->project->protocols()->where('protocol_id', $this->protocolIdToAdd)->exists()) {
            session()->flash('error', 'این پروتکل از قبل به طرح اضافه شده است.');
            return;
        }
        $this->project->protocols()->attach($this->protocolIdToAdd);
        $this->confirmingProtocolAddition = false;
        $this->project->load('protocols');
        session()->flash('message', 'پروتکل جدید با موفقیت به طرح اضافه شد.');
    }

    public function confirmProtocolRemoval($protocolId)
    {
        $this->authorize('update', $this->project);
        $this->protocolIdToRemove = $protocolId;
        $this->confirmingProtocolRemoval = true;
    }

    public function removeProtocol()
    {
        $this->authorize('update', $this->project);
        $this->project->protocols()->detach($this->protocolIdToRemove);
        $this->confirmingProtocolRemoval = false;
        $this->project->load('protocols');
        session()->flash('message', 'پروتکل با موفقیت از طرح حذف شد.');
    }

    // Project Edit Methods
    public function confirmProjectEdit()
    {
        $this->authorize('update', $this->project);
        $this->resetErrorBag();
        $this->editingState = $this->project->toArray();
        $this->confirmingProjectEdit = true;
    }

    public function updateProject()
    {
        $this->authorize('update', $this->project);

        $validatedData = $this->validate([
            'editingState.title' => 'required|string|max:255',
            'editingState.type' => 'required|string|in:approved,thesis,personal',
            'editingState.status' => 'required|string|in:معلق,در حال اجرا,نیاز به اجرا مجدد,شکست خورده,کامل شده',
            'editingState.description' => 'nullable|string',
            'editingState.budget' => 'nullable|numeric|min:0',
            'editingState.approval_date' => 'nullable|date',
            'editingState.start_date' => 'required|date',
            'editingState.end_date' => 'nullable|date|after_or_equal:editingState.start_date',
        ])['editingState'];

        $this->project->update($validatedData);

        $this->confirmingProjectEdit = false;
        session()->flash('message', 'اطلاعات طرح با موفقیت به‌روزرسانی شد.');
    }

    // ==[ NEW ]== Budget Management Methods
    /**
     * Show the main modal for managing the project's material budget.
     */
    public function confirmBudgetManagement()
    {
        $this->authorize('manageBudget', $this->project);
        $this->resetErrorBag();
        $this->reset(['inventoryItemSearch', 'inventorySearchResults', 'selectedInventoryItemId', 'max_allowed_quantity', 'selectedInventoryItem']);
        $this->confirmingBudgetManagement = true;
    }

    /**
     * Live search for inventory items within the project's research team.
     */
    public function updatedInventoryItemSearch($value)
    {
        if (strlen($this->inventoryItemSearch) < 2) {
            $this->inventorySearchResults = [];
            return;
        }

        // Get the ID of the project's research team.
        $researchTeamId = $this->project->research_team_id;

        // Get IDs of items already in the budget to exclude them from search results.
        $existingBudgetItemIds = $this->project->budgetItems()->pluck('inventory_item_id');

        $this->inventorySearchResults = InventoryItem::where('research_team_id', $researchTeamId)
            ->whereNotIn('id', $existingBudgetItemIds)
            ->where('name', 'like', '%' . $this->inventoryItemSearch . '%')
            ->take(5)
            ->get();
    }

    /**
     * Select an inventory item from the search results.
     */
    public function selectInventoryItem(InventoryItem $item)
    {
        $this->selectedInventoryItemId = $item->id;
        $this->selectedInventoryItem = $item; // ذخیره کل آبجکت آیتم
        $this->inventoryItemSearch = $item->name;
        $this->inventorySearchResults = [];
    }

    /**
     * Add the selected inventory item to the project's budget.
     */
    public function addBudgetItem()
    {
        $this->authorize('update', $this->project);
        $this->validate([
            'selectedInventoryItemId' => 'required|exists:inventory_items,id',
            'max_allowed_quantity' => 'required|numeric|min:0.01',
        ]);

        $this->project->budgetItems()->attach($this->selectedInventoryItemId, [
            'max_allowed_quantity' => $this->max_allowed_quantity,
            'total_consumed_quantity' => 0 // Initial consumption is always zero
        ]);

        $this->project->load('budgetItems'); // Refresh the relationship data
        $this->reset(['inventoryItemSearch', 'inventorySearchResults', 'selectedInventoryItemId', 'max_allowed_quantity', 'selectedInventoryItem']);
        session()->flash('budget_message', 'ماده جدید با موفقیت به بودجه طرح اضافه شد.');
    }

    /**
     * Show confirmation modal before removing a budget item.
     */
    public function confirmBudgetItemRemoval($pivotId)
    {
        $this->authorize('update', $this->project);
        $this->budgetItemToRemoveId = $pivotId;
        $this->confirmingBudgetItemRemoval = true;
    }

    /**
     * Remove an item from the project's budget.
     */
    public function removeBudgetItem()
    {
        $this->authorize('update', $this->project);

        // We use the pivot ID to detach the record.
        // Note: detach() needs the ID of the related model, not the pivot ID.
        $budgetItem = $this->project->budgetItems()->wherePivot('id', $this->budgetItemToRemoveId)->first();
        if ($budgetItem) {
            $this->project->budgetItems()->detach($budgetItem->id);
        }

        $this->confirmingBudgetItemRemoval = false;
        $this->project->load('budgetItems');
        session()->flash('message', 'ماده با موفقیت از بودجه طرح حذف شد.');
    }

    /**
     * ==[ NEW ]== Show the modal for editing a budget item's max quantity.
     */
    public function confirmBudgetEdit($pivotId)
    {
        $this->authorize('manageBudget', $this->project);
        $this->resetErrorBag();
        $budgetItem = $this->project->budgetItems()->wherePivot('id', $pivotId)->first();
        if ($budgetItem) {
            $this->editingBudgetItemId = $pivotId;
            $this->editingMaxAllowedQuantity = $budgetItem->pivot->max_allowed_quantity;
            $this->confirmingBudgetEdit = true;
        }
    }

    /**
     * ==[ NEW ]== Update the max_allowed_quantity for a budget item.
     */
    public function updateBudgetItem()
    {
        $this->authorize('manageBudget', $this->project);
        $this->validate([
            'editingMaxAllowedQuantity' => 'required|numeric|min:0.01',
        ]);

        $this->project->budgetItems()->wherePivot('id', $this->editingBudgetItemId)->update([
            'max_allowed_quantity' => $this->editingMaxAllowedQuantity
        ]);

        $this->confirmingBudgetEdit = false;
        $this->project->load('budgetItems');
        session()->flash('message', 'سقف مجاز با موفقیت به‌روزرسانی شد.');
    }

    /**
     * [NEW] Show the modal for editing a collaborator's permissions.
     */
    public function confirmPermissionEdit(User $user)
    {
        $this->authorize('manageCollaborators', $this->project);
        $this->editingCollaborator = $user;

        // واکشی تمام دسترسی‌های ممکن
        $this->permissionsForSelectedRole = Permission::all();

        // واکشی دسترسی‌های فعلی کاربر در این طرح
        $this->customPermissions = $user->projectPermissions()
            ->wherePivot('project_id', $this->project->id)
            ->pluck('permission_id')->map(fn($id) => (string) $id)->toArray();

        $this->confirmingPermissionEdit = true;
    }

    /**
     * [NEW] Update the custom permissions for the collaborator in this project.
     */
    public function updatePermissions()
    {
        $this->authorize('manageCollaborators', $this->project);
        $this->validate([
            'customPermissions' => 'nullable|array',
            'customPermissions.*' => 'exists:bench_permissions,id',
        ]);

        $permissionsToSync = collect($this->customPermissions)->mapWithKeys(fn($permissionId) => [$permissionId => ['project_id' => $this->project->id]]);

        // ابتدا تمام دسترسی‌های قدیمی کاربر در این طرح را حذف می‌کنیم
        $this->editingCollaborator->projectPermissions()->wherePivot('project_id', $this->project->id)->detach();
        // سپس دسترسی‌های جدید را اضافه می‌کنیم
        $this->editingCollaborator->projectPermissions()->attach($permissionsToSync);
        $this->editingCollaborator->load('projectPermissions');

        $this->confirmingPermissionEdit = false;
        session()->flash('message', 'دسترسی‌های همکار با موفقیت به‌روزرسانی شد.');
    }

    public function assignWarehouseManager()
    {
        $this->authorize('update', $this->project); // یا یک دسترسی خاص‌تر

        $this->validate([
            'warehouseManagerId' => 'nullable|exists:users,id'
        ]);
        $managerId = $this->warehouseManagerId === '' ? null : $this->warehouseManagerId;
        // کاربر انتخاب شده باید حتماً یکی از همکاران طرح باشد
        if ($managerId && !$this->project->collaborators->contains($managerId)) {
            session()->flash('error', 'کاربر انتخاب شده باید یکی از همکاران طرح باشد.');
            return;
        }

        $this->project->update([
            'warehouse_manager_id' => $managerId
        ]);
        $this->project->refresh();
        $this->project->load('warehouseManager'); // بارگذاری مجدد رابطه
        $this->warehouseManagerId = $this->project->warehouse_manager_id; // همگام‌سازی پراپرتی
        session()->flash('message', 'مسئول گردش انبار با موفقیت تعیین شد.');
    }


    // [NEW] متد برای تغییر تب فعال
    public function setActiveTab(string $tabName)
    {
        $this->activeTab = $tabName;
    }

    public function render()
    {
        $existingUserIds = $this->project->collaborators->pluck('id')->push($this->project->owner_id);
        $users = User::whereNotIn('id', $existingUserIds)->get();

        $existingProtocolIds = $this->project->protocols->pluck('id');
        $protocols = protocols::whereNotIn('id', $existingProtocolIds)->get();


        return view('livewire.project-detail', [
            'users' => $users,
            'protocols' => $protocols,
        ])->layout('layouts.app');
    }
}
