<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\Task;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Component;
use Illuminate\Support\Facades\Notification;
use App\Notifications\TaskStatusChangedNotification;
use Illuminate\Support\Facades\Auth;

class ProjectKanbanBoard extends Component
{
    use AuthorizesRequests;

    public Project $project;
    public $tasksByStatus = [];

    public function mount(Project $project)
    {
        $this->project = $project;
        $this->loadTasks();
    }

    /**
     * Load tasks and group them by status for the Kanban board.
     */
    public function loadTasks()
    {
        $tasks = $this->project->tasks()->with('assignees')->get();
        $this->tasksByStatus = $tasks->groupBy('status')->all();
    }

    /**
     * This method is called from the frontend when a task is dropped into a new column.
     */
    public function onTaskDrop($taskId, $newStatus)
    {
        $task = Task::find($taskId);
        if ($task) {
            $this->authorize('changeStatus', $task);

            $oldStatus = $task->status;
            $changer = Auth::user();

            // به‌روزرسانی وضعیت وظیفه
            $task->status = $newStatus;
            $task->completed_at = $newStatus === 'انجام شده' ? now() : null;
            $task->save();

            // [NEW LOGIC] ارسال اعلان به تمام ذینفعان
            // بارگذاری روابط لازم برای جلوگیری از کوئری‌های اضافه
            $task->load(['assignees', 'creator', 'project.owner']);

            // 1. جمع‌آوری تمام دریافت‌کنندگان بالقوه
            $recipients = collect($task->assignees);
            $recipients->push($task->creator);
            $recipients->push($task->project->owner);

            // 2. حذف کاربری که وضعیت را تغییر داده و حذف موارد تکراری
            $uniqueRecipients = $recipients->unique('id')
                ->filter(function ($user) use ($changer) {
                    return $user && $user->id !== $changer->id;
                });

            // 3. ارسال اعلان‌ها
            if ($uniqueRecipients->isNotEmpty()) {
                Notification::send($uniqueRecipients, new TaskStatusChangedNotification($task->fresh(), $changer, $oldStatus));
            }

            // Refresh the tasks on the board.
            $this->loadTasks();
        }
    }

    /**
     * [NEW] Dispatch an event to open the task detail modal.
     */
    public function showTask(Task $task)
    {
        $this->dispatch('showTaskModal', task: $task);
    }

    public function render()
    {
        return view('livewire.project-kanban-board');
    }
}
