<?php

namespace App\Livewire;

use App\Models\Institution;
use App\Models\ResearchTeam;
use App\Models\Project;
use App\Models\Tag;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests; // این خط برای استفاده از authorize اضافه می‌شود
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Livewire\Component;

class ProjectManager extends Component
{
    use AuthorizesRequests; // این Trait قابلیت authorize را به کامپوننت اضافه می‌کند

    // Properties for the "Create" modal form
    public $title, $type, $research_team_id, $description, $category, $custom_category, $start_date;
    public $generatedSku = '';
    public $confirmingProjectAddition = false;

    // Properties for Tags
    public string $tagSearch = '';
    public $tagSearchResults = [];
    public $selectedTags = [];
    public $originalState = '';
    public array $originalTags = []; // پراپرتی جدید برای ذخیره برچسب‌های اولیه



    // ... (کدهای مربوط به ویرایش و حذف در اینجا قرار خواهند گرفت)

    public function updatedTagSearch()
    {
        if (strlen($this->tagSearch) < 2) {
            $this->tagSearchResults = [];
            return;
        }
        $this->tagSearchResults = Tag::where('name', 'like', '%' . $this->tagSearch . '%')->take(5)->get();
    }

    public function addTag($tagName)
    {
        $tag = Tag::firstOrCreate(
            ['name' => $tagName],
            ['slug' => Str::slug($tagName)]
        );
        if (!in_array($tag->id, array_column($this->selectedTags, 'id'))) {
            $this->selectedTags[] = ['id' => $tag->id, 'name' => $tag->name];
        }
        $this->tagSearch = '';
        $this->tagSearchResults = [];
    }

    public function removeTag($tagId)
    {
        $this->selectedTags = array_filter($this->selectedTags, fn($tag) => $tag['id'] != $tagId);
    }

    // Properties for the "Edit" modal form
    public ?Project $editingProject;
    public array $editingState = [];
    public $confirmingProjectEdit = false;

    // Properties for the "Delete" modal form
    public $projectIdToDelete;
    public $confirmingProjectDeletion = false;

    protected function rules()
    {
        return [
            'editingState.title' => 'required|string|max:255',
            'editingState.type' => 'required|in:approved,thesis,personal',
            'editingState.research_team_id' => 'required|exists:research_teams,id',
            'editingState.description' => 'nullable|string',
        ];
    }

    public function confirmProjectAddition()
    {
        $this->authorize('create', Project::class);
        $this->resetErrorBag();
        $this->reset(['title', 'type', 'research_team_id', 'description']);
        $this->generatedSku = '---'; // ریست کردن کد هنگام باز شدن مودال
        $this->confirmingProjectAddition = true;
    }

    /**
     * This method runs when the user selects a research team in the form.
     */
    public function generateSku($institutionCode)
    {
        // تولید کد شناسایی منحصر به فرد
        do {
            $randomNumber = str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT);
            $sku = $institutionCode . '-' . $randomNumber;
        } while (Project::where('sku', $sku)->exists());

        $this->generatedSku = $sku;
    }

    public function updatedResearchTeamId($teamId)
    {
        if ($teamId) {
            $team = ResearchTeam::with('institution')->find($teamId);
            if ($team && $team->institution->code) {
                $this->generateSku($team->institution->code);
            }
        } else {
            $this->generatedSku = '';
        }
    }


//    public function updatedInstitutionId($value)
//    {
//        if ($value) {
//            $institution = Institution::find($value);
//            if ($institution && $institution->code) {
//                // تولید کد شناسایی منحصر به فرد
//                do {
//                    $randomNumber = str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT);
//                    $sku = $institution->code . '-' . $randomNumber;
//                } while (Project::where('sku', $sku)->exists());
//
//                $this->generatedSku = $sku;
//            } else {
//                $this->generatedSku = 'کد نهاد یافت نشد';
//            }
//        } else {
//            $this->generatedSku = '---';
//        }
//    }

    public function saveProject()
    {
        $this->authorize('create', Project::class);

        $this->validate([
            'title' => 'required|string|max:255',
            'type' => 'required|in:approved,thesis,personal',
            'research_team_id' => 'required|exists:research_teams,id', // قانون جدید
            'start_date' => 'required|date',
            'category' => 'required|string',
            'custom_category' => 'required_if:category,other|nullable|string|max:255',
        ]);

        $project = Project::create([
            'title' => $this->title,
            'sku' => $this->generatedSku,
            'type' => $this->type,
            'research_team_id' => $this->research_team_id, // ستون جدید
            'start_date' => $this->start_date,
            'category' => $this->category === 'other' ? $this->custom_category : $this->category,
            'description' => $this->description,
            'owner_id' => Auth::id(),
            'status' => 'pending',
        ]);
        $tagIds = array_column($this->selectedTags, 'id');
        $project->tags()->sync($tagIds);

        $this->confirmingProjectAddition = false;
        session()->flash('message', 'طرح جدید با موفقیت ایجاد شد.');

    }

//    public function confirmProjectEdit(Project $project)
//    {
//        $this->authorize('update', $project);
//        $this->resetErrorBag();
//        $this->editingProject = $project;
//        $this->selectedTags = $project->tags->map(fn($tag) => ['id' => $tag->id, 'name' => $tag->name])->all();
//        $this->editingState = $project->toArray();
//        $this->confirmingProjectEdit = true;
//    }

    public function confirmProjectEdit($projectId)
    {
        $project = Project::with('tags')->findOrFail($projectId);

        $this->authorize('update', $project);

        $this->editingProject = $project;
        $this->selectedTags = $project->tags->map(fn($tag) => [
            'id' => $tag->id,
            'name' => $tag->name
        ])->all();

        $this->editingState = [
            'title' => $project->title,
            'type' => $project->type,
            'start_date' => $project->start_date,
            'category' => $project->category,
            'custom_category' => $project->custom_category, // اضافه شد
            'description' => $project->description,
            'sku' => $project->sku,
        ];

        // ذخیره وضعیت اولیه برای مقایسه تغییرات
        $this->originalState = $this->editingState;
        $this->originalTags = $this->selectedTags; // ذخیره برچسب‌های اولیه

        $this->confirmingProjectEdit = true;
    }

    public function updateProject()
    {
        $this->authorize('update', $this->editingProject);

        // 1. بررسی تغییرات در فیلدها
        $hasChanges = false;
        $fieldsToCheck = ['title', 'type', 'start_date', 'category', 'custom_category', 'description'];

        foreach ($fieldsToCheck as $field) {
            // اگر فیلد در editingState وجود ندارد، ادامه بده
            if (!array_key_exists($field, $this->editingState)) {
                continue;
            }

            if ($this->editingState[$field] != $this->originalState[$field]) {
                $hasChanges = true;
                break;
            }
        }

        // 2. بررسی تغییرات در برچسب‌ها
        $originalTagIds = array_column($this->originalTags, 'id');
        $currentTagIds = array_column($this->selectedTags, 'id');

        sort($originalTagIds);
        sort($currentTagIds);

        $tagsChanged = $originalTagIds !== $currentTagIds;

        // 3. اگر هیچ تغییری وجود ندارد
        if (!$hasChanges && !$tagsChanged) {
            $this->confirmingProjectEdit = false;
            session()->flash('info', 'هیچ تغییری در طرح ایجاد نشد.');
            return;
        }

        // 4. اعتبارسنجی (فقط فیلدهای لازم)
        $rules = [
            'editingState.title' => 'required|string|max:255',
            'editingState.type' => 'required|in:approved,thesis,personal',
            'editingState.start_date' => 'required|date',
            'editingState.category' => 'required|string',
            'editingState.custom_category' => 'required_if:editingState.category,other|nullable|string|max:255',
            'editingState.description' => 'nullable|string|max:255',];


        $validated = $this->validate($rules);

        // 5. آماده سازی داده‌ها برای ذخیره
        $data = $validated['editingState'];



        // 6. ذخیره تغییرات
        $this->editingProject->update([
            'title' => $data['title'],
            'type' => $data['type'],
            'start_date' => $data['start_date'],
            'category' => $data['category'],
            'custom_category' => $data['custom_category'],
            'description' => $data['description'] ?? null,
        ]);

        // 7. به‌روزرسانی برچسب‌ها
        $tagIds = array_column($this->selectedTags, 'id');
        $this->editingProject->tags()->sync($tagIds);

        $this->confirmingProjectEdit = false;
        session()->flash('message', 'طرح با موفقیت به‌روزرسانی شد.');
    }
    public function confirmProjectDeletion($id)
    {
        $this->projectIdToDelete = $id;
        $this->confirmingProjectDeletion = true;
    }

    public function deleteProject()
    {
        $project = Project::findOrFail($this->projectIdToDelete);
        $this->authorize('delete', $project);

        $project->delete();

        $this->confirmingProjectDeletion = false;
        session()->flash('message', 'طرح با موفقیت حذف شد.');
    }

//    public function render()
//    {
//        $user = Auth::user();
//        $projectsQuery = Project::all();
////
////        // اگر کاربر نقش مدیریتی یا تحقیقاتی ندارد (و سوپر ادمین یا کاربر مانیتورینگ هم نیست)،
////        // فقط پروژه‌هایی را که مالک یا همکار آن‌هاست، نمایش بده
//        if (!$user->hasRole('super_admin') && !$user->hasRole('monitoring_user') && !$user->hasRole('institution_admin') && !$user->hasRole('researcher')) {
//            $projectsQuery->where(function ($query) use ($user) {
//                $query->where('owner_id', $user->id)
//                    ->orWhereHas('collaborators', function ($q) use ($user) {
//                        $q->where('users.id', $user->id);
//                    });
//            });
//        }
////        // نکته: در آینده می‌توانیم این بخش را برای institution_admin و researcher نیز به پروژه‌های همان نهاد محدود کنیم
////
////        $projects = $projectsQuery->get();
////        $institutions = Institution::all();
//        // ما شناسه‌های تمام تیم‌هایی را که کاربر در آن‌ها عضو است، استخراج می‌کنیم
//        $userTeamIds = $user->researchTeams()->pluck('research_teams.id');
//
//        // سپس، تمام پروژه‌های مربوط به این تیم‌ها را نمایش می‌دهیم
////        $projects = Project::whereIn('research_team_id', $userTeamIds)
////            ->with('researchTeam.institution', 'tags', 'owner')
////            ->get();
////        $projectsQuery->where(function ($query) use ($user) {
////                $query->where('owner_id', $user->id)
////                    ->orWhereHas('collaborators', function ($q) use ($user) {
////                        $q->where('users.id', $user->id);
////                    });
////            });
//
//
//        // ما لیست تیم‌های کاربر را برای استفاده در فرم "ایجاد" ارسال می‌کنیم
//        $researchTeams = $user->researchTeams()->get();
//
//        return view('livewire.project-manager', [
//            'projects' => $projectsQuery,
//            'researchTeams' => $researchTeams,
//        ])->layout('layouts.app');
//    }
    public function render()
    {
        $user = Auth::user();

        // Start a new query builder instance, scoped to the current user
        $projectsQuery = Project::query()
            ->with(['researchTeam.institution', 'tags', 'owner']) // Eager load relationships for performance
            ->where(function ($query) use ($user) {
                $query->where('owner_id', $user->id) // Projects owned by the user
                ->orWhereHas('collaborators', function ($q) use ($user) { // Or projects where the user is a collaborator
                    $q->where('users.id', $user->id);
                });
            })
            ->latest(); // Order by the newest projects first

        // [FIXED] Get only the teams where the user is an admin, for the "Create Project" modal
        $researchTeams = $user->researchTeams()->wherePivot('role', 'admin')->get();

        return view('livewire.project-manager', [
            'projects' => $projectsQuery->paginate(10), // Paginate the results
            'researchTeams' => $researchTeams,
        ])->layout('layouts.app');
    }
}
