<?php

namespace App\Livewire;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Illuminate\Support\Facades\Notification;
use Livewire\WithPagination;
use App\Notifications\TaskAssignedNotification;
use Illuminate\Notifications\Notifiable;
use App\Notifications\TaskStatusChangedNotification; // [NEW]
use Hekmatinasser\Verta\Verta;




class ProjectTaskManager extends Component
{
    use AuthorizesRequests;
    use WithPagination; // [NEW]
    use Notifiable;


    public Project $project;

    // Properties for Create/Edit Task Modal
    public bool $confirmingTaskAction = false;
    public ?Task $editingTask = null;
    public array $state = [];
    public array $assigneeIds = [];
    public bool $confirmingTaskCreation = false;

    // Properties for Delete Task Modal
    public ?Task $taskToDelete = null;
    public bool $confirmingTaskDeletion = false;
    // [NEW] برای استفاده از استایل‌های بوت‌استرپ در صفحه‌بندی
    protected $paginationTheme = 'bootstrap';
    // [NEW] Properties for Dependencies
    public array $dependencyIds = [];
    public $allTasksForDependencies = [];

    protected function rules()
    {
        return [
            'state.title' => 'required|string|max:255',
            'state.description' => 'nullable|string',
            'state.priority' => 'required|in:پایین,متوسط,بالا',
            'state.start_date' => 'nullable|date',
            'state.due_date' => 'nullable|date|after_or_equal:state.start_date',
            'assigneeIds' => 'required|array|min:1',
            'assigneeIds.*' => 'exists:users,id',
            'dependencyIds' => 'nullable|array', // [NEW]
            'dependencyIds.*' => 'exists:tasks,id', // [NEW]
        ];
    }

    public function mount(Project $project)
    {
        $this->project = $project->load('collaborators', 'owner');
    }

//    public function confirmTaskAction(?Task $task = null)
//    {
//        $this->resetErrorBag();
//        if ($task && $task->exists) {
//            $this->authorize('update', $task);
//            $this->editingTask = $task;
//            $this->state = $task->toArray();
//            $this->assigneeIds = $task->assignees->pluck('id')->map(fn($id) => (string) $id)->toArray();
//        } else {
//            $this->authorize('create', [Task::class, $this->project]);
//            $this->editingTask = null;
//            $this->state = ['priority' => 'متوسط'];
//            $this->assigneeIds = [];
//        }
//        $this->confirmingTaskAction = true;
//    }
    public function confirmTaskAction(?Task $task = null)
    {
        $this->resetErrorBag();
        if ($task && $task->exists) {
            $this->authorize('update', $task);
            $this->editingTask = $task;
            $this->state = $task->toArray();
            // [FIX] Manually format the dates for the HTML input[type=date]
            if ($task->start_date) {
                $this->state['start_date'] = Verta::instance($task->start_date)->format('Y/m/d');
            }
            if ($task->due_date) {
                $this->state['due_date'] = Verta::instance($task->due_date)->format('Y/m/d');
            }
            $this->assigneeIds = $task->assignees->pluck('id')->map(fn($id) => (string) $id)->toArray();
            $this->dependencyIds = $task->dependencies->pluck('id')->map(fn($id) => (string) $id)->toArray(); // [NEW]
        } else {
            $this->authorize('create', [Task::class, $this->project]);
            $this->editingTask = null;
            $this->state = ['priority' => 'متوسط'];
            $this->assigneeIds = [];
            $this->dependencyIds = []; // [NEW]
        }

        // [NEW] Load all other tasks in the project as potential dependencies
        $this->allTasksForDependencies = $this->project->tasks()
            ->when($this->editingTask, function ($query) {
                // A task cannot depend on itself
                $query->where('id', '!=', $this->editingTask->id);
            })
            ->get();

        $this->confirmingTaskAction = true;
    }
    /**
     * Show the modal for creating a new task.
     */
    public function confirmTaskCreation()
    {
        // $this->authorize('create', [Task::class, $this->project]);
        $this->resetErrorBag();
        $this->editingTask = null;
        $this->state = [
            'priority' => 'متوسط', // مقدار پیش‌فرض
        ];
        $this->assigneeIds = [];
        $this->confirmingTaskCreation = true;
    }

    /**
     * Save or update a task.
     */
//    public function saveTask()
//    {
//        $this->validate();
//
//        if ($this->editingTask) {
//            // Logic for updating a task
//            $this->authorize('update', $this->editingTask);
//            $originalAssigneeIds = $this->editingTask->assignees->pluck('id');
//
//            $this->editingTask->update($this->state);
//            $this->editingTask->assignees()->sync($this->assigneeIds);
//
//            // Send notification only to newly added assignees
//            $newAssigneeIds = collect($this->assigneeIds)->map(fn($id) => (int)$id);
//            $newlyAddedUsers = User::whereIn('id', $newAssigneeIds->diff($originalAssigneeIds))->get();
//            if ($newlyAddedUsers->isNotEmpty()) {
//                Notification::send($newlyAddedUsers, new TaskAssignedNotification($this->editingTask->fresh(), Auth::user()));
//            }
//
//            session()->flash('task_message', 'وظیفه با موفقیت به‌روزرسانی شد.');
//
//        } else {
//            // Logic for creating a new task
//            $this->authorize('create', [Task::class, $this->project]);
//            $taskData = $this->state + ['project_id' => $this->project->id, 'creator_id' => Auth::id()];
//            $task = Task::create($taskData);
//            $task->assignees()->sync($this->assigneeIds);
//
//            // Send notification to all assignees of the new task
//            Notification::send($task->assignees, new TaskAssignedNotification($task, Auth::user()));
//
//            session()->flash('task_message', 'وظیفه جدید با موفقیت ایجاد شد.');
//        }
//
//        $this->confirmingTaskAction = false;
//        $this->editingTask = null;
//
//    }

    public function saveTask()
    {
        // 1. Convert Jalali date strings to Carbon date objects first.
        try {
            $this->state['start_date'] = !empty($this->state['start_date'])
                ? Verta::parse($this->state['start_date'])->toCarbon()
                : null;

            $this->state['due_date'] = !empty($this->state['due_date'])
                ? Verta::parse($this->state['due_date'])->toCarbon()
                : null;
        } catch (\Exception $e) {
            $this->addError('state.start_date', 'فرمت تاریخ وارد شده نامعتبر است. لطفا از تاریخ شمار استفاده کنید.');
            return;
        }
        $validated = $this->validate();
        $taskData = $validated['state'];



        if ($this->editingTask) {
            $this->authorize('update', $this->editingTask);
            $this->editingTask->update($taskData);
            $this->editingTask->assignees()->sync($this->assigneeIds);
            $this->editingTask->dependencies()->sync($this->dependencyIds); // [NEW]
            session()->flash('task_message', 'وظیفه با موفقیت به‌روزرسانی شد.');
        } else {
            $this->authorize('create', [Task::class, $this->project]);
            $taskCreationData = $taskData + ['project_id' => $this->project->id, 'creator_id' => Auth::id()];
            $task = Task::create($taskCreationData);
            $task->assignees()->sync($this->assigneeIds);
            $task->dependencies()->sync($this->dependencyIds); // [NEW]
            session()->flash('task_message', 'وظیفه جدید با موفقیت ایجاد شد.');
        }

        $this->confirmingTaskAction = false;
        $this->editingTask = null;
    }
    public function confirmTaskDeletion(Task $task)
    {
        $this->authorize('delete', $task);
        $this->taskToDelete = $task;
        $this->confirmingTaskDeletion = true;
    }

    public function deleteTask()
    {
        if (!$this->taskToDelete) return;
        $this->authorize('delete', $this->taskToDelete);
        $this->taskToDelete->delete();

        $this->confirmingTaskDeletion = false;
        $this->taskToDelete = null; // [FIX] پاکسازی وضعیت برای جلوگیری از خطاهای آینده
        session()->flash('task_message', 'وظیفه با موفقیت حذف شد.');
    }

    /**
     * Change the status of a task.
     */
    public function changeStatus(Task $task, $status)
    {
        $this->authorize('changeStatus', $task);
        if (in_array($status, ['برای انجام', 'در حال انجام', 'انجام شده'])) {

            $oldStatus = $task->status;
            $changer = Auth::user();

            $task->update([
                'status' => $status,
                'completed_at' => $status === 'انجام شده' ? now() : null,
            ]);

            // [NEW LOGIC]
            // بارگذاری روابط لازم برای جلوگیری از کوئری‌های اضافه
            $task->load(['assignees', 'creator', 'project.owner']);

            // 1. جمع‌آوری تمام دریافت‌کنندگان بالقوه
            $recipients = collect($task->assignees);
            $recipients->push($task->creator);
            $recipients->push($task->project->owner);

            // 2. حذف کاربری که وضعیت را تغییر داده و حذف موارد تکراری
            $uniqueRecipients = $recipients->unique('id')
                ->filter(function ($user) use ($changer) {
                    return $user && $user->id !== $changer->id;
                });

            // 3. ارسال اعلان‌ها
            if ($uniqueRecipients->isNotEmpty()) {
                Notification::send($uniqueRecipients, new TaskStatusChangedNotification($task->fresh(), $changer, $oldStatus));
            }

            session()->flash('task_message', 'وضعیت وظیفه به‌روزرسانی شد.');
        }
    }
    /**
     * [NEW] Dispatch an event to open the task detail modal.
     */
    public function showTask(Task $task)
    {
        $this->dispatch('showTaskModal', task: $task);
    }

    public function render()
    {
        $tasks = $this->project->tasks()->with(['assignees', 'creator'])
            ->latest()->paginate(5); // [UPDATED]

        return view('livewire.project-task-manager', [
            'tasks' => $tasks,
        ]);
    }
}
