<?php

namespace App\Livewire;

use App\Models\Auditor;
use App\Models\Equipment;
use App\Models\StorageLocation;
use App\Models\TemperatureLog;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;

class PublicTempLogger extends Component
{
    public ?StorageLocation $room = null;
    public ?Auditor $auditor = null;

    // Step 1: PIN Authentication
    public string $pin = '';
    public bool $pinVerified = false;

    // Step 2: Logging
    public ?Equipment $selectedEquipment = null;
    public string $temperature = '';
    public string $notes = '';

    public function mount($uuid)
    {
        $this->room = StorageLocation::where('uuid', $uuid)->where('location_type', 'room')->firstOrFail();
    }

    public function verifyPin()
    {
        $this->validate(['pin' => 'required|digits:4']);

        $auditors = Auditor::where('is_active', true)->get();

        foreach ($auditors as $auditor) {
            if (Hash::check($this->pin, $auditor->pin)) {
                $this->auditor = $auditor;
                $this->pinVerified = true;
                $this->reset('pin');
                return;
            }
        }

        $this->addError('pin', 'پین کد وارد شده نامعتبر یا بازرس غیرفعال است.');
    }

    public function selectEquipment(Equipment $equipment)
    {
        $this->selectedEquipment = $equipment;
        $this->reset(['temperature', 'notes']);
    }

    public function saveTemperatureLog()
    {
        if (!$this->pinVerified || !$this->selectedEquipment) {
            return;
        }

        $this->validate([
            'temperature' => 'required|numeric',
            'notes' => 'nullable|string|max:500',
        ]);

        TemperatureLog::create([
            'equipment_id' => $this->selectedEquipment->id,
            'auditor_id' => $this->auditor->id,
            'temperature' => $this->temperature,
            'notes' => $this->notes,
        ]);

        session()->flash('log_success', 'دمای تجهیز ' . $this->selectedEquipment->name . ' با موفقیت ثبت شد.');
        $this->reset(['selectedEquipment', 'temperature', 'notes']);
    }

    public function logout()
    {
        // Store the UUID before resetting properties
        $roomUuid = $this->room->uuid;

        // Reset all public properties
        $this->reset();

        // Re-mount the component with the saved UUID
        $this->mount($roomUuid);
    }

    public function render()
    {
        $equipmentList = [];
        if ($this->pinVerified) {
            $equipmentList = Equipment::where('storage_location_id', $this->room->id)
                ->where('requires_temp_log', true)
                ->get();
        }

        return view('livewire.public-temp-logger', [
            'equipmentList' => $equipmentList,
        ])->layout('layouts.guest'); // Using a guest layout without sidebar/navbar
    }
}
