<?php


namespace App\Livewire;

use App\Models\ResearchTeam;
use App\Models\User;
use App\Models\TeamInvitation;
use App\Models\LabNotebookTemplate;
use App\Notifications\TeamInvitationNotification;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Livewire\Component;
use App\Models\TeamAnnouncement;
use App\Models\OnboardingChecklistItem;
use Illuminate\Support\Facades\Storage;
use App\Actions\Fortify\HandlesOnboardingTasks;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Spatie\Permission\PermissionRegistrar;
use Livewire\WithFileUploads; // [NEW]

class ResearchTeamDashboard extends Component
{
    use HandlesOnboardingTasks, WithFileUploads;
    public $managedTeams;
    public ?ResearchTeam $activeTeam = null;

    // Properties for assigning calibration manager
    public $selectedCalibrationManagerId;
    public $selectedPurchasingManagerId;
    // [REWRITTEN] Properties for the new Add/Invite System
    public bool $confirmingMemberAddition = false;
    public string $userSearch = '';
    public $userSearchResults = [];
    public string $roleToAdd = 'member';

    public ?User $memberToRemove = null;
    public bool $confirmingMemberRemoval = false;

    public ?User $editingMember = null;
    public bool $confirmingRoleChange = false;
    public string $newRoleForMember = '';
    // [NEW] Properties for Viewing Member Profile
    public ?User $viewingMember = null;
    public bool $confirmingMemberView = false;
    // [NEW] Properties for Team Settings
    public bool $confirmingTeamSettings = false;
    public array $teamState = [];
    // [NEW] Properties for Team Announcements
    public bool $confirmingAnnouncementAction = false;
    public ?TeamAnnouncement $editingAnnouncement = null;
    public array $announcementState = [];

    public ?TeamAnnouncement $announcementToDelete = null;
    public bool $confirmingAnnouncementDeletion = false;
    // [NEW] Properties for Onboarding Checklist
    public string $newChecklistItem = '';
    // [NEW] Properties for Permission Management
    public bool $confirmingPermissionEdit = false;
    public ?User $editingPermissionsFor = null;
    public $allTeamPermissions = [];
    public array $memberPermissions = [];
    // [NEW] Properties for Announcement Attachment
    public $announcementAttachment;
    public $existingAttachmentName;
    // [NEW] Properties for Lab Notebook Templates
    public bool $confirmingTemplateAction = false;
    public ?LabNotebookTemplate $editingTemplate = null;
    public array $templateState = [];
    public ?LabNotebookTemplate $templateToDelete = null;
    public bool $confirmingTemplateDeletion = false;
    // [NEW] Property for Warehouse Manager Card
    public $selectedWarehouseManagerId;
    public ?User $warehouseManager = null;

    protected function rules()
    {
        return [
            'userSearch' => 'required|string',
            'roleToAdd' => 'required|string|in:researcher,member',
        ];
    }

    public function mount()
    {
        $this->managedTeams = Auth::user()->researchTeams()->wherePivot('role', 'admin')->get();

        if ($this->managedTeams->isNotEmpty()) {
            $this->selectTeam($this->managedTeams->first()->id);
        }
    }

    public function selectTeam($teamId)
    {
        $team = $this->managedTeams->find($teamId);
        if ($team) {
            // [UPDATED] Load invitations along with other relations
            $this->activeTeam = $team->load('users', 'calibrationManager', 'purchasingManager', 'warehouseManager', 'invitations', 'projects.tasks.assignees', 'announcements.user', 'onboardingChecklistItems', 'labNotebookTemplates.creator');
            $this->selectedCalibrationManagerId = $this->activeTeam->calibration_manager_id;
            $this->selectedPurchasingManagerId = $this->activeTeam->purchasing_manager_id;
            $this->selectedWarehouseManagerId = $this->activeTeam->warehouse_manager_id;
        }
    }

    public function assignCalibrationManager()
    {
        if (!$this->activeTeam) return;

        $this->validate(['selectedCalibrationManagerId' => 'nullable|exists:users,id']);

        $newManagerId = $this->selectedCalibrationManagerId ?: null;

        // Ensure the selected user is a member of the team
        if ($newManagerId && !$this->activeTeam->users->contains('id', $newManagerId)) {
            session()->flash('calibration_error', 'کاربر انتخاب شده عضو این تیم نیست.');
            return;
        }
        $teamId = $this->activeTeam->id;
        $roleName = "calibration_manager-Team-{$teamId}";

        DB::transaction(function () use ($newManagerId, $roleName, $teamId) {
            // Find or create the team-specific role
            $calibrationManagerRole = Role::firstOrCreate(['name' => $roleName, 'guard_name' => 'web']);

            // Define and sync team-specific permissions for this role
            $permissions = [
                "calibration-request-approve-team-{$teamId}",
                "calibration-request-edit-team-{$teamId}",
                "calibration-request-reject-team-{$teamId}",
                "calibration-request-finalize-offers-team-{$teamId}",
            ];
            foreach ($permissions as $permissionName) {
                Permission::firstOrCreate(['name' => $permissionName, 'guard_name' => 'web']);
            }
            $calibrationManagerRole->syncPermissions($permissions);

            // Find the current manager for THIS TEAM
            $currentManager = $this->activeTeam->calibrationManager;

            // 1. Remove the team-specific role from the old manager
            if ($currentManager) {
                $currentManager->removeRole($calibrationManagerRole);
            }

            // 2. Assign the team-specific role to the new manager
            if ($newManagerId && $newManager = User::find($newManagerId)) {
                $newManager->assignRole($calibrationManagerRole);
            }

            // 3. Update the team's warehouse_manager_id
            $this->activeTeam->update(['calibration_manager_id' => $newManagerId]);
        });
        // Clear the permission cache to reflect changes immediately
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        session()->flash('calibration_message', 'مسئول تجهیزات و کالیبراسیون با موفقیت برای این تیم تعیین شد.');
        $this->selectTeam($this->activeTeam->id); // Refresh data
    }

    public function assignPurchasingManager()
    {
        if (!$this->activeTeam) return;

        $this->validate(['selectedPurchasingManagerId' => 'nullable|exists:users,id']);

        $newManagerId = $this->selectedPurchasingManagerId ?: null;
//        $oldManagerId = $this->activeTeam->purchasing_manager_id;

        // Ensure the selected user is a member of the team
        if ($newManagerId && !$this->activeTeam->users->contains('id', $newManagerId)) {
            session()->flash('purchasing_error', 'کاربر انتخاب شده عضو این تیم نیست.');
            return;
        }
        $teamId = $this->activeTeam->id;
        $roleName = "purchasing_manager-Team-{$teamId}";

        DB::transaction(function () use ($newManagerId, $roleName, $teamId) {
            // Find or create the team-specific role
            $purchasingManagerRole = Role::firstOrCreate(['name' => $roleName, 'guard_name' => 'web']);

            // Define and sync team-specific permissions for this role
            $permissions = [
                "purchase-request-approve-team-{$teamId}",
                "purchase-request-edit-team-{$teamId}",
                "purchase-request-reject-team-{$teamId}",
                "purchase-request-finalize-offers-team-{$teamId}",
            ];
            foreach ($permissions as $permissionName) {
                Permission::firstOrCreate(['name' => $permissionName, 'guard_name' => 'web']);
            }
            $purchasingManagerRole->syncPermissions($permissions);

            // Find the current manager for THIS TEAM
            $currentManager = $this->activeTeam->warehouseManager;

            // 1. Remove the team-specific role from the old manager
            if ($currentManager) {
                $currentManager->removeRole($purchasingManagerRole);
            }

            // 2. Assign the team-specific role to the new manager
            if ($newManagerId && $newManager = User::find($newManagerId)) {
                $newManager->assignRole($purchasingManagerRole);
            }

            // 3. Update the team's warehouse_manager_id
            $this->activeTeam->update(['purchasing_manager_id' => $newManagerId]);
        });
        // Clear the permission cache to reflect changes immediately
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        session()->flash('purchasing_message', 'مسئول خرید با موفقیت برای این تیم تعیین شد.');
        $this->selectTeam($this->activeTeam->id); // Refresh data
    }

    /**
     * [REFACTORED] Simplified method for assigning the warehouse manager.
     */
    public function assignWarehouseManager()
    {
        if (!$this->activeTeam) return;

        $this->validate(['selectedWarehouseManagerId' => 'nullable|exists:users,id']);

        $newManagerId = $this->selectedWarehouseManagerId ?: null;

        if ($newManagerId && !$this->activeTeam->users->contains('id', $newManagerId)) {
            session()->flash('warehouse_error', 'کاربر انتخاب شده عضو این تیم نیست.');
            return;
        }

        $teamId = $this->activeTeam->id;
        $roleName = "Warehouse_Manager-Team-{$teamId}";

        DB::transaction(function () use ($newManagerId, $roleName, $teamId) {
            // Find or create the team-specific role
            $warehouseManagerRole = Role::firstOrCreate(['name' => $roleName, 'guard_name' => 'web']);

            // Define and sync team-specific permissions for this role
            $permissions = [
                "manage_inventory-team-{$teamId}",
                "manage_storage_locations-team-{$teamId}",
                "manage_warehouse_requests-team-{$teamId}",
            ];
            foreach ($permissions as $permissionName) {
                Permission::firstOrCreate(['name' => $permissionName, 'guard_name' => 'web']);
            }
            $warehouseManagerRole->syncPermissions($permissions);

            // Find the current manager for THIS TEAM
            $currentManager = $this->activeTeam->warehouseManager;

            // 1. Remove the team-specific role from the old manager
            if ($currentManager) {
                $currentManager->removeRole($warehouseManagerRole);
            }

            // 2. Assign the team-specific role to the new manager
            if ($newManagerId && $newManager = User::find($newManagerId)) {
                $newManager->assignRole($warehouseManagerRole);
            }

            // 3. Update the team's warehouse_manager_id
            $this->activeTeam->update(['warehouse_manager_id' => $newManagerId]);
        });

        // Clear the permission cache to reflect changes immediately
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        session()->flash('warehouse_message', 'مسئول انبار با موفقیت تعیین شد.');

        // Refresh the component state
        $this->selectTeam($this->activeTeam->id);
    }

    /**
     * Show the modal for adding or inviting a new member.
     */
    public function confirmMemberAddition()
    {
        $this->resetErrorBag();
        $this->reset(['userSearch', 'userSearchResults', 'roleToAdd']);
        $this->roleToAdd = 'member';
        $this->confirmingMemberAddition = true;
    }

    /**
     * Live search for existing users to add to the team.
     */
    public function updatedUserSearch($value)
    {
        if (strlen($this->userSearch) < 2) {
            $this->userSearchResults = [];
            return;
        }
        $existingMemberIds = $this->activeTeam->users->pluck('id');
        $this->userSearchResults = User::whereNotIn('id', $existingMemberIds)
            ->where(function ($query) {
                $query->where('email', 'like', '%' . $this->userSearch . '%')
                    ->orWhere('username', 'like', '%' . $this->userSearch . '%')
                    ->orWhere('first_name', 'like', '%' . $this->userSearch . '%')
                    ->orWhere('last_name', 'like', '%' . $this->userSearch . '%');
            })
            ->take(5)
            ->get();
    }

    /**
     * Add an existing user directly to the team.
     */
    public function addMember(User $user)
    {
        // 1. افزودن کاربر به تیم
        $this->activeTeam->users()->attach($user->id, ['role' => $this->roleToAdd]);

        // 2. [NEW] ایجاد وظایف چک‌لیست برای کاربر جدید
        $this->createOnboardingTasks($user, $this->activeTeam);

        $this->confirmingMemberAddition = false;
        session()->flash('member_message', 'عضو جدید با موفقیت به تیم اضافه شد و وظایف اولیه برای او ایجاد گردید.');
        $this->selectTeam($this->activeTeam->id);
    }

    /**
     * [NEW] Invite a new member by email if they are not found in the system.
     */
    public function inviteMemberByEmail()
    {
        $this->validate([
            'userSearch' => 'required|email',
            'roleToAdd' => 'required|string|in:researcher,member',
        ]);

        $invitationEmail = $this->userSearch;

        // Check if user is already in the team
        if ($this->activeTeam->users()->where('email', $invitationEmail)->exists()) {
            $this->addError('userSearch', 'این کاربر از قبل عضو تیم است.');
            return;
        }

        // Check if an invitation already exists
        if ($this->activeTeam->invitations()->where('email', $invitationEmail)->exists()) {
            $this->addError('userSearch', 'یک دعوت‌نامه فعال برای این ایمیل از قبل ارسال شده است.');
            return;
        }

        $invitation = $this->activeTeam->invitations()->create([
            'email' => $invitationEmail,
            'role' => $this->roleToAdd,
            'token' => Str::random(40),
            'expires_at' => now()->addDays(7),
        ]);

        Notification::route('mail', $invitationEmail)
            ->notify(new TeamInvitationNotification($invitation));

        $this->confirmingMemberAddition = false;
        session()->flash('member_message', 'دعوت‌نامه با موفقیت ارسال شد.');
        $this->selectTeam($this->activeTeam->id);
    }

    public function cancelInvitation(TeamInvitation $invitation)
    {
        $invitation->delete();
        session()->flash('member_message', 'دعوت‌نامه با موفقیت لغو شد.');
        $this->selectTeam($this->activeTeam->id);
    }


    public function confirmMemberRemoval(User $user)
    {
        $this->memberToRemove = $user;
        $this->confirmingMemberRemoval = true;
    }

    public function removeMember()
    {
        if ($this->memberToRemove) {
            $this->activeTeam->users()->detach($this->memberToRemove->id);
            $this->confirmingMemberRemoval = false;
            $this->memberToRemove = null;
            session()->flash('member_message', 'عضو با موفقیت از تیم حذف شد.');
            $this->selectTeam($this->activeTeam->id);
        }
    }

    /**
     * [NEW] Show the modal for changing a member's role.
     */
    public function confirmRoleChange(User $member)
    {
        $this->resetErrorBag();

        // [FIX] Re-fetch the member from the relationship to ensure pivot data is loaded.
        $memberWithPivot = $this->activeTeam->users()->find($member->id);


        if ($memberWithPivot) {
            // [NEW] جلوگیری از تغییر نقش مدیر
            if ($memberWithPivot->pivot->role === 'admin') {
                session()->flash('member_error', 'امکان تغییر نقش مدیر تیم وجود ندارد.');
                return;
            }
            $this->editingMember = $memberWithPivot;
            $this->newRoleForMember = $memberWithPivot->pivot->role;
            $this->confirmingRoleChange = true;
        } else {
            session()->flash('error', 'کاربر مورد نظر در این تیم یافت نشد.');
        }
    }

    /**
     * [NEW] Update the role for the selected member.
     */
    public function updateMemberRole()
    {
        if (!$this->editingMember) return;

        $this->validate(['newRoleForMember' => 'required|string|in:researcher,member']);

        $this->activeTeam->users()->updateExistingPivot($this->editingMember->id, [
            'role' => $this->newRoleForMember,
        ]);

        $this->confirmingRoleChange = false;
        $this->editingMember = null;
        session()->flash('member_message', 'نقش عضو با موفقیت به‌روزرسانی شد.');
        $this->selectTeam($this->activeTeam->id);
    }

    public function viewMember(User $member)
    {
        // Load system-wide roles for the user
        $this->viewingMember = $member->load('roles');
        $this->confirmingMemberView = true;
    }
    /**
     * [NEW] A helper function to calculate and format workload chart data.
     */
    public function getWorkloadChartData(): array
    {
        if (!$this->activeTeam) {
            return [];
        }

        $workload = [];
        $members = $this->activeTeam->users;
        $projectIds = $this->activeTeam->projects->pluck('id');

        foreach ($members as $member) {
            $taskCount = $member->assignedTasks()->whereIn('project_id', $projectIds)->count();
            $workload[$member->name] = $taskCount;
        }

        return [
            'labels' => array_keys($workload),
            'datasets' => [
                [
                    'label' => 'تعداد وظایف محول شده',
                    'data' => array_values($workload),
                    'backgroundColor' => '#4f46e5', // indigo-600
                ]
            ],
        ];
    }

    /**
     * [NEW] A helper function to get the aggregated consumed resources report.
     */
    public function getConsumedResourcesReport(): array
    {
        if (!$this->activeTeam) {
            return [];
        }

        $report = [];
        // Iterate through each project of the team
        foreach ($this->activeTeam->projects as $project) {
            // Iterate through each budgeted item in the project
            foreach ($project->budgetItems as $item) {
                if ($item->pivot->total_consumed_quantity > 0) {
                    if (!isset($report[$item->id])) {
                        // Initialize if it's the first time we see this item
                        $report[$item->id] = [
                            'name' => $item->name,
                            'unit' => $item->unit,
                            'current_stock' => $item->quantity, // [NEW] Add current stock
                            'total_consumed' => 0,
                        ];
                    }
                    // Add the consumed quantity to the total
                    $report[$item->id]['total_consumed'] += $item->pivot->total_consumed_quantity;
                }
            }
        }
        return $report;
    }
    /**
     * [NEW] Show the modal for editing team settings.
     */
    public function confirmTeamSettings()
    {
        $this->resetErrorBag();
        $this->teamState = [
            'name' => $this->activeTeam->name,
            'description' => $this->activeTeam->description,
        ];
        $this->confirmingTeamSettings = true;
    }

    /**
     * [NEW] Save the updated team settings.
     */
    public function saveTeamSettings()
    {
        $this->validate([
            'teamState.name' => 'required|string|max:255',
            'teamState.description' => 'nullable|string',
        ]);

        $this->activeTeam->update($this->teamState);

        $this->confirmingTeamSettings = false;
        session()->flash('settings_message', 'اطلاعات تیم با موفقیت به‌روزرسانی شد.');
        $this->selectTeam($this->activeTeam->id); // Refresh data
    }
    /**
     * [NEW] Show the modal for creating or editing a team announcement.
     */
    public function confirmAnnouncementAction(?TeamAnnouncement $announcement = null)
    {
        $this->resetErrorBag();
        $this->reset(['announcementAttachment', 'existingAttachmentName']);
        if ($announcement && $announcement->exists) {
            // $this->authorize('update', $announcement);
            $this->editingAnnouncement = $announcement;
            $this->announcementState = $announcement->toArray();
            $this->existingAttachmentName = $announcement->attachment_name;
        } else {
            // $this->authorize('create', [TeamAnnouncement::class, $this->activeTeam]);
            $this->editingAnnouncement = null;
            $this->announcementState = [
                'level' => 'info',
                'is_pinned' => false,
            ];
        }
        $this->confirmingAnnouncementAction = true;
    }

    /**
     * [NEW] Save the new or updated team announcement.
     */
    public function saveAnnouncement()
    {
        $this->validate([
            'announcementState.title' => 'required|string|max:255',
            'announcementState.body' => 'required|string',
            'announcementState.level' => 'required|in:info,warning,danger',
            'announcementState.is_pinned' => 'nullable|boolean',
            'announcementAttachment' => 'nullable|file|max:10240', // 10MB Max
        ]);
        // [NEW] Handle file upload
        if ($this->announcementAttachment) {
            // Delete old file if it exists
            if ($this->editingAnnouncement && $this->editingAnnouncement->attachment_path) {
                Storage::disk('public')->delete($this->editingAnnouncement->attachment_path);
            }
            // Store new file
            $path = $this->announcementAttachment->store('team_announcements', 'public');
            $this->announcementState['attachment_path'] = $path;
            $this->announcementState['attachment_name'] = $this->announcementAttachment->getClientOriginalName();
            $this->announcementState['attachment_size'] = $this->announcementAttachment->getSize();
        }

        if ($this->editingAnnouncement) {
            // $this->authorize('update', $this->editingAnnouncement);
            $this->editingAnnouncement->update($this->announcementState);
            session()->flash('announcement_message', 'اطلاعیه با موفقیت به‌روزرسانی شد.');
        } else {
            // $this->authorize('create', [TeamAnnouncement::class, $this->activeTeam]);
            $this->activeTeam->announcements()->create($this->announcementState + ['user_id' => Auth::id()]);
            session()->flash('announcement_message', 'اطلاعیه جدید با موفقیت ایجاد شد.');
        }

        $this->confirmingAnnouncementAction = false;
        $this->selectTeam($this->activeTeam->id);
    }

    /**
     * [NEW] Remove the attachment from an announcement.
     */
    public function removeAttachment()
    {
        if ($this->editingAnnouncement && $this->editingAnnouncement->attachment_path) {
            Storage::disk('public')->delete($this->editingAnnouncement->attachment_path);
            $this->editingAnnouncement->update([
                'attachment_path' => null,
                'attachment_name' => null,
                'attachment_size' => null,
            ]);
            $this->existingAttachmentName = null;
            session()->flash('announcement_message', 'فایل پیوست با موفقیت حذف شد.');
            $this->selectTeam($this->activeTeam->id);
        }
    }

    /**
     * [NEW] Show the confirmation modal for deleting an announcement.
     */
    public function confirmAnnouncementDeletion(TeamAnnouncement $announcement)
    {
        // $this->authorize('delete', $announcement);
        $this->announcementToDelete = $announcement;
        $this->confirmingAnnouncementDeletion = true;
    }

    /**
     * [NEW] Delete the selected announcement.
     */
    public function deleteAnnouncement()
    {
        if (!$this->announcementToDelete) return;
        // $this->authorize('delete', $this->announcementToDelete);

        $this->announcementToDelete->delete();
        $this->confirmingAnnouncementDeletion = false;
        session()->flash('announcement_message', 'اطلاعیه با موفقیت حذف شد.');
        $this->selectTeam($this->activeTeam->id);
    }
    /**
     * [NEW] Add a new item to the onboarding checklist.
     */
    public function addChecklistItem()
    {
        $this->validate(['newChecklistItem' => 'required|string|max:255']);

        $this->activeTeam->onboardingChecklistItems()->create([
            'title' => $this->newChecklistItem,
            'order' => $this->activeTeam->onboardingChecklistItems()->count() + 1,
        ]);

        $this->newChecklistItem = '';
        $this->activeTeam->load('onboardingChecklistItems'); // Refresh the list
    }

    /**
     * [NEW] Remove an item from the onboarding checklist.
     */
    public function removeChecklistItem($itemId)
    {
        $item = $this->activeTeam->onboardingChecklistItems()->find($itemId);
        if ($item) {
            $item->delete();
            $this->activeTeam->load('onboardingChecklistItems'); // Refresh the list
        }
    }

    /**
     * [NEW] Update the order of checklist items after drag-and-drop.
     */
    public function updateChecklistOrder($orderedIds)
    {
        foreach ($orderedIds as $item) {
            OnboardingChecklistItem::find($item['value'])->update(['order' => $item['order']]);
        }
        $this->activeTeam->load('onboardingChecklistItems'); // Refresh the list
    }

    /**
     * [NEW] Show the modal for editing a member's permissions.
     */
    public function confirmPermissionEdit(User $member)
    {
        $this->resetErrorBag();
        $this->editingPermissionsFor = $member;

        // واکشی تمام مجوزهایی که با "team:" شروع می‌شوند
        $this->allTeamPermissions = Permission::where('name', 'like', 'team:%')->get();

        // واکشی مجوزهای فعلی کاربر در این تیم
        $this->memberPermissions = $member->teamPermissions()
            ->where('research_team_id', $this->activeTeam->id)
            ->pluck('permission_id')->map(fn($id) => (string) $id)->toArray();

        $this->confirmingPermissionEdit = true;
    }

    /**
     * [NEW] Update the custom permissions for the selected member in this team.
     */
    public function updateMemberPermissions()
    {
        if (!$this->editingPermissionsFor) return;

        $permissionsToSync = collect($this->memberPermissions)->mapWithKeys(fn($permissionId) => [
            $permissionId => ['research_team_id' => $this->activeTeam->id]
        ]);

        // همگام‌سازی مجوزهای کاربر در این تیم
        $this->editingPermissionsFor->teamPermissions()->wherePivot('research_team_id', $this->activeTeam->id)->sync($permissionsToSync);

        $this->confirmingPermissionEdit = false;
        session()->flash('member_message', 'مجوزهای عضو با موفقیت به‌روزرسانی شد.');
    }

    /**
     * [NEW] Show the modal for creating or editing a lab notebook template.
     */
    public function confirmTemplateAction(?LabNotebookTemplate $template = null)
    {
        $this->resetErrorBag();
        if ($template && $template->exists) {
            $this->editingTemplate = $template;
            $this->templateState = $template->toArray();
        } else {
            $this->editingTemplate = null;
            $this->templateState = [];
        }
        $this->confirmingTemplateAction = true;
        // Dispatch event to initialize Jodit editor in the modal
        $this->dispatch('init-template-editor', content: $this->templateState['body'] ?? '');
    }

    /**
     * [NEW] Save the new or updated lab notebook template.
     */
    public function saveTemplate()
    {
        $this->validate([
            'templateState.title' => 'required|string|max:255',
            'templateState.description' => 'nullable|string',
            'templateState.body' => 'required|string',
        ]);

        if ($this->editingTemplate) {
            $this->editingTemplate->update($this->templateState);
            session()->flash('template_message', 'قالب با موفقیت به‌روزرسانی شد.');
        } else {
            $this->activeTeam->labNotebookTemplates()->create($this->templateState + ['creator_id' => Auth::id()]);
            session()->flash('template_message', 'قالب جدید با موفقیت ایجاد شد.');
        }

        $this->confirmingTemplateAction = false;
        $this->selectTeam($this->activeTeam->id); // Refresh data
    }

    /**
     * [NEW] Show the confirmation modal for deleting a template.
     */
    public function confirmTemplateDeletion(LabNotebookTemplate $template)
    {
        $this->templateToDelete = $template;
        $this->confirmingTemplateDeletion = true;
    }

    /**
     * [NEW] Delete the selected template.
     */
    public function deleteTemplate()
    {
        if (!$this->templateToDelete) return;

        $this->templateToDelete->delete();
        $this->confirmingTemplateDeletion = false;
        session()->flash('template_message', 'قالب با موفقیت حذف شد.');
        $this->selectTeam($this->activeTeam->id); // Refresh data
    }

    public function render()
    {
        return view('livewire.research-team-dashboard', [
            // [NEW] Pass the report data to the view
            'consumedResourcesReport' => $this->getConsumedResourcesReport(),
            'warehouseManager' => $this->warehouseManager,
        ])->layout('layouts.app');
    }
}
