<?php

namespace App\Livewire;

use App\Models\ResearchTeam;
use App\Models\StorageLocation;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Livewire\Component;


class StorageLocationManager extends Component
{
    public $teams = [];
    public $locations;

    // Properties for the "Create Location" modal
    public $confirmingLocationCreation = false;
    public ?StorageLocation $parentLocation = null;
    public $newLocationName;
    public $newLocationType;
    public $allowedLocationTypes = [];

    // Properties for the "Edit Location" modal
    public ?StorageLocation $editingLocation = null;
    public $confirmingLocationEdit = false;
    public $editingLocationName;

    // Properties for the "Delete Location" modal
    public $locationIdToDelete;
    public $confirmingLocationDeletion = false;
    public $selected_team_for_new_location;
    public $generatedLocationCode = '';
// Properties for Grid Containers (Box/Rack)
    public $rows;
    public $columns;
    // Properties for auto-generating children
    public $subLocationCount = 1;
    public $subLocationBaseName = '';
    public $canHaveAutoChildren = false;

    // Properties for Move Modal
    public ?StorageLocation $movingLocation = null;
    public $confirmingLocationMove = false;
    public $newParentId;
    public $potentialNewParents = [];
    // Properties for Bulk Actions
    public $selectedLocations = [];
    public $selectAll = false;

    // The hierarchical rules for locations
    protected const HIERARCHY_RULES = [
        'building' => ['room'],
        'room' => ['freezer', 'refrigerator', 'shelf'],
        'freezer' => ['floor'],
        'refrigerator' => ['floor'],
        'tank' => ['canister'],
        'shelf' => ['rack', 'box'],
        'floor' => ['rack', 'box'],
        'canister' => ['rack', 'box', 'cane'],
        'rack' => [],
        'box' => [],
        'tube' => [],
        'cane'=> [],
    ];
    // Rules for auto-generating children
    protected const AUTO_CHILD_RULES = [
        'freezer' => 'طبقه',
        'refrigerator' => 'طبقه',
        'shelf' => 'طبقه',
        'floor' => 'رک', // <-- قانون جدید: هنگام ایجاد طبقه، می‌توان رک ساخت

//        'rack' => 'باکس',
    ];

    public function mount()
    {
        $user = Auth::user();
        $this->teams = $user->researchTeams()->get(); // بارگذاری تیم‌های کاربر
        $this->loadLocations();
    }

//    public function loadLocations()
//    {
//        $user = Auth::user();
//        $userTeamIds = $user->researchTeams()->pluck('research_teams.id');
//
//        // We fetch all locations for the user's teams and build a tree structure.
//        $allLocations = StorageLocation::whereIn('research_team_id', $userTeamIds)
//            ->with('children') // Eager load children
//            ->get();
//
//        // We only pass the top-level locations (roots) to the view.
//        $this->locations = $allLocations->whereNull('parent_id');
//    }

    public function loadLocations()
    {
        $user = Auth::user();
//        $this->teams = $user->researchTeams()->get();
        $userTeamIds = $this->teams->pluck('id');

        $allLocations = StorageLocation::whereIn('research_team_id', $userTeamIds)
            ->with(['children'=> function ($query) {
        $query->with('children'); // Load grandchildren for path updates
    }])
            ->get();

        $this->locations = $allLocations->whereNull('parent_id');
    }

    /**
     * Show the modal for creating a new location.
     * If a parentId is provided, it's for a child location.
     */
//    public function confirmLocationCreation($parentId = null)
//    {
//        $this->resetErrorBag();
//        $this->reset(['newLocationName', 'newLocationType', 'parentLocation', 'allowedLocationTypes']);
//
//        if ($parentId) {
//            $this->parentLocation = StorageLocation::find($parentId);
//            // Determine allowed child types based on the parent's type
//            $this->allowedLocationTypes = self::HIERARCHY_RULES[$this->parentLocation->location_type] ?? [];
//        } else {
//            // For top-level locations, only 'building' is allowed.
//            $this->allowedLocationTypes = ['building'];
//        }
//
//        $this->confirmingLocationCreation = true;
//    }

    public function confirmLocationCreation($parentId = null)
    {
        $this->resetErrorBag();
        $this->reset(['newLocationName', 'newLocationType', 'parentLocation', 'allowedLocationTypes', 'selected_team_for_new_location', 'generatedLocationCode', 'rows', 'columns', 'subLocationCount', 'subLocationBaseName', 'canHaveAutoChildren']);
        if ($parentId) {
            $this->parentLocation = StorageLocation::find($parentId);
            $this->allowedLocationTypes = self::HIERARCHY_RULES[$this->parentLocation->location_type] ?? [];
        } else {
            $this->allowedLocationTypes = ['building'];
        }

        $this->confirmingLocationCreation = true;
    }

    /**
     * This method runs when the user selects a new location type.
     */
    public function updatedNewLocationType($type)
    {
        if (array_key_exists($type, self::AUTO_CHILD_RULES)) {
            $this->canHaveAutoChildren = true;
            $this->subLocationBaseName = self::AUTO_CHILD_RULES[$type];
        } else {
            $this->canHaveAutoChildren = false;
        }
    }

    private function generateLocationCode(ResearchTeam $team): string
    {
        do {
            $randomNumber = str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT);
            $code = ($team->research_teams_code ?? 'LOC') . '-' . $randomNumber;
        } while (StorageLocation::where('location_code', $code)->exists());

        return $code;
    }
    /**
     * Create a new storage location.
     */
    public function saveLocation()
    {
        $rules = [
            'newLocationName' => 'required|string|max:255',
            'newLocationType' => 'required|in:' . implode(',', array_keys(self::HIERARCHY_RULES)),
        ];

        if (!$this->parentLocation) {
            $rules['selected_team_for_new_location'] = 'required|exists:research_teams,id';
        }

        if ($this->canHaveAutoChildren) {
            $rules['subLocationCount'] = 'required|integer|min:1|max:50';
        }

        if (in_array($this->newLocationType, ['tube', 'rack'])) {
            $rules['rows'] = 'required|integer|min:1';
            $rules['columns'] = 'required|integer|min:1';
        }

        $this->validate($rules);

        DB::transaction(function () {
            $teamId = $this->parentLocation ? $this->parentLocation->research_team_id : $this->selected_team_for_new_location;
            $team = ResearchTeam::find($teamId);

            do {
                $code = ($team->research_teams_code ?? 'LOC') . '-' . str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT);
            } while (StorageLocation::where('location_code', $code)->exists());

            $parentLocation = StorageLocation::create([
                'name' => $this->newLocationName,
                'location_code' => $code,
                'location_type' => $this->newLocationType,
                'parent_id' => $this->parentLocation?->id,
                'research_team_id' => $teamId,
                'rows' => $this->rows,
                'columns' => $this->columns,
            ]);

            $parentPath = $this->parentLocation ? $this->parentLocation->path . $this->parentLocation->id . '/' : '/';
            $parentLocation->update(['path' => $parentPath]);

            // Auto-generate children if applicable
//            if ($this->canHaveAutoChildren && $this->subLocationCount > 0) {
//                for ($i = 1; $i <= $this->subLocationCount; $i++) {
//                    $childName = $this->subLocationBaseName . ' ' . $i;
//                    $childType = self::HIERARCHY_RULES[$parentLocation->location_type][0] ?? 'box'; // Default to 'box'
//
//                    do {
//                        $childCode = ($team->research_teams_code ?? 'LOC') . '-' . str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT);
//                    } while (StorageLocation::where('location_code', $childCode)->exists());
//
//                    $childLocation = $parentLocation->children()->create([
//                        'name' => $childName,
//                        'location_code' => $childCode,
//                        'location_type' => $childType,
//                        'research_team_id' => $teamId,
//                    ]);

            if ($this->canHaveAutoChildren && $this->subLocationCount > 0) {
                $childType = self::HIERARCHY_RULES[$parentLocation->location_type][0] ?? 'box';
                for ($i = 1; $i <= $this->subLocationCount; $i++) {
                    $childName = $this->subLocationBaseName . ' ' . $i;
                    $childCode = $this->generateLocationCode($team);
                    $childLocation = $parentLocation->children()->create([
                        'name' => $childName,
                        'location_code' => $childCode,
                        'location_type' => $childType,
                        'research_team_id' => $teamId,
                    ]);
                    $childLocation->update(['path' => $parentLocation->path . $parentLocation->id . '/']);
                }
            }
        });

        $this->confirmingLocationCreation = false;
        $this->loadLocations();
        session()->flash('message', 'مکان جدید با موفقیت ایجاد شد.');
    }


    /**
     * Show the modal for editing a location.
     */
    public function confirmLocationEdit(StorageLocation $location)
    {
        $this->resetErrorBag();
        $this->editingLocation = $location;
        $this->editingLocationName = $location->name;
        $this->confirmingLocationEdit = true;
    }

    /**
     * Update the location's name.
     */
    public function updateLocation()
    {
        $this->validate([
            'editingLocationName' => 'required|string|max:255',
        ]);

        $this->editingLocation->update(['name' => $this->editingLocationName]);

        $this->confirmingLocationEdit = false;
        $this->loadLocations();
        session()->flash('message', 'نام مکان با موفقیت ویرایش شد.');
    }

    /**
     * Show the confirmation modal for deleting a location.
     */
    public function confirmLocationDeletion($locationId)
    {
        $this->locationIdToDelete = $locationId;
        $this->confirmingLocationDeletion = true;
    }

    /**
     * Delete a storage location.
     */
    public function deleteLocation()
    {
        // We might add checks here later to prevent deleting locations with items.
        StorageLocation::find($this->locationIdToDelete)->delete();

        $this->confirmingLocationDeletion = false;
        $this->loadLocations();
        session()->flash('message', 'مکان با موفقیت حذف شد.');
    }

    /**
     * Show the modal for moving a location.
     */
    public function confirmLocationMove(StorageLocation $location)
    {
        $this->resetErrorBag();
        $this->movingLocation = $location;
        $this->newParentId = null;

        // Find the allowed parent types for the location being moved.
        $allowedParentTypes = [];
        foreach (self::HIERARCHY_RULES as $parentType => $childTypes) {
            if (in_array($location->location_type, $childTypes)) {
                $allowedParentTypes[] = $parentType;
            }
        }

        // Fetch all potential new parents that match the allowed types.
        // Also, exclude the location itself and its descendants.
        $descendantIds = $this->getDescendantIds($location);
        $descendantIds[] = $location->id;

        $this->potentialNewParents = StorageLocation::whereIn('location_type', $allowedParentTypes)
            ->where('research_team_id', $location->research_team_id)
            ->whereNotIn('id', $descendantIds)
            ->get();

        $this->confirmingLocationMove = true;
    }

    /**
     * Move the location and its children to a new parent.
     */
    public function moveLocation()
    {
        if (!$this->movingLocation) return;

        $this->validate([
            'newParentId' => 'required|exists:storage_locations,id',
        ]);

        DB::transaction(function () {
            $newParent = StorageLocation::find($this->newParentId);

            // Update the moving location itself
            $this->movingLocation->update([
                'parent_id' => $newParent->id,
                'path' => $newParent->path . $newParent->id . '/',
            ]);

            // Recursively update all children
            $this->updateChildrenPaths($this->movingLocation);
        });

        $this->confirmingLocationMove = false;
        $this->loadLocations();
        session()->flash('message', 'مکان با موفقیت جابجا شد.');
    }

    /**
     * Recursively get all descendant IDs for a location.
     */
    private function getDescendantIds(StorageLocation $location): array
    {
        $ids = [];
        foreach ($location->children as $child) {
            $ids[] = $child->id;
            $ids = array_merge($ids, $this->getDescendantIds($child));
        }
        return $ids;
    }

    /**
     * Recursively update the path for all children of a location.
     */
    private function updateChildrenPaths(StorageLocation $parent)
    {
        foreach ($parent->children as $child) {
            $child->update([
                'path' => $parent->path . $parent->id . '/'
            ]);

            if ($child->children->isNotEmpty()) {
                $this->updateChildrenPaths($child);
            }
        }
    }

    public function updatedSelectAll($value)
    {
        if ($value) {
            $this->selectedLocations = $this->locations->pluck('id')->map(fn($id) => (string) $id)->toArray();
        } else {
            $this->selectedLocations = [];
        }
    }
    public function render()
    {
        $this->loadLocations();
        $this->selectAll = count($this->selectedLocations) === $this->locations->count() && $this->locations->count() > 0;
        return view('livewire.storage-location-manager')->layout('layouts.app');
    }
}
