<?php

namespace App\Livewire;

use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestItem;
use App\Models\Supplier;
use App\Models\SupplierResponse;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Component;

class SupplierDashboard extends Component
{
    use AuthorizesRequests;

    public ?PurchaseRequest $viewingRequest = null;
    public ?SupplierResponse $editingResponse = null;
    public $confirmingRequestView = false;

    // Properties for the editable form inside the modal
    public $items = [];
    public $responseStatus = '';
    public $responseNotes = '';
    // Properties for Retraction
    public $responseIdToRetract;
    public $confirmingResponseRetraction = false;

    /**
     * Show the modal for creating a new response to a request.
     * This method now only shows items that are available for bidding.
     */
    public function viewRequest(PurchaseRequest $request)
    {
        $this->resetErrorBag();
        $this->editingResponse = null;
        $this->viewingRequest = $request->load('requester.currentTeam', 'project');

        // We only populate the form with items that are still 'pending_offer'
        $availableItems = $request->items()->where('status', 'pending_offer')->get();

        $this->items = $availableItems->map(function ($item) {
            return [
                'purchase_request_item_id' => $item->id,
                'item_name' => $item->item_name,
                'description' => $item->description,
                'quantity' => $item->quantity,
                'unit' => $item->unit,
                'supplied_quantity' => $item->quantity,
                'price_per_unit' => '',
                'supplier_notes' => '',
                'is_accepted' => true,
                'status' => $item->status,
            ];
        })->toArray();

        $this->responseStatus = '';
        $this->responseNotes = '';
        $this->confirmingRequestView = true;
    }

    /**
     * Show the modal for EDITING an existing response.
     */
//    public function confirmResponseEdit(SupplierResponse $response)
//    {
//        $this->resetErrorBag();
//        $this->editingResponse = SupplierResponse::with('items.originalItem')->find($response->id);
//        $this->viewingRequest = $this->editingResponse->purchaseRequest;
//
//        $this->responseStatus = $this->editingResponse->status;
//        $this->responseNotes = $this->editingResponse->notes;
//
//        $this->items = $this->editingResponse->items->map(function ($responseItem) {
//            return [
//                'id' => $responseItem->id,
//                'purchase_request_item_id' => $responseItem->originalItem ? $responseItem->originalItem->id : null,
//                'item_name' => $responseItem->originalItem ? $responseItem->originalItem->item_name : null,
//                'description' => $responseItem->originalItem ? $responseItem->originalItem->description : null,
//                'quantity' => $responseItem->originalItem ? $responseItem->originalItem->quantity : null,
//                'unit' => $responseItem->originalItem ? $responseItem->originalItem->unit : null,
//                'supplied_quantity' => $responseItem->supplied_quantity,
//                'price_per_unit' => $responseItem->price_per_unit,
//                'supplier_notes' => $responseItem->supplier_notes,
//                'is_accepted' => $responseItem->is_accepted,
//            ];
//        })->toArray();
//
//        $this->confirmingRequestView = true;
//    }

//    public function confirmResponseEdit(SupplierResponse $response)
//    {
//        $this->resetErrorBag();
//
//        $this->editingResponse = SupplierResponse::with('items.originalItem')->find($response->id);
//        $this->viewingRequest = $this->editingResponse->purchaseRequest;
//
//        $this->responseStatus = $this->editingResponse->status;
//        $this->responseNotes = $this->editingResponse->notes;
//
//        $this->items = $this->editingResponse->purchaseRequest->items->map(function ($requestItem) {
//            $responseItem = $this->editingResponse->items->firstWhere('purchase_request_item_id', $requestItem->id);
//
//            return [
//                'purchase_request_item_id' => $requestItem->id,
//                'item_name' => $requestItem->item_name,
//                'description' => $requestItem->description,
//                'quantity' => $requestItem->quantity,
//                'unit' => $requestItem->unit,
//                'supplied_quantity' => $responseItem ? $responseItem->supplied_quantity : $requestItem->quantity,
//                'price_per_unit' => $responseItem ? $responseItem->price_per_unit : '',
//                'supplier_notes' => $responseItem ? $responseItem->supplier_notes : '',
//                'is_accepted' => $responseItem ? $responseItem->is_accepted : false,
//            ];
//        })->toArray();
//
//        $this->confirmingRequestView = true;
//    }

    public function confirmResponseEdit(SupplierResponse $response)
    {
        $this->resetErrorBag();
//        $this->editingResponse = $response->load('items.originalItem');
//        $this->viewingRequest = $this->editingResponse->purchaseRequest->load('items');

        $this->editingResponse = $response;
        $this->viewingRequest = $response->purchaseRequest->load('items.finalResponseItem.response.supplier');

        $this->responseStatus = $this->editingResponse->status;
        $this->responseNotes = $this->editingResponse->notes;

        // Map ALL original request items, and fill with existing response data if available
        $this->items = $this->viewingRequest->items->map(function ($requestItem) {
            $responseItem = $this->editingResponse->items->firstWhere('purchase_request_item_id', $requestItem->id);

            return [
                'purchase_request_item_id' => $requestItem->id,
                'item_name' => $requestItem->item_name,
                'description' => $requestItem->description,
                'quantity' => $requestItem->quantity,
                'unit' => $requestItem->unit,
                'supplied_quantity' => $responseItem ? $responseItem->supplied_quantity : $requestItem->quantity,
                'price_per_unit' => $responseItem ? $responseItem->price_per_unit : '',
                'supplier_notes' => $responseItem ? $responseItem->supplier_notes : '',
                'is_accepted' => $responseItem ? $responseItem->is_accepted : false,
                // ما وضعیت آیتم اصلی را نیز برای استفاده در View ارسال می‌کنیم
                'status' => $requestItem->status,
                'winner_supplier' => $requestItem->finalResponseItem?->response?->supplier?->company_name,
            ];
        })->toArray();

        $this->confirmingRequestView = true;
    }

    /**
     * Save the changes to an existing response using updateOrCreate.
     */
    public function updateResponse()
    {
        if (!$this->editingResponse) return;

        $validatedData = $this->validate([
            'responseStatus' => 'required|in:صدور پیش فاکتور,فاکتور رسمی,تامین شده',
            'responseNotes' => 'nullable|string',
            'items.*.purchase_request_item_id' => 'required|exists:purchase_request_items,id',
            'items.*.is_accepted' => 'required|boolean',
            'items.*.price_per_unit' => 'required_if:items.*.is_accepted,true|nullable|numeric|min:0',
            'items.*.supplied_quantity' => 'required_if:items.*.is_accepted,true|nullable|numeric|min:0',
            'items.*.supplier_notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validatedData) {
            $this->editingResponse->update([
                'status' => $validatedData['responseStatus'],
                'notes' => $validatedData['responseNotes'],
            ]);

            foreach ($this->items as $itemData) {
                // ما فقط آیتم‌هایی را که هنوز در انتظار پیشنهاد هستند، به‌روزرسانی می‌کنیم
                $originalItem = PurchaseRequestItem::find($itemData['purchase_request_item_id']);
                if ($originalItem && $originalItem->status === 'pending_offer') {
                    $this->editingResponse->items()->updateOrCreate(
                        ['purchase_request_item_id' => $itemData['purchase_request_item_id']],
                        [
                            'is_accepted' => $itemData['is_accepted'],
                            'supplied_quantity' => $itemData['is_accepted'] ? $itemData['supplied_quantity'] : null,
                            'price_per_unit' => $itemData['is_accepted'] ? $itemData['price_per_unit'] : null,
                            'supplier_notes' => $itemData['supplier_notes'],
                        ]
                    );
                }
            }
        });

        $this->confirmingRequestView = false;
        session()->flash('message', 'پاسخ شما با موفقیت ویرایش شد.');
    }

    public function submitResponse()
    {
        if (!$this->viewingRequest) return;

        $validatedData = $this->validate([
            'responseStatus' => 'required|in:صدور پیش فاکتور,فاکتور رسمی,تامین شده',
            'responseNotes' => 'nullable|string',
            'items.*.purchase_request_item_id' => 'required|exists:purchase_request_items,id', // <-- قانون جدید اضافه شد
            'items.*.is_accepted' => 'required|boolean',
            'items.*.price_per_unit' => 'required_if:items.*.is_accepted,true|nullable|numeric|min:0',
            'items.*.supplied_quantity' => 'required_if:items.*.is_accepted,true|nullable|numeric|min:0',
            'items.*.supplier_notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validatedData) {
            // Step 1: Create the main SupplierResponse record
            $response = $this->viewingRequest->responses()->create([
                'supplier_id' => Auth::id(),
                'status' => $validatedData['responseStatus'],
                'notes' => $validatedData['responseNotes'],
            ]);

            // Step 2: Create each SupplierResponseItem
            foreach ($this->items as $itemData) {
                $response->items()->create([
                    'purchase_request_item_id' => $itemData['purchase_request_item_id'],
                    'is_accepted' => $itemData['is_accepted'],
                    'supplied_quantity' => $itemData['is_accepted'] ? $itemData['supplied_quantity'] : null,
                    'price_per_unit' => $itemData['is_accepted'] ? $itemData['price_per_unit'] : null,
                    'supplier_notes' => $itemData['supplier_notes'],
                ]);
            }
            // --- منطق جدید: تغییر وضعیت پس از اولین پاسخ ---
            if ($this->viewingRequest->status === 'pending_merchant') {
                $this->viewingRequest->update(['status' => 'price_received']);
            }
        });

        $this->confirmingRequestView = false;
        session()->flash('message', 'پاسخ شما برای درخواست با موفقیت ثبت شد.');
    }

    /**
     * Show the confirmation modal for retracting a response.
     */
    public function confirmResponseRetraction($responseId)
    {
        $this->responseIdToRetract = $responseId;
        $this->confirmingResponseRetraction = true;
    }

    /**
     * Retract (delete) the supplier's response.
     */
    public function retractResponse()
    {
        $response = SupplierResponse::with('items')->find($this->responseIdToRetract);

        if (!$response || $response->supplier_id !== Auth::id()) {
            abort(403);
        }

        // --- قدم ۱: بررسی اینکه آیا هیچ آیتمی از این پاسخ، برنده شده است یا خیر ---
        $winningItemIds = $response->items->pluck('id');
        $isAnyItemFinalized = PurchaseRequestItem::whereIn('final_supplier_response_item_id', $winningItemIds)->exists();

        if ($isAnyItemFinalized) {
            $this->confirmingResponseRetraction = false;
            session()->flash('error', 'شما نمی‌توانید این پیشنهاد را بازپس بگیرید زیرا حداقل یکی از آیتم‌های آن توسط درخواست‌کننده تایید نهایی شده است.');
            return;
        }

        // --- قدم ۲: حذف پاسخ و مدیریت هوشمند وضعیت ---
        DB::transaction(function () use ($response) {
            $purchaseRequest = $response->purchaseRequest;

            $response->items()->delete();
            $response->delete();

            // اگر این آخرین پاسخ بود، وضعیت درخواست اصلی را به "در انتظار بازرگان" برمی‌گردانیم
            if ($purchaseRequest->responses()->count() === 0) {
                $purchaseRequest->update(['status' => 'pending_merchant']);
            }
        });

        $this->confirmingResponseRetraction = false;
        session()->flash('message', 'پاسخ شما با موفقیت بازپس گرفته شد.');
    }

    public function render()
    {
        $supplierId = Auth::id();

        // --- منطق جدید: نمایش درخواست‌ها با هر دو وضعیت ---
        $requests = PurchaseRequest::whereIn('status', ['pending_merchant', 'price_received', 'final_approval'])
            ->whereHas('items', function ($query) {
                $query->where('status', 'pending_offer');
            })
            ->with(['requester.currentTeam', 'project', 'responses' => function ($query) use ($supplierId) {
                $query->where('supplier_id', $supplierId);},
                // ما آیتم‌های نهایی شده را نیز برای بررسی دسترسی در View، بارگذاری می‌کنیم
                'items' => function ($query) {
                    $query->whereNotNull('final_supplier_response_item_id');
                }
            ])
            ->latest()
            ->get();

        return view('livewire.supplier-dashboard', [
            'requests' => $requests
        ])->layout('layouts.supplier-app');
    }
}
