<?php
//
//namespace App\Livewire;
//
//use App\Models\PurchaseRequest;
//use App\Models\Shipment;
//use Illuminate\Support\Facades\Auth;
//use Illuminate\Support\Facades\DB;
//use Livewire\Component;
//
//class SupplierOrders extends Component
//{
//    // Properties for Shipment Modal
//    public ?PurchaseRequest $requestForShipment = null;
//    public $confirmingShipment = false;
//    public array $itemsToShip = [];
//    public $shipping_method;
//    public $tracking_code;
//
//
//    /**
//     * Show the modal for creating a new shipment.
//     */
//    public function confirmShipment(PurchaseRequest $request)
//    {
//        $this->resetErrorBag();
//        $this->requestForShipment = $request;
//        $this->reset(['shipping_method', 'tracking_code']);
//
//        // Load only the winning items for this supplier that have NOT been shipped yet.
//        $this->itemsToShip = $request->items()
//            ->whereHas('finalResponseItem.response', function ($query) {
//                $query->where('supplier_id', Auth::guard('supplier')->id());
//            })
//            ->whereDoesntHave('finalResponseItem.shipment') // <-- کلید اصلی: فقط آیتم‌های ارسال نشده
//            ->with('finalResponseItem')
//            ->get()
//            ->map(function ($item) {
//                return [
//                    'id' => $item->finalResponseItem->id,
//                    'name' => $item->item_name,
//                    'quantity' => $item->finalResponseItem->supplied_quantity,
//                    'unit' => $item->unit,
//                    'selected' => true, // Select all by default
//                ];
//            })->toArray();
//
//        if (empty($this->itemsToShip)) {
//            session()->flash('message', 'تمام آیتم‌های این سفارش قبلاً ارسال شده‌اند.');
//            return;
//        }
//
//        $this->confirmingShipment = true;
//    }
//
//    /**
//     * Create a new shipment record and associate items with it.
//     */
//    public function createShipment()
//    {
//        if (!$this->requestForShipment) return;
//
//        $validatedData = $this->validate([
//            'shipping_method' => 'required|string|max:255',
//            'tracking_code' => 'nullable|string|max:255',
//            'itemsToShip.*.selected' => 'boolean',
//        ]);
//
//        $selectedItemIds = collect($this->itemsToShip)
//            ->where('selected', true)
//            ->pluck('id')
//            ->all();
//
//        if (empty($selectedItemIds)) {
//            $this->addError('items', 'حداقل یک آیتم باید برای ارسال انتخاب شود.');
//            return;
//        }
//
//        DB::transaction(function () use ($validatedData, $selectedItemIds) {
//            // Step 1: Create the shipment record
//            $shipment = Shipment::create([
//                'supplier_id' => Auth::guard('supplier')->id(),
//                'shipping_method' => $validatedData['shipping_method'],
//                'tracking_code' => $validatedData['tracking_code'],
//                'dispatched_at' => now(),
//            ]);
//
//            // Step 2: Associate the selected items with this shipment
//            \App\Models\SupplierResponseItem::whereIn('id', $selectedItemIds)
//                ->update(['shipment_id' => $shipment->id]);
//        });
//
//        $this->confirmingShipment = false;
//        session()->flash('message', 'اطلاعات ارسال با موفقیت ثبت شد.');
//    }
//    public function render()
//    {
//        $supplierId = Auth::guard('supplier')->id();
//
//        // --- کوئری جدید برای گروه‌بندی سفارش‌ها ---
//        // ما درخواست‌های خریدی را پیدا می‌کنیم که حداقل یک آیتم برنده شده توسط این تأمین‌کننده را داشته باشند.
//        $requestsWithWinningItems = PurchaseRequest::whereHas('items.finalResponseItem.response', function ($query) use ($supplierId) {
//            $query->where('supplier_id', $supplierId);
//        })
//            ->with([
//                // ما فقط آیتم‌هایی را بارگذاری می‌کنیم که این تأمین‌کننده برنده شده است.
//                'items' => function ($query) use ($supplierId) {
//                    $query->whereHas('finalResponseItem.response', function ($subQuery) use ($supplierId) {
//                        $subQuery->where('supplier_id', $supplierId);
//                    });
//                },
//                'items.finalResponseItem', // اطلاعات پاسخ برنده شده
//                'requester.currentTeam' // اطلاعات سازمان خریدار
//            ])
//            ->latest()
//            ->get();
//
//        return view('livewire.supplier-orders', [
//            'requests' => $requestsWithWinningItems
//        ])->layout('layouts.supplier-app');
//    }
//}




namespace App\Livewire;

use App\Models\PurchaseRequest;
use App\Models\Shipment;
use App\Models\ShipmentItem;
use App\Models\SupplierResponseItem;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Component;

class SupplierOrders extends Component
{
    // Properties for Shipment Modal
    public ?PurchaseRequest $requestForShipment = null;
    public bool $confirmingShipment = false;
    public array $itemsToShip = [];
    public $shipping_method;
    public $tracking_code;
    public $notes; // [NEW] Property for shipment notes


    /**
     * Show the modal for creating a new shipment.
     * This method is now rewritten to work with the new pivot table structure.
     */
    public function confirmShipment(PurchaseRequest $request)
    {
        $this->resetErrorBag();
        $this->requestForShipment = $request;
        $this->reset(['shipping_method', 'tracking_code', 'notes']);

        // Get all winning SupplierResponseItem IDs for this supplier and this request.
        $winningResponseItemIds = $request->items()
            ->where('status', 'offer_accepted')
            ->whereHas('finalResponseItem.response', function ($query) {
                $query->where('supplier_id', Auth::guard('supplier')->id());
            })
            ->pluck('final_supplier_response_item_id');

        // Get the IDs of items that are already in a shipment.
        $shippedResponseItemIds = ShipmentItem::whereIn('supplier_response_item_id', $winningResponseItemIds)
            ->pluck('supplier_response_item_id');

        // The items to ship are the ones that are winning but not yet shipped.
        $unshippedResponseItemIds = $winningResponseItemIds->diff($shippedResponseItemIds);

        // Now, fetch the full item details for the unshipped items.
        // [BUGFIX] We add whereHas('originalItem') to ensure that we only process
        // response items that have a valid link to their original request item.
        // This prevents a "property on null" error if data is inconsistent.
        $unshippedItems = SupplierResponseItem::whereIn('id', $unshippedResponseItemIds)
            ->whereHas('originalItem')
            ->with('originalItem')
            ->get();

        $this->itemsToShip = $unshippedItems->map(function ($responseItem) {
            return [
                'id' => $responseItem->id,
                'name' => $responseItem->originalItem->item_name,
                'quantity' => $responseItem->supplied_quantity,
                'unit' => $responseItem->originalItem->unit,
                'selected' => true, // Select all by default
            ];
        })->toArray();

        if (empty($this->itemsToShip)) {
            session()->flash('message', 'تمام آیتم‌های این سفارش قبلاً ارسال شده‌اند.');
            return;
        }

        $this->confirmingShipment = true;
    }

    /**
     * Create a new shipment record and associate items with it using the pivot table.
     */
    public function createShipment()
    {
        if (!$this->requestForShipment) return;

        $validatedData = $this->validate([
            'shipping_method' => 'required|string|max:255',
            'tracking_code' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'itemsToShip.*.selected' => 'boolean',
        ]);

        $selectedItemIds = collect($this->itemsToShip)
            ->where('selected', true)
            ->pluck('id')
            ->all();

        if (empty($selectedItemIds)) {
            $this->addError('items', 'حداقل یک آیتم باید برای ارسال انتخاب شود.');
            return;
        }

        DB::transaction(function () use ($validatedData, $selectedItemIds) {
            // Step 1: Create the main shipment record
            $shipment = Shipment::create([
                'supplier_id' => Auth::guard('supplier')->id(),
                'purchase_request_id' => $this->requestForShipment->id,
                'shipping_method' => $validatedData['shipping_method'],
                'tracking_code' => $validatedData['tracking_code'],
                'notes' => $validatedData['notes'],
                'dispatched_at' => now(),
                'status' => 'shipped',
            ]);

            // Step 2: Create records in the pivot table 'shipment_items'
            $shipmentItemsData = collect($selectedItemIds)->map(function ($itemId) use ($shipment) {
                return [
                    'shipment_id' => $shipment->id,
                    'supplier_response_item_id' => $itemId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            })->all();

            ShipmentItem::insert($shipmentItemsData);
        });

        $this->confirmingShipment = false;
        session()->flash('message', 'اطلاعات ارسال با موفقیت ثبت شد.');
    }

    public function render()
    {
        $supplierId = Auth::guard('supplier')->id();

        // Fetch purchase requests where this supplier has winning items.
        $requestsWithWinningItems = PurchaseRequest::whereHas('items.finalResponseItem.response', function ($query) use ($supplierId) {
            $query->where('supplier_id', $supplierId);
        })
            ->with([
                // Load only the items this supplier won.
                'items' => function ($query) use ($supplierId) {
                    $query->whereHas('finalResponseItem.response', function ($subQuery) use ($supplierId) {
                        $subQuery->where('supplier_id', $supplierId);
                    });
                },
                'items.finalResponseItem.shipmentItems', // Eager load the pivot relationship
                'requester.currentTeam'
            ])
            ->latest()
            ->get();

        return view('livewire.supplier-orders', [
            'requests' => $requestsWithWinningItems
        ])->layout('layouts.supplier-app');
    }
}

