<?php

namespace App\Livewire;

use App\Models\StorageLocation;
use Livewire\Component;

class TankViewer extends Component
{
    public StorageLocation $tank;
    public $canisters = [];

    // Properties to manage the selected items in the view
    public ?StorageLocation $selectedCanister = null;
    public $racksOrCanes = []; // To hold the contents of the selected canister
    // [NEW] Properties for rack and box navigation
    public ?StorageLocation $selectedRack = null;
    public $boxes = [];

    public function mount(StorageLocation $location) // Route model binding will pass the tank here
    {
        // Ensure the location is a tank
        if ($location->location_type !== 'tank') {
            abort(404, 'این مکان یک تانک ازت نیست.');
        }

        $this->tank = $location;
        $this->loadCanisters();
    }

    /**
     * Loads the canisters (children) of the main tank.
     */
    public function loadCanisters()
    {
//        $this->canisters = $this->tank->children()->get();
        $this->canisters = $this->tank->children()->where('location_type', 'canister')->orderBy('name')->get();

    }

    /**
     * Sets the selected canister and loads its contents (racks or canes).
     *
     * @param int $canisterId
     */
    public function selectCanister(int $canisterId)
    {
        // Deselect if the same canister is clicked again
        if ($this->selectedCanister && $this->selectedCanister->id === $canisterId) {
            $this->selectedCanister = null;
            $this->racksOrCanes = [];
            $this->selectedRack = null; // Also reset rack selection
            $this->boxes = [];
            return;
        }
// [MERGED] Reset rack selection when a new canister is chosen
        $this->selectedRack = null;
        $this->boxes = [];
        $this->selectedCanister = StorageLocation::with('children')->find($canisterId);
        $this->racksOrCanes = $this->selectedCanister ? $this->selectedCanister->children : [];
    }
    /**
     * [NEW] Select a rack and load its children (boxes).
     *
     * @param int $rackId
     */
    public function selectRack($rackId)
    {
        $this->selectedRack = StorageLocation::find($rackId);
        $this->boxes = $this->selectedRack ? $this->selectedRack->children()->where('location_type', 'box')->orderBy('name')->get() : [];
    }

    /**
     * [NEW] Go back from the boxes view to the racks view.
     */
    public function goBackToCanisterView()
    {
        $this->selectedRack = null;
        $this->boxes = [];
    }
    public function render()
    {
        return view('livewire.tank-viewer')->layout('layouts.app');
    }
}
