<?php

namespace App\Livewire;

use App\Models\Task;
use App\Models\TimeLog;
use App\Models\User;
use App\Notifications\UserMentionedInCommentNotification;
use Illuminate\Support\Facades\Notification;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\LabNotebookEntry;
use Carbon\Carbon;


class TaskDetailModal extends Component
{
    use AuthorizesRequests;
    use WithFileUploads;

    public ?Task $task = null;
    public bool $showingModal = false;

    // Properties for Comments
    public string $newComment = '';

    // Properties for Attachments
    public $newAttachment;
    // [NEW] Properties for Time Tracking
    public ?TimeLog $activeTimer = null;
    // [NEW] Properties for Linking Lab Notebook Entries
    public string $entrySearch = '';
    public $entrySearchResults = [];
    protected $listeners = ['showTaskModal'];

    public function showTaskModal(Task $task)
    {
        $this->task = $task->load(['comments.user', 'attachments.user', 'assignees', 'creator']);
        // Check if the current user has an active timer for this task
        $this->activeTimer = $this->task->timeLogs()
            ->where('user_id', Auth::id())
            ->whereNull('ended_at')
            ->first();

        $this->showingModal = true;
    }

    public function addComment()
    {
        if (!$this->task) return;
        // $this->authorize('addComment', $this->task);

        $this->validate(['newComment' => 'required|string|min:3']);

        $commenter = Auth::user();

        $comment = $this->task->comments()->create([
            'user_id' => $commenter->id,
            'body' => $this->newComment,
        ]);

        // پیدا کردن و ارسال اعلان برای کاربران mention شده
        $this->notifyMentionedUsers($comment);

        $this->newComment = '';
        $this->task->load('comments.user'); // Refresh comments
    }
    protected function notifyMentionedUsers($comment)
    {
        // با استفاده از یک عبارت منظم، تمام کلماتی که با @ شروع می‌شوند را پیدا می‌کنیم
        preg_match_all('/@([\w\-]+)/', $comment->body, $matches);

        if (!empty($matches[1])) {
            $mentionedUsernames = $matches[1];

            // کاربرانی که username آن‌ها در لیست mention شده‌ها وجود دارد را پیدا می‌کنیم
            // و کاربری که خودش نظر را ثبت کرده، از لیست دریافت‌کنندگان حذف می‌کنیم
            $recipients = User::whereIn('username', $mentionedUsernames)
                ->where('id', '!=', $comment->user_id)
                ->get();

            if ($recipients->isNotEmpty()) {
                Notification::send($recipients, new UserMentionedInCommentNotification($this->task, $comment, $comment->user));
            }
        }
    }
    public function saveAttachment()
    {
        if (!$this->task) return;
        // $this->authorize('addAttachment', $this->task);

        $this->validate(['newAttachment' => 'required|file|max:10240']); // 10MB Max

        $path = $this->newAttachment->store('attachments', 'public');

        $this->task->attachments()->create([
            'user_id' => Auth::id(),
            'file_path' => $path,
            'file_name' => $this->newAttachment->getClientOriginalName(),
            'file_size' => $this->newAttachment->getSize(),
            'mime_type' => $this->newAttachment->getMimeType(),
        ]);

        $this->newAttachment = null;
        $this->task->load('attachments.user'); // Refresh attachments
    }

    public function downloadAttachment($attachmentId)
    {
        $attachment = $this->task->attachments()->findOrFail($attachmentId);
        // $this->authorize('download', $attachment);
        return Storage::disk('public')->download($attachment->file_path, $attachment->file_name);
    }
    /**
     * [NEW] Start or stop the timer for the current user on the current task.
     */
    public function toggleTimer()
    {
        if (!$this->task) return;
        // You can add a policy check here if needed
        // $this->authorize('logTime', $this->task);

        // If a timer is already running, stop it.
        if ($this->activeTimer) {
            $this->activeTimer->ended_at = now();
            $this->activeTimer->duration_in_seconds = $this->activeTimer->started_at->diffInSeconds($this->activeTimer->ended_at);
            $this->activeTimer->save();

            $this->activeTimer = null; // Reset the active timer
        }
        // Otherwise, start a new timer.
        else {
            $this->activeTimer = $this->task->timeLogs()->create([
                'user_id' => Auth::id(),
                'started_at' => now(),
            ]);
        }

        // Refresh the time logs relationship
        $this->task->load('timeLogs.user');
    }
    /**
     * [NEW] Live search for lab notebook entries within the same project.
     */
    public function updatedEntrySearch($value)
    {
        if (strlen($this->entrySearch) < 2 || !$this->task) {
            $this->entrySearchResults = [];
            return;
        }

        // Get IDs of entries already linked to this task
        $linkedEntryIds = $this->task->labNotebookEntries->pluck('id');

        $this->entrySearchResults = LabNotebookEntry::where('project_id', $this->task->project_id)
            ->where('title', 'like', '%' . $this->entrySearch . '%')
            ->whereNotIn('id', $linkedEntryIds) // Exclude already linked entries
            ->take(5)
            ->get();
    }

    /**
     * [CORRECTED] Link a lab notebook entry to the current task by its ID.
     */
    public function linkEntry($entryId)
    {
        if (!$this->task) return;
        $entry = LabNotebookEntry::find($entryId);
        if (!$entry) return;

        // $this->authorize('linkEntry', $this->task);

        $this->task->labNotebookEntries()->attach($entry->id);
        $this->task->load('labNotebookEntries');
        $this->entrySearch = '';
        $this->entrySearchResults = [];
    }

    /**
     * [CORRECTED] Unlink a lab notebook entry from the current task by its ID.
     */
    public function unlinkEntry($entryId)
    {
        if (!$this->task) return;
        $entry = LabNotebookEntry::find($entryId);
        if (!$entry) return;

        // $this->authorize('unlinkEntry', $this->task);

        $this->task->labNotebookEntries()->detach($entry->id);
        $this->task->load('labNotebookEntries');
    }
    public function render()
    {
        return view('livewire.task-detail-modal');
    }
}
