<?php

namespace App\Livewire;

use App\Models\Auditor;
use App\Models\Equipment;
use App\Models\TemperatureLog;
use Hekmatinasser\Verta\Verta;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Mccarlosen\LaravelMpdf\Facades\LaravelMpdf as PDF;

class TemperatureLogHistory extends Component
{
    public Equipment $equipment;

    // Filters
    public $auditorFilter = '';
    public $startDateFilter = '';
    public $endDateFilter = '';

    public function mount(Equipment $equipment)
    {
        $this->equipment = $equipment;
    }

    public function getFilteredLogs()
    {
        return TemperatureLog::where('equipment_id', $this->equipment->id)
            ->when($this->auditorFilter, fn($query) => $query->where('auditor_id', $this->auditorFilter))
            ->when($this->startDateFilter, fn($query) => $query->where('created_at', '>=', Verta::parse($this->startDateFilter)->startOfDay()))
            ->when($this->endDateFilter, fn($query) => $query->where('created_at', '<=', Verta::parse($this->endDateFilter)->endOfDay()))
            ->with('auditor')
            ->latest()
            ->get();
    }

    public function exportCsv()
    {
        // ... (Logic from EquipmentManager's exportHistoryCsv method)
    }

    public function exportPdf()
    {
        $data = [
            'logs' => $this->getFilteredLogs(),
            'equipment' => $this->equipment,
            'filters' => [
                'auditor' => $this->auditorFilter ? Auditor::find($this->auditorFilter)->name : 'همه',
                'start_date' => $this->startDateFilter ?: '---',
                'end_date' => $this->endDateFilter ?: '---',
            ]
        ];
        $pdf = PDF::loadView('pdf.temperature-log', $data, [], ['default_font' => 'IranSans']);
        return $pdf->stream('temp-log-' . $this->equipment->system_sku . '.pdf');
    }

    public function updated()
    {
        $this->dispatch('updateChart', data: $this->getFilteredLogs()->pluck('temperature', 'created_at'));
    }

    public function render()
    {
        $logs = $this->getFilteredLogs();
        $auditors = Auditor::whereIn('research_team_id', Auth::user()->researchTeams()->pluck('research_teams.id'))->get();

        // Dispatch initial chart data on render
        $this->dispatch('updateChart', data: $logs->pluck('temperature', 'created_at'));

        return view('livewire.temperature-log-history', [
            'logs' => $logs,
            'auditors' => $auditors,
        ])->layout('layouts.guest');
    }
}
